package io.dyte.core.media

import kotlinx.cinterop.Arena
import kotlinx.cinterop.CValue
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.toCValues
import platform.CFNetwork.CFHTTPMessageCopySerializedMessage
import platform.CFNetwork.CFHTTPMessageCreateResponse
import platform.CFNetwork.CFHTTPMessageSetBody
import platform.CFNetwork.CFHTTPMessageSetHeaderFieldValue
import platform.CFNetwork.kCFHTTPVersion1_1
import platform.CoreFoundation.CFDataRef
import platform.CoreFoundation.CFRelease
import platform.CoreFoundation.CFStringRef
import platform.CoreFoundation.kCFAllocatorDefault
import platform.CoreGraphics.CGAffineTransform
import platform.CoreGraphics.CGAffineTransformMakeScale
import platform.CoreImage.CIContext
import platform.CoreImage.CIImage
import platform.CoreImage.CIImageRepresentationOption
import platform.CoreImage.JPEGRepresentationOfImage
import platform.CoreMedia.CMGetAttachment
import platform.CoreMedia.CMSampleBufferGetImageBuffer
import platform.CoreMedia.CMSampleBufferRef
import platform.CoreVideo.CVPixelBufferGetHeight
import platform.CoreVideo.CVPixelBufferGetWidth
import platform.CoreVideo.CVPixelBufferLockBaseAddress
import platform.CoreVideo.CVPixelBufferRef
import platform.CoreVideo.CVPixelBufferUnlockBaseAddress
import platform.CoreVideo.kCVPixelBufferLock_ReadOnly
import platform.Foundation.CFBridgingRelease
import platform.Foundation.CFBridgingRetain
import platform.Foundation.NSData
import platform.Foundation.NSMakeRange
import platform.Foundation.NSNumber
import platform.Foundation.NSString
import platform.Foundation.subdataWithRange
import platform.ImageIO.kCGImageDestinationLossyCompressionQuality
import platform.ReplayKit.RPVideoSampleOrientationKey
import platform.darwin.dispatch_async
import platform.darwin.dispatch_queue_create
import platform.darwin.dispatch_queue_t
import toByteArray

private const val BUFFER_MAX_LENGTH = 10240

@OptIn(ExperimentalForeignApi::class)
class SampleUploader(private val connection: ClientSocketConnection) {
  private val imageContext = CIContext()
  private var isReady = kotlin.concurrent.AtomicInt(0)

  private var dataToSend: NSData? = null
  private var byteIndex = 0

  private val serialQueue: dispatch_queue_t = dispatch_queue_create("SampleUploader", null)

  @OptIn(ExperimentalForeignApi::class) private val arena = Arena()

  init {
    setupConnection()
  }

  private fun setupConnection() {
    connection.didOpen = { isReady.value = 1 }
    connection.streamHasSpaceAvailable = {
      dispatch_async(serialQueue) {
        val success = sendDataChunk()
        isReady.value = if (success) 0 else 1
      }
    }
  }

  fun send(sampleBuffer: CMSampleBufferRef): Boolean {
    if (isReady.value == 0) return false

    isReady.value = 0
    dataToSend = prepareSample(sampleBuffer)
    byteIndex = 0

    dispatch_async(serialQueue) {
      val success = sendDataChunk()
      if (success.not()) {
        isReady.value = 1
      }
    }

    return true
  }

  private fun sendDataChunk(): Boolean =
    with(arena) {
      if (dataToSend == null) return false

      var bytesLeft = dataToSend!!.length().toInt() - byteIndex
      var length = minOf(bytesLeft, BUFFER_MAX_LENGTH)

      val range = NSMakeRange(byteIndex.toULong(), length.toULong())

      val data = dataToSend!!.subdataWithRange(range).toByteArray().toUByteArray().toCValues()
      arena.apply { length = connection.writeToStream(data.getPointer(this), length) }
      if (length > 0) {
        byteIndex += length
        bytesLeft -= length

        if (bytesLeft == 0) {
          dataToSend = null
          byteIndex = 0
        }
      }

      return true
    }

  private fun prepareSample(buffer: CMSampleBufferRef): NSData? {
    val imageBuffer =
      CMSampleBufferGetImageBuffer(buffer)
        ?: run {
          return null
        }

    CVPixelBufferLockBaseAddress(imageBuffer, kCVPixelBufferLock_ReadOnly)
    val scaleFactor = 2.0
    val width = CVPixelBufferGetWidth(imageBuffer).toDouble() / scaleFactor
    val height = CVPixelBufferGetHeight(imageBuffer).toDouble() / scaleFactor
    val RPVideoSampleOrientationKeyCString =
      CFBridgingRetain(RPVideoSampleOrientationKey as NSString) as CFStringRef
    val orientation =
      CMGetAttachment(imageBuffer, RPVideoSampleOrientationKeyCString, null) as? NSNumber
    val scaleTransform = CGAffineTransformMakeScale(1.0 / scaleFactor, 1.0 / scaleFactor)
    val bufferData = jpegDataFromBuffer(imageBuffer, scaleTransform)

    CVPixelBufferUnlockBaseAddress(imageBuffer, kCVPixelBufferLock_ReadOnly)

    val messageData =
      bufferData
        ?: run {
          return null
        }

    val httpResponse =
      CFHTTPMessageCreateResponse(kCFAllocatorDefault, 200L, null, kCFHTTPVersion1_1)
    val contentLengthString = "Content-Length"
    val contentLengthCString = CFBridgingRetain(contentLengthString as NSString) as CFStringRef
    val contentLengthValueString = "${messageData.length}"
    val contentLengthValueCString = CFBridgingRetain(contentLengthValueString) as CFStringRef
    CFHTTPMessageSetHeaderFieldValue(httpResponse, contentLengthCString, contentLengthValueCString)
    val bufferWidthString = "Buffer-Width"
    val bufferWidthCString = CFBridgingRetain(bufferWidthString as NSString) as CFStringRef
    val bufferWidthValueString = "$width"
    val bufferWidthValueCString =
      CFBridgingRetain(bufferWidthValueString as NSString) as CFStringRef
    CFHTTPMessageSetHeaderFieldValue(httpResponse, bufferWidthCString, bufferWidthValueCString)
    val bufferHeightString = "Buffer-Height"
    val bufferHeightCString = CFBridgingRetain(bufferHeightString as NSString) as CFStringRef
    val bufferHeightValueString = "$height"
    val bufferHeightValueCString =
      CFBridgingRetain(bufferHeightValueString as NSString) as CFStringRef
    CFHTTPMessageSetHeaderFieldValue(httpResponse, bufferHeightCString, bufferHeightValueCString)

    val bufferOrientationString = "Buffer-Orientation"
    val bufferOrientationCString =
      CFBridgingRetain(bufferOrientationString as NSString) as CFStringRef
    val bufferOrientationValueString = "${orientation?.intValue ?: 0}"
    val bufferOrientationValueCString =
      CFBridgingRetain(bufferOrientationValueString as NSString) as CFStringRef
    CFHTTPMessageSetHeaderFieldValue(
      httpResponse,
      bufferOrientationCString,
      bufferOrientationValueCString,
    )

    // NSData can be toll-free bridged to CFDataRef, safe to suppress the warning
    @Suppress("UNCHECKED_CAST") val message = CFBridgingRetain(messageData) as CFDataRef?
    CFHTTPMessageSetBody(httpResponse, message)

    val serializedMessage = CFHTTPMessageCopySerializedMessage(httpResponse)

    // Since we are using CFBridgingRetain, we need to release it manually.
    // It should be safe to release it before returning since it we already have the
    // serialized message to work with.
    CFRelease(message)
    CFRelease(httpResponse)
    CFRelease(RPVideoSampleOrientationKeyCString)
    CFRelease(bufferHeightCString)
    CFRelease(bufferWidthCString)
    CFRelease(bufferHeightValueCString)
    CFRelease(bufferWidthValueCString)
    CFRelease(bufferOrientationCString)
    CFRelease(bufferOrientationValueCString)
    CFRelease(contentLengthCString)
    CFRelease(contentLengthValueCString)
    return CFBridgingRelease(serializedMessage) as NSData?
  }

  private fun jpegDataFromBuffer(
    buffer: CVPixelBufferRef,
    scaleTransform: CValue<CGAffineTransform>,
  ): NSData? {
    val ciImage = CIImage(cVPixelBuffer = buffer).imageByApplyingTransform(scaleTransform)

    val colorSpace = ciImage.colorSpace ?: return null

    // NSString and CFStringRef can be toll-free bridged
    val options =
      mapOf(
        CFBridgingRelease(kCGImageDestinationLossyCompressionQuality)
          as CIImageRepresentationOption to 1.0f
      )

    // Map<CIImageRepresentationOption, Float> is the expected type,
    // this cast is safe to suppress
    @Suppress("UNCHECKED_CAST")
    return imageContext.JPEGRepresentationOfImage(ciImage, colorSpace, options as Map<Any?, *>)
  }
}
