package io.dyte.core.media

import DyteWebRTC.DyteRTCCVPixelBuffer
import DyteWebRTC.DyteRTCVideoCapturer
import DyteWebRTC.DyteRTCVideoCapturerDelegateProtocol
import DyteWebRTC.DyteRTCVideoFrame
import DyteWebRTC.RTCVideoRotation_0
import DyteWebRTC.RTCVideoRotation_180
import DyteWebRTC.RTCVideoRotation_270
import DyteWebRTC.RTCVideoRotation_90
import kotlinx.cinterop.CPointer
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.addressOf
import kotlinx.cinterop.alloc
import kotlinx.cinterop.convert
import kotlinx.cinterop.memScoped
import kotlinx.cinterop.pin
import kotlinx.cinterop.ptr
import kotlinx.cinterop.value
import platform.CFNetwork.CFHTTPMessageAppendBytes
import platform.CFNetwork.CFHTTPMessageCopyBody
import platform.CFNetwork.CFHTTPMessageCopyHeaderFieldValue
import platform.CFNetwork.CFHTTPMessageCreateEmpty
import platform.CFNetwork.CFHTTPMessageIsHeaderComplete
import platform.CFNetwork.CFHTTPMessageRef
import platform.CoreFoundation.CFRelease
import platform.CoreFoundation.CFStringRef
import platform.CoreFoundation.CFTypeRef
import platform.CoreFoundation.kCFAllocatorDefault
import platform.CoreImage.CIContext
import platform.CoreImage.CIImage
import platform.CoreVideo.CVPixelBufferCreate
import platform.CoreVideo.CVPixelBufferLockBaseAddress
import platform.CoreVideo.CVPixelBufferRef
import platform.CoreVideo.CVPixelBufferRefVar
import platform.CoreVideo.CVPixelBufferRelease
import platform.CoreVideo.CVPixelBufferUnlockBaseAddress
import platform.CoreVideo.kCVPixelFormatType_32BGRA
import platform.CoreVideo.kCVReturnSuccess
import platform.Foundation.CFBridgingRelease
import platform.Foundation.CFBridgingRetain
import platform.Foundation.NSData
import platform.Foundation.NSInputStream
import platform.Foundation.NSLog
import platform.Foundation.NSProcessInfo
import platform.Foundation.NSStream
import platform.Foundation.NSStreamDelegateProtocol
import platform.Foundation.NSStreamEvent
import platform.Foundation.NSStreamEventEndEncountered
import platform.Foundation.NSStreamEventErrorOccurred
import platform.Foundation.NSStreamEventHasBytesAvailable
import platform.Foundation.NSStreamEventOpenCompleted
import platform.Foundation.NSString
import platform.ImageIO.CGImagePropertyOrientation
import platform.ImageIO.kCGImagePropertyOrientationDown
import platform.ImageIO.kCGImagePropertyOrientationLeft
import platform.ImageIO.kCGImagePropertyOrientationRight
import platform.darwin.ByteVar
import platform.darwin.NSEC_PER_SEC

const val KMaxReadLength = 20024L

class ScreenCapturer(private val dr: DyteRTCVideoCapturerDelegateProtocol) :
  DyteRTCVideoCapturer(dr), NSStreamDelegateProtocol {
  private var connection: SocketConnection? = null
  private var message: Message? = null
  private var readLength = 0L
  private var startTimeStampNs = -1L
  var eventsDelegate: CapturerEventsDelegate? = null

  fun setConnection(connection: SocketConnection?) {
    if (this.connection != connection) {
      this.connection?.close()
      this.connection = connection
    }
  }

  fun startCaptureWithConnection(connection: SocketConnection) {
    startTimeStampNs = -1
    this.connection = connection
    this.message = null
    this.connection?.openWithStreamDelegate(this)
  }

  fun stopCapture() {
    this.connection?.close()
    this.connection = null
  }

  @OptIn(ExperimentalForeignApi::class)
  private fun readBytesFromStream(stream: NSInputStream) {
    if (!stream.hasBytesAvailable()) {
      return
    }

    if (message == null) {
      message = Message()
      readLength = KMaxReadLength

      message?.didComplete = { success, message ->
        if (success) {
          didCaptureVideoFrame(
            message.imageBuffer as CVPixelBufferRef,
            message.imageOrientation.convert(),
          )
        } else {
          NSLog("DYTESS: didComplete FAILED")
        }
        message.dealloc()
        this.message = null
      }
    }

    val buffer = UByteArray(readLength.toInt())
    val bfpinned = buffer.pin()
    val buffer0 = bfpinned.addressOf(0)
    val numberOfBytesRead = stream.read(buffer0, readLength.toULong())
    if (numberOfBytesRead < 0) {
      NSLog("DYTESS: error reading bytes from stream")
      return
    }
    //        NSLog("DYTESS: READ $numberOfBytesRead")

    readLength = message?.appendBytes(buffer0, numberOfBytesRead.toULong()) ?: KMaxReadLength
    bfpinned.unpin()
    if (readLength == -1L || readLength > KMaxReadLength) {
      readLength = KMaxReadLength
    }
  }

  @OptIn(ExperimentalForeignApi::class)
  private fun didCaptureVideoFrame(
    pixelBuffer: CVPixelBufferRef,
    orientation: CGImagePropertyOrientation,
  ) {
    val currentTimeStampNs =
      (NSProcessInfo.processInfo.systemUptime * NSEC_PER_SEC.toDouble()).toLong()
    if (startTimeStampNs < 0) {
      startTimeStampNs = currentTimeStampNs
    }

    val rtcPixelBuffer = DyteRTCCVPixelBuffer(pixelBuffer)
    val frameTimeStampNs = currentTimeStampNs - startTimeStampNs

    val rotation =
      when (orientation) {
        kCGImagePropertyOrientationLeft -> RTCVideoRotation_90
        kCGImagePropertyOrientationDown -> RTCVideoRotation_180
        kCGImagePropertyOrientationRight -> RTCVideoRotation_270
        else -> RTCVideoRotation_0
      }

    val videoFrame = DyteRTCVideoFrame(rtcPixelBuffer, rotation, frameTimeStampNs)
    dr?.capturer(this, videoFrame)
  }

  override fun stream(aStream: NSStream, handleEvent: NSStreamEvent) {
    when (handleEvent) {
      NSStreamEventOpenCompleted -> {}
      NSStreamEventHasBytesAvailable -> {
        readBytesFromStream(aStream as NSInputStream)
      }
      NSStreamEventEndEncountered -> {
        stopCapture()
        eventsDelegate?.capturerDidEnd(this)
      }
      NSStreamEventErrorOccurred -> {}
    }
  }
}

@OptIn(ExperimentalForeignApi::class)
class Message {

  var imageBuffer: CVPixelBufferRef? = null
  var imageOrientation: Int = 0
  private var framedMessage: CFHTTPMessageRef? = null
  var didComplete: ((Boolean, Message) -> Unit)? = null

  fun dealloc() {
    CVPixelBufferRelease(imageBuffer)
  }

  fun appendBytes(buffer: CPointer<ByteVar>, length: ULong): Long {
    if (framedMessage == null) {
      framedMessage = CFHTTPMessageCreateEmpty(kCFAllocatorDefault, false)
    }

    CFHTTPMessageAppendBytes(framedMessage, buffer, length.toLong())
    if (!CFHTTPMessageIsHeaderComplete(framedMessage)) {
      return -1
    }
    val contentLength =
      CFHTTPMessageCopyHeaderFieldValue(framedMessage, "Content-Length".toCString())
        ?.toKString()
        ?.toLong() ?: return -1

    val bodyData = CFHTTPMessageCopyBody(framedMessage)?.toNSData()
    val bodyLength = bodyData?.length?.toLong() ?: 0

    val missingBytesCount = contentLength - bodyLength
    //        NSLog("DYTESS: Missing bytes $missingBytesCount")
    if (missingBytesCount == 0L) {
      val success = unwrapMessage(framedMessage!!)
      didComplete?.invoke(success, this)

      CFRelease(framedMessage)
      framedMessage = null
    }

    return missingBytesCount
  }

  private fun unwrapMessage(framedMessage: CFHTTPMessageRef): Boolean {
    val widthStr =
      CFHTTPMessageCopyHeaderFieldValue(framedMessage, "Buffer-Width".toCString())?.toKString()
    val heightStr =
      CFHTTPMessageCopyHeaderFieldValue(framedMessage, "Buffer-Height".toCString())?.toKString()
    val orientationStr =
      CFHTTPMessageCopyHeaderFieldValue(framedMessage, "Buffer-Orientation".toCString())
        ?.toKString()
    val width = widthStr?.toFloatOrNull() ?: return false
    val height = heightStr?.toFloatOrNull() ?: return false
    imageOrientation = orientationStr?.toIntOrNull() ?: 0

    val messageData = CFHTTPMessageCopyBody(framedMessage)?.toNSData()

    val status = memScoped {
      val buf = alloc<CVPixelBufferRefVar>()
      val result =
        CVPixelBufferCreate(
          kCFAllocatorDefault,
          width.toULong(),
          height.toULong(),
          kCVPixelFormatType_32BGRA,
          null,
          buf.ptr,
        )
      imageBuffer = buf.value
      result
    }

    if (status != kCVReturnSuccess) {
      NSLog("DYTESS:CVPixelBufferCreate failed")
      return false
    }

    messageData?.let { copyImageData(it, imageBuffer!!) }

    return true
  }

  private fun copyImageData(data: NSData?, pixelBuffer: CVPixelBufferRef?) {
    pixelBuffer?.let { buffer ->
      CVPixelBufferLockBaseAddress(buffer, 0u)

      val image = data?.let { CIImage.imageWithData(it) }

      image?.let { img ->
        val context = CIContext.contextWithOptions(null)
        context.render(img, toCVPixelBuffer = buffer)
      }

      CVPixelBufferUnlockBaseAddress(buffer, 0u)
    }
  }

  private fun releaseImageBuffer() {
    if (imageBuffer != null) {
      CVPixelBufferRelease(imageBuffer)
      imageBuffer = null
    }
  }

  private fun destroyMessage() {
    releaseImageBuffer()
  }

  protected fun finalize() {
    destroyMessage()
  }
}

@OptIn(ExperimentalForeignApi::class)
private fun CFTypeRef.toNSData(): NSData {
  return CFBridgingRelease(this) as NSData
}

@OptIn(ExperimentalForeignApi::class)
private fun CFTypeRef.toKString(): String {
  return CFBridgingRelease(this) as String
}

@OptIn(ExperimentalForeignApi::class)
fun String.toCString(): CFStringRef {
  // Using 'CreateNSStringFromKString(String?):
  // kotlinx.cinterop.NativePtr /* = kotlin.native.internal.NativePtr */' is an error.
  // Use plain Kotlin cast of String to NSString.
  @Suppress("CAST_NEVER_SUCCEEDS") return CFBridgingRetain(this as NSString) as CFStringRef
}
