package io.dyte.core.media

import io.dyte.core.native.sockaddr_un
import kotlinx.cinterop.Arena
import kotlinx.cinterop.BetaInteropApi
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.ObjCAction
import kotlinx.cinterop.alloc
import kotlinx.cinterop.autoreleasepool
import kotlinx.cinterop.convert
import kotlinx.cinterop.ptr
import kotlinx.cinterop.reinterpret
import kotlinx.cinterop.value
import platform.CoreFoundation.CFReadStreamRefVar
import platform.CoreFoundation.CFStreamCreatePairWithSocket
import platform.CoreFoundation.CFWriteStreamRefVar
import platform.CoreFoundation.kCFAllocatorDefault
import platform.Foundation.CFBridgingRelease
import platform.Foundation.NSInputStream
import platform.Foundation.NSLog
import platform.Foundation.NSOutputStream
import platform.Foundation.NSQualityOfServiceUserInitiated
import platform.Foundation.NSRunLoop
import platform.Foundation.NSRunLoopCommonModes
import platform.Foundation.NSStreamDelegateProtocol
import platform.Foundation.NSString
import platform.Foundation.NSThread
import platform.Foundation.performSelector
import platform.Foundation.run
import platform.darwin.DISPATCH_SOURCE_TYPE_READ
import platform.darwin.NSObject
import platform.darwin.dispatch_resume
import platform.darwin.dispatch_source_cancel
import platform.darwin.dispatch_source_create
import platform.darwin.dispatch_source_set_event_handler
import platform.darwin.dispatch_source_t
import platform.darwin.sel_registerName
import platform.posix.AF_UNIX
import platform.posix.SOCK_STREAM
import platform.posix.accept
import platform.posix.bind
import platform.posix.close
import platform.posix.listen
import platform.posix.memset
import platform.posix.socket
import platform.posix.strncpy
import platform.posix.unlink

class SocketConnection : NSObject() {
  private var serverSocket: Int? = null
  private var listeningSource: dispatch_source_t = null

  private lateinit var networkThread: NSThread
  @OptIn(ExperimentalForeignApi::class) private val arena = Arena()

  private var inputStream: NSInputStream? = null
  private var outputStream: NSOutputStream? = null

  @OptIn(BetaInteropApi::class)
  fun setupNetworkThread() {
    networkThread = NSThread {
      do {
        autoreleasepool { NSRunLoop.currentRunLoop.run() }
      } while (!NSThread.currentThread.isCancelled())
    }

    networkThread.qualityOfService = NSQualityOfServiceUserInitiated
  }

  @OptIn(ExperimentalForeignApi::class)
  fun setupSocketWithFilePath(filepath: NSString): Boolean =
    with(arena) {
      val addr = alloc<sockaddr_un>()

      memset(addr.ptr, 0, sockaddr_un.size.convert())
      addr.sun_family = AF_UNIX.convert()

      // https://man7.org/linux/man-pages/man7/unix.7.html
      if (filepath.length > 108UL) { // size of addr.sun_path
        println("File path too long")
        return false
      }

      unlink(filepath.toString())
      strncpy(addr.sun_path, filepath.toString(), 108UL) // size of sun_path

      val status = bind(serverSocket!!, addr.ptr.reinterpret(), sockaddr_un.size.convert())
      if (status < 0) {
        println("failure in binding socket")
        return false
      }

      return true
    }

  @OptIn(ExperimentalForeignApi::class)
  fun openWithStreamDelegate(streamDelegate: NSStreamDelegateProtocol) {
    val status = listen(serverSocket!!, 10)
    if (status < 0) {
      NSLog("DYTESS: failure in socket listening")
      return
    }

    val listeningSource =
      dispatch_source_create(DISPATCH_SOURCE_TYPE_READ, serverSocket!!.convert(), 0u, null)

    val dispatchHandler: () -> Unit = {
      val clientSocket = accept(serverSocket!!, null, null)
      if (clientSocket >= 0) {
        arena.apply {
          val readStreamPtr = alloc<CFReadStreamRefVar>()
          val writeStreamPtr = alloc<CFWriteStreamRefVar>()

          CFStreamCreatePairWithSocket(
            kCFAllocatorDefault,
            clientSocket,
            readStreamPtr.ptr,
            writeStreamPtr.ptr,
          )

          val readStream = readStreamPtr.value
          val writeStream = writeStreamPtr.value

          inputStream = CFBridgingRelease(readStream) as NSInputStream
          inputStream!!.delegate = streamDelegate
          inputStream!!.setProperty("kCFBooleanTrue", "kCFStreamPropertyShouldCloseNativeSocket")

          outputStream = CFBridgingRelease(writeStream) as NSOutputStream?
          outputStream!!.setProperty("kCFBooleanTrue", "kCFStreamPropertyShouldCloseNativeSocket")

          networkThread.start()
          performSelector(sel_registerName("scheduleStreams"), networkThread, null, true)

          inputStream!!.open()
          outputStream!!.open()
        }
      } else {
        println("SS: failure accepting connection")
      }
    }

    dispatch_source_set_event_handler(listeningSource, dispatchHandler)

    this.listeningSource = listeningSource
    dispatch_resume(listeningSource)
  }

  @OptIn(ExperimentalForeignApi::class)
  fun close() {
    performSelector(sel_registerName("unscheduleStreams"), networkThread, null, true)

    inputStream?.delegate = null
    outputStream?.delegate = null

    inputStream?.close()
    outputStream?.close()

    networkThread.cancel()

    dispatch_source_cancel(listeningSource)
    close(serverSocket!!)
    arena.clear()
  }

  @OptIn(BetaInteropApi::class)
  @ObjCAction
  fun scheduleStreams() {
    inputStream!!.scheduleInRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
    outputStream!!.scheduleInRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
  }

  @OptIn(BetaInteropApi::class)
  @ObjCAction
  fun unscheduleStreams() {
    inputStream!!.removeFromRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
    outputStream!!.removeFromRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
  }

  companion object {
    fun initWithFilePath(filepath: String): SocketConnection? {
      val sc = SocketConnection()

      sc.setupNetworkThread()

      sc.serverSocket = socket(AF_UNIX, SOCK_STREAM, 0)
      if (sc.serverSocket!! < 0) {
        println("Failure creating socket")
        return null
      }

      // Using 'CreateNSStringFromKString(String?):
      // kotlinx.cinterop.NativePtr /* = kotlin.native.internal.NativePtr */' is an error.
      // Use plain Kotlin cast of String to NSString.
      @Suppress("CAST_NEVER_SUCCEEDS")
      if (!sc.setupSocketWithFilePath(filepath as NSString)) {
        close(sc.serverSocket!!)
        return null
      }

      return sc
    }
  }
}
