package dyte.io.uikit.molecules

import android.content.Context
import android.os.CountDownTimer
import android.util.AttributeSet
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.Utils
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.view.DyteGridPaginatorView
import dyte.io.uikit.view.DyteRecordingIndicator
import dyte.io.uikit.wiptoken.DyteUITokens
import io.dyte.core.DyteMobileClient
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteRoomParticipants

/*
TODO:
 1. Fix size of SwitchCameraImageView 48x48dp
 2. Set minHeight to 56dp
 2. Add ripple touch effect on SwitchCameraImageView
 */
class DyteMeetingHeaderView : ConstraintLayout {
  private lateinit var meetingTitleTextView: TextView
  private lateinit var meetingMetaTextView: TextView
  private lateinit var switchCameraImageView: ImageView
  private lateinit var recordingIndicator: DyteRecordingIndicator
  private lateinit var paginator: DyteGridPaginatorView

  private lateinit var meeting: DyteMobileClient
  private var meetingTimeElapsed: String = "00:00"

  // TODO : this logic can not be in view
  private val timer = object : CountDownTimer(Long.MAX_VALUE, 1000) {
    override fun onTick(p0: Long) {
      meeting.let {
        try {
          val diff = Utils.getDiff(Utils.getUtcTimeNow(), it.meta.meetingStartedTimestamp)
          meetingTimeElapsed = diff
          refreshMeetingMeta()
        } catch (e: Exception) {
          e.printStackTrace()
        }
      }
    }

    override fun onFinish() {
    }
  }

  private val meetingRoomEventsListener = object : DyteRecordingEventsListener {
    override fun onMeetingRecordingStarted() {
      super.onMeetingRecordingStarted()
      showRecordingIndicator()
    }

    override fun onMeetingRecordingEnded() {
      super.onMeetingRecordingEnded()
      hideRecordingIndicator()
    }
  }

  private val selfEventsListener = object : DyteSelfEventsListener {
    override fun onVideoUpdate(videoEnabled: Boolean) {
      super.onVideoUpdate(videoEnabled)
      if (videoEnabled) {
        switchCameraImageView.visible()
      } else {
        switchCameraImageView.gone()
      }
    }
  }

  private val participantEventsListener = object : DyteParticipantEventsListener {
    override fun onUpdate(participants: DyteRoomParticipants) {
      super.onUpdate(participants)
      refreshMeetingMeta()
    }
  }

  constructor(context: Context) : super(context) {
    init()
  }

  constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
    init()
  }

  constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
    context,
    attrs,
    defStyleAttr
  ) {
    init()
  }

  constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int, defStyleRes: Int) : super(
    context,
    attrs,
    defStyleAttr,
    defStyleRes
  ) {
    init()
  }

  private fun init(uiTokens: DyteUITokens = tokens) {
    inflate(context, R.layout.view_dyte_meeting_header, this)
    meetingTitleTextView = findViewById(R.id.textview_dyte_header_meeting_title)
    meetingMetaTextView = findViewById(R.id.textview_dyte_header_meeting_meta)
    switchCameraImageView = findViewById(R.id.imageview_dyte_header_switch_camera)
    recordingIndicator = findViewById(R.id.dyterecordingindicator_dyte_header)
    paginator = findViewById(R.id.imageview_dyte_grid_pagination)
    applyUiTokens(uiTokens)
    paginator.refresh(DyteUIKitBuilder.meeting)
  }

  fun applyUiTokens(uiTokens: DyteUITokens) {
    setBackgroundColor(uiTokens.colors.background.shade900)
    recordingIndicator.applyUiTokens(uiTokens)
  }

  fun activate(meeting: DyteMobileClient) {
    this.meeting = meeting
    meetingTitleTextView.text = this.meeting.meta.meetingTitle
    refreshMeetingMeta()
    refreshRecordingIndicator()

    switchCameraImageView.setOnClickListener {
      onSwitchCameraClicked()
    }

    this.meeting.addSelfEventsListener(selfEventsListener)
    this.meeting.addRecordingEventsListener(meetingRoomEventsListener)
    this.meeting.addParticipantEventsListener(participantEventsListener)
    timer.start()
  }

  override fun onDetachedFromWindow() {
    super.onDetachedFromWindow()
    meeting.removeSelfEventsListener(selfEventsListener)
    meeting.removeParticipantEventsListener(participantEventsListener)
    meeting.removeRecordingEventsListener(meetingRoomEventsListener)
    timer.cancel()
  }

  private fun refreshRecordingIndicator() {
    if (meeting.recording.recordingState == RECORDING) {
      showRecordingIndicator()
    } else {
      hideRecordingIndicator()
    }
  }

  private fun refreshMeetingMeta() {
    meetingMetaTextView.text =
      "${meeting.participants.joined.size} participants ● $meetingTimeElapsed"
    if (meeting.participants.pageCount == 1) {
      paginator.gone()
    } else {
      paginator.visible()

    }
  }

  private fun onSwitchCameraClicked() {
    val selectedVideoDevice = meeting.localUser.getSelectedVideoDevice()
    val videoDevices = meeting.localUser.getVideoDevices()
    meeting.localUser.setVideoDevice(
      videoDevices.first { device -> device.type != selectedVideoDevice.type }
    )
  }

  private fun showRecordingIndicator() {
    recordingIndicator.visibility = View.VISIBLE
  }

  private fun hideRecordingIndicator() {
    recordingIndicator.visibility = View.GONE
  }
}