package dyte.io.uikit.screens.chat

import android.Manifest
import android.app.Activity
import android.app.Activity.RESULT_OK
import android.app.Dialog
import android.content.Intent
import android.content.pm.PackageManager
import android.content.res.ColorStateList
import android.graphics.Rect
import android.graphics.drawable.GradientDrawable
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.provider.MediaStore
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.fragment.app.DialogFragment
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.R.dimen
import dyte.io.uikit.atoms.DyteIconView
import dyte.io.uikit.atoms.DyteImageBrandButton
import dyte.io.uikit.atoms.DyteInputFieldAtom
import dyte.io.uikit.atoms.DyteLabelAtom
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.FileUtils
import dyte.io.uikit.utils.FileUtils.ImageFile
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.setBackground
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.listeners.DyteChatEventsListener
import kotlinx.coroutines.launch

class DyteChatBottomSheet : BottomSheetDialogFragment() {
  private lateinit var sendButton: DyteImageBrandButton
  private lateinit var messageEditText: DyteInputFieldAtom
  private lateinit var emptyChatView: LinearLayout
  private lateinit var chatRecyclerView: RecyclerView
  private lateinit var messageListAdapter: ChatMessageListAdapter
  private lateinit var optionsButton: ImageView
  private lateinit var optionsViewHandle: FrameLayout
  private lateinit var sendFileOptionView: LinearLayout
  private lateinit var sendImageOptionView: LinearLayout

  private lateinit var divFile: DyteIconView
  private lateinit var dlaFile: DyteLabelAtom
  private lateinit var divImage: DyteIconView
  private lateinit var dlaImage: DyteLabelAtom

  private lateinit var chatInputView: View

  private val meeting by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }

  private var chatEventsListener = object : DyteChatEventsListener {
    override fun onChatUpdates(messages: List<DyteChatMessage>) {
      super.onChatUpdates(messages)
      refreshMessages(messages)
    }
  }

  private lateinit var onReadImagesPermissionGranted: () -> Unit

  private val requestReadImagesPermissionLauncher = registerForActivityResult(
    ActivityResultContracts.RequestPermission()
  ) { isGranted: Boolean ->
    if (isGranted) {
      onReadImagesPermissionGranted()
    } else {
      // Permission request was denied.
      showReadImagesPermissionDenied()
    }
  }

  private lateinit var onReadFilesPermissionGranted: () -> Unit

  private val requestReadFilesPermissionLauncher = registerForActivityResult(
    ActivityResultContracts.RequestMultiplePermissions()
  ) { permissionGrantMap: Map<String, Boolean> ->
    var allReadMediaPermissionsGranted = true
    for (permission in permissionGrantMap.keys) {
      if (permissionGrantMap[permission] == false) {
        allReadMediaPermissionsGranted = false
        break
      }
    }

    if (allReadMediaPermissionsGranted) {
      onReadFilesPermissionGranted()
    } else {
      showReadFilesPermissionDenied()
    }
  }

  private val chatFileDownloader: ChatFileDownloader by lazy { ChatFileDownloader() }

  override fun onCreate(savedInstanceState: Bundle?) {
    setStyle(DialogFragment.STYLE_NORMAL, R.style.AppBottomSheetDialogTheme)
    super.onCreate(savedInstanceState)
  }

  override fun onCreateView(
    inflater: LayoutInflater,
    container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View? {
    return inflater.inflate(R.layout.bottom_sheet_chat, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)
    (requireDialog() as BottomSheetDialog).dismissWithAnimation = true

    val shape = GradientDrawable()
    shape.setColor(tokens.colors.background.shade1000)
    val radius = tokens.borderRadius.getRadius(
      BorderRadiusSize.TWO,
      requireContext().resources.displayMetrics.density
    )
    shape.cornerRadii = floatArrayOf(radius, radius, radius, radius, 0f, 0f, 0f, 0f)
    view.findViewById<View>(R.id.clChatContainer).background = shape

    setUpHeader(view)
    setUpChildViews(view)
    meeting.addChatEventsListener(chatEventsListener)
    refreshMessages(meeting.chat.messages)

    chatInputView = view.findViewById(R.id.chat_input_view)
    chatInputView.setBackgroundColor(tokens.colors.background.shade900)
    messageEditText.setBackground(tokens.borderRadius, tokens.colors.background.shade900)

    if (meeting.localUser.permissions.chat.canSendText || meeting.localUser.permissions.chat.canSendFiles) {
      chatInputView.visible()

      if (meeting.localUser.permissions.chat.canSendFiles) {
        optionsButton.visible()
      } else {
        optionsButton.gone()
      }

      if (meeting.localUser.permissions.chat.canSendText) {
        messageEditText.visible()
        sendButton.visible()
      } else {
        messageEditText.gone()
        sendButton.gone()
      }
    } else {
      chatInputView.gone()
    }

    DyteUIKitBuilder.dyteUIKit.onChatScreenOpened()
  }

  override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
    val dialog = super.onCreateDialog(savedInstanceState)

    dialog.setOnShowListener {
      val bottomSheet: FrameLayout = (dialog as BottomSheetDialog)
        .findViewById(com.google.android.material.R.id.design_bottom_sheet)
        ?: return@setOnShowListener

      val behavior = BottomSheetBehavior.from(bottomSheet)
      behavior.state = BottomSheetBehavior.STATE_EXPANDED
      behavior.skipCollapsed = true
    }
    return dialog
  }

  override fun onDestroyView() {
    super.onDestroyView()
    // DyteUIKitBuilder.dyteUIKit.onChatScreenClosed()
    meeting.removeChatEventsListener(chatEventsListener)
  }

  override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
    super.onActivityResult(requestCode, resultCode, data)

    when (requestCode) {
      REQUEST_CODE_SELECT_IMAGE -> if (resultCode == Activity.RESULT_OK && data != null) {
        val imageUris = extractSelectedImagesUris(data)
        sendChatImageMessages(imageUris)
      }
      REQUEST_CODE_SELECT_FILE -> if (resultCode == Activity.RESULT_OK && data != null) {
        val fileUris = extractSelectedFilesUris(data)
        sendChatFileMessages(fileUris)
      }
    }
  }

  private fun setUpHeader(view: View) {
    view.findViewById<TextView>(R.id.header_text_view_title).text = "Chat"
    val closeButton = view.findViewById<View>(R.id.header_button_close)
    closeButton.backgroundTintList = ColorStateList.valueOf(tokens.colors.background.shade1000)
    closeButton.setOnClickListener {
      dismiss()
    }
  }

  private fun setUpChildViews(view: View) {
    emptyChatView = view.findViewById(R.id.chat_empty_view)
    chatRecyclerView = view.findViewById(R.id.chat_recycler_view)
    messageEditText = view.findViewById(R.id.chat_edit_text_message)
    sendButton = view.findViewById(R.id.chat_image_view_send)
    sendButton.setImageResource(R.drawable.ic_chat_send_24)
    optionsButton = view.findViewById(R.id.chat_image_view_options)
    optionsViewHandle = view.findViewById(R.id.chat_options_view_handle)
    sendFileOptionView = view.findViewById(R.id.chat_option_send_file)
    sendImageOptionView = view.findViewById(R.id.chat_option_send_image)

    dlaFile = view.findViewById(R.id.dlaFile)
    divFile = view.findViewById(R.id.divFile)
    dlaImage = view.findViewById(R.id.dlaImage)
    divImage = view.findViewById(R.id.divImage)

    dlaFile.setTextColor(tokens.colors.text.onBackground.shade600)
    divFile.setBackground(tokens.borderRadius, tokens.colors.background.shade1000)
    divFile.imageTintList = ColorStateList.valueOf(tokens.colors.text.onBackground.shade600)

    dlaImage.setTextColor(tokens.colors.text.onBackground.shade600)
    divImage.setBackground(tokens.borderRadius, tokens.colors.background.shade1000)
    divImage.imageTintList = ColorStateList.valueOf(tokens.colors.text.onBackground.shade600)

    view.findViewById<View>(R.id.chat_options_view)
      .setBackground(tokens.borderRadius, tokens.colors.background.shade1000)

    // TODO: Check if hasFixedSize is needed for recycler view
    messageListAdapter = ChatMessageListAdapter(
      onImageClick = {
        showFullImage(it)
      },
      onDownloadFileClick = { fileName, link ->
        onDownloadFileClicked(fileName, link)
      }
    )
    chatRecyclerView.adapter = messageListAdapter
    chatRecyclerView.addItemDecoration(
      ChatItemDecoration(resources.getDimensionPixelSize(dimen.dyte_vertical_spacing_chat))
    )

    setUpTextInputViews()
    setUpMoreOptionsViews()
  }

  private fun setUpTextInputViews() {
    if (meeting.localUser.permissions.chat.canSendText.not()) {
      sendButton.visibility = View.INVISIBLE
      sendButton.isEnabled = false
      messageEditText.isEnabled = false
    } else {
      sendButton.setOnClickListener {
        onSendButtonClicked()
      }
    }
  }

  private fun setUpMoreOptionsViews() {
    if (meeting.localUser.permissions.chat.canSendFiles.not()) {
      optionsButton.visibility = View.INVISIBLE
      optionsButton.isEnabled = false
    } else {
      optionsButton.setOnClickListener {
        onOptionsButtonClicked()
      }

      optionsViewHandle.setOnClickListener {
        dismissMoreOptions()
      }

      sendFileOptionView.setOnClickListener {
        onSendFileOptionClicked()
        dismissMoreOptions()
      }

      sendImageOptionView.setOnClickListener {
        onSendImageOptionClicked()
        dismissMoreOptions()
      }
    }
  }

  private fun dismissMoreOptions() {
    optionsViewHandle.visibility = View.GONE
  }

  // TODO: Discuss if layout manager init code can be kept here for clarity instead of XML
  private fun refreshMessages(messages: List<DyteChatMessage>) {
    if (!isAdded) {
      return
    }

    if (messages.isNotEmpty()) {
      emptyChatView.visibility = View.GONE
      chatRecyclerView.visibility = View.VISIBLE
      messageListAdapter.submitList(ArrayList(messages))
    } else {
      emptyChatView.visibility = View.VISIBLE
      chatRecyclerView.visibility = View.GONE
    }
  }

  private fun onSendButtonClicked() {
    val message = messageEditText.text.toString().trim()
    if (message.isNotBlank()) {
        meeting.chat.sendTextMessage(message)
        messageEditText.text?.clear()
    }
  }

  private fun onOptionsButtonClicked() {
    if (optionsViewHandle.visibility != View.VISIBLE) {
      optionsViewHandle.visibility = View.VISIBLE
    } else {
      optionsViewHandle.visibility = View.GONE
    }
  }

  private fun onSendImageOptionClicked() {
    if (isReadImagesPermissionGranted()) {
      openImagePicker()
    } else {
      onReadImagesPermissionGranted = { openImagePicker() }
      requestReadImagesPermission()
    }
  }

  private fun onSendFileOptionClicked() {
    if (isReadFilesPermissionGranted()) {
      openFilePicker()
    } else {
      onReadFilesPermissionGranted = { openFilePicker() }
      requestReadFilesPermission()
    }
  }

  private fun showFullImage(imageUrl: String) {
    val viewImageIntent = DyteImageViewerActivity.getIntent(requireContext(), imageUrl)
    startActivity(viewImageIntent)
  }

  private fun openImagePicker() {
    val galleryIntent = Intent(Intent.ACTION_PICK, MediaStore.Images.Media.EXTERNAL_CONTENT_URI)
    galleryIntent.putExtra(Intent.EXTRA_ALLOW_MULTIPLE, true)
    startActivityForResult(galleryIntent, REQUEST_CODE_SELECT_IMAGE)
  }

  private fun openFilePicker() {
    val intent = Intent(Intent.ACTION_GET_CONTENT).apply {
      addCategory(Intent.CATEGORY_OPENABLE)
      putExtra(Intent.EXTRA_ALLOW_MULTIPLE, true)
      type = "*/*"
    }

    startActivityForResult(intent, REQUEST_CODE_SELECT_FILE)
  }

  private fun extractSelectedImagesUris(imageData: Intent): List<Uri> {
    return extractSelectedFilesUris(imageData)
  }

  private fun extractSelectedFilesUris(fileData: Intent): List<Uri> {
    val fileUris = mutableListOf<Uri>()
    if (fileData.clipData != null) { // Multiple files were selected
      val count = fileData.clipData?.itemCount ?: 0
      for (i in 0 until count) {
        val uri: Uri? = fileData.clipData?.getItemAt(i)?.uri
        uri?.let { fileUris.add(uri) }
      }
    } else if (fileData.data != null) { // Single file selected
      val uri: Uri? = fileData.data
      uri?.let { fileUris.add(uri) }
    }

    return fileUris
  }

  private fun sendChatImageMessages(imageUris: List<Uri>) {
    imageUris.forEach { uri ->
      meeting.chat.sendImageMessage(uri)
    }
  }

  private fun sendChatFileMessages(fileUris: List<Uri>) {
    fileUris.forEach { uri ->
      meeting.chat.sendFileMessage(uri)
    }
  }

  private fun isReadImagesPermissionGranted(): Boolean {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
      ContextCompat.checkSelfPermission(
        requireContext(),
        Manifest.permission.READ_MEDIA_IMAGES
      ) == PackageManager.PERMISSION_GRANTED
    } else {
      ContextCompat.checkSelfPermission(
        requireContext(),
        Manifest.permission.READ_EXTERNAL_STORAGE
      ) == PackageManager.PERMISSION_GRANTED
    }
  }

  private fun requestReadImagesPermission() {
    // Needed to read IMAGES created by other apps
    val readImagesPermission = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
      Manifest.permission.READ_MEDIA_IMAGES
    } else {
      Manifest.permission.READ_EXTERNAL_STORAGE
    }
    if (shouldShowRequestPermissionRationale(readImagesPermission)) {
      showReadImagePermissionPurpose(readImagesPermission)
    } else {
      requestReadImagesPermissionLauncher.launch(readImagesPermission)
    }
  }

  private fun showReadImagePermissionPurpose(readImagesPermission: String) {
    AlertDialog.Builder(requireContext()).apply {
      setTitle("Allow gallery access?")
      setMessage("We require the read permission to send the images selected by you in the Chat.")
      setPositiveButton("Allow") { _, _ ->
        requestReadImagesPermissionLauncher.launch(readImagesPermission)
      }
      setNegativeButton("Cancel") { _, _ -> }
    }.show()
  }

  private fun showReadImagesPermissionDenied() {
    AlertDialog.Builder(requireContext()).apply {
      setTitle("Feature unavailable")
      setMessage(
        "We require the Storage permission in order to send images in chat, " +
          "but it has been permanently denied. Please enable the \"Storage\" permission from app settings."
      )
      setPositiveButton("Got it") { _, _ -> }
    }.show()
  }

  private fun isReadFilesPermissionGranted(): Boolean {
    // Needed to read MEDIA files like Images, Audio and Video created by other apps
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
      for (permission in requiredMediaFilePermissionsApi33) {
        if (
          ContextCompat.checkSelfPermission(
            requireContext(),
            permission
          ) != PackageManager.PERMISSION_GRANTED
        ) {
          return false
        }
      }

      return true
    } else {
      return ContextCompat.checkSelfPermission(
        requireContext(),
        Manifest.permission.READ_EXTERNAL_STORAGE
      ) == PackageManager.PERMISSION_GRANTED
    }
  }

  private fun requestReadFilesPermission() {
    val readFilesPermissions = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
      requiredMediaFilePermissionsApi33
    } else {
      arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE)
    }
    if (shouldShowRequestFilesPermissionRationale(readFilesPermissions)) {
      showReadFilesPermissionPurpose(readFilesPermissions)
    } else {
      requestReadFilesPermissionLauncher.launch(readFilesPermissions)
    }
  }

  private fun shouldShowRequestFilesPermissionRationale(permissionList: Array<String>): Boolean {
    for (permission in permissionList) {
      if (shouldShowRequestPermissionRationale(permission)) {
        return true
      }
    }

    return false
  }

  private fun showReadFilesPermissionPurpose(readFilesPermissions: Array<String>) {
    AlertDialog.Builder(requireContext()).apply {
      setTitle("Allow files access?")
      setMessage("We require the read permission to send the files selected by you in the Chat.")
      setPositiveButton("Allow") { _, _ ->
        requestReadFilesPermissionLauncher.launch(readFilesPermissions)
      }
      setNegativeButton("Cancel") { _, _ -> }
    }.show()
  }

  private fun showReadFilesPermissionDenied() {
    AlertDialog.Builder(requireContext()).apply {
      setTitle("Feature unavailable")
      setMessage(
        "We require the Storage permission in order to send files in chat, " +
          "but it has been permanently denied. Please enable the \"Storage\" permission from app settings."
      )
      setPositiveButton("Got it") { _, _ -> }
    }.show()
  }

  private fun onDownloadFileClicked(fileName: String, link: String) {
    Toast.makeText(requireContext(), "Downloading file...", Toast.LENGTH_SHORT).show()
    downloadFile(fileName, link)
  }

  private fun downloadFile(name: String, url: String) {
    chatFileDownloader.enqueueDownload(requireContext(), name, url)
  }

  companion object {
    private const val REQUEST_CODE_SELECT_IMAGE = 201
    private const val REQUEST_CODE_SELECT_FILE = 202

    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    private val requiredMediaFilePermissionsApi33 = arrayOf(
      Manifest.permission.READ_MEDIA_IMAGES,
      Manifest.permission.READ_MEDIA_AUDIO,
      Manifest.permission.READ_MEDIA_VIDEO
    )
  }
}

internal class ChatItemDecoration(private val spacing: Int) : RecyclerView.ItemDecoration() {
  override fun getItemOffsets(
    outRect: Rect, view: View,
    parent: RecyclerView, state: RecyclerView.State
  ) {
    with(outRect) {
      if (parent.getChildAdapterPosition(view) == 0) {
        top = spacing
      }
      bottom = spacing
    }
  }
}