package dyte.io.uikit.screens.chat

import android.content.res.ColorStateList
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.recyclerview.widget.RecyclerView
import coil.load
import dyte.io.uikit.R
import dyte.io.uikit.atoms.DyteImageButton
import dyte.io.uikit.atoms.DyteLabelAtom
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.FileUtils
import dyte.io.uikit.utils.Utils.px
import dyte.io.uikit.wiptoken.DyteUITokens
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteMessageType.TEXT
import io.dyte.core.feat.DyteTextMessage

/**
 * Base class for Chat messages view holders.
 * Different messages like text, image, file can extend this class
 * and bind their specific message layout.
 *
 * Note: Always use [bindChatMessage] in the [RecyclerView.Adapter.onBindViewHolder]
 */
abstract class BaseMessageViewHolder<T : DyteChatMessage>(
  itemView: View
) : RecyclerView.ViewHolder(itemView) {

  /**
   * Workaround to allow a downcast of the DyteChatMessage to T.
   */
  @Suppress("UNCHECKED_CAST")
  internal fun bindChatMessage(message: DyteChatMessage) {
    message as T
    bind(message)
  }

  abstract fun bind(message: T)
}

class TextMessageViewHolder(itemView: View) : BaseMessageViewHolder<DyteTextMessage>(itemView) {
  private val senderNameTextView: TextView =
    itemView.findViewById(R.id.chat_message_text_view_sender_name)
  private val messageTextView: TextView = itemView.findViewById(R.id.chat_message_text_view)
  private val timeTextView: TextView = itemView.findViewById(R.id.chat_message_text_view_time)

  override fun bind(message: DyteTextMessage) {
    senderNameTextView.text = message.displayName
    if (message.type == TEXT) {
      messageTextView.text = message.message
    } else {
      messageTextView.text = "Message type not supported yet"
    }
    timeTextView.text = message.time

    timeTextView.setTextColor(tokens.colors.text.onBackground.shade1000)

    timeTextView.textSize = 7.px.toFloat()
    senderNameTextView.textSize = 8.px.toFloat()
    messageTextView.textSize = 8.px.toFloat()
  }
}

class ImageMessageViewHolder(
  itemView: View,
  private val onImageClick: (String) -> Unit
) : BaseMessageViewHolder<DyteImageMessage>(itemView) {
  private val senderNameTextView: TextView =
    itemView.findViewById(R.id.chat_image_message_text_view_sender_name)
  private val messageImageView: ImageView =
    itemView.findViewById(R.id.chat_image_message_image_view)
  private val messageTimeTextView: TextView =
    itemView.findViewById(R.id.chat_image_message_text_view_time)

  override fun bind(message: DyteImageMessage) {
    senderNameTextView.text = message.displayName
    messageImageView.load(message.link)
    messageTimeTextView.text = message.time
    messageImageView.setOnClickListener { onImageClick(message.link) }

    messageTimeTextView.setTextColor(tokens.colors.text.onBackground.shade1000)

    messageTimeTextView.textSize = 7.px.toFloat()
    senderNameTextView.textSize = 8.px.toFloat()
  }
}

internal class FileMessageViewHolder(
  itemView: View,
  private val onDownloadFileClick: (fileName: String, link: String) -> Unit
) : BaseMessageViewHolder<DyteFileMessage>(itemView) {
  private val senderNameTextView: TextView =
    itemView.findViewById(R.id.dytelabelatom_chat_file_message_sender_name)
  private val messageTimeTextView: TextView =
    itemView.findViewById(R.id.dytelabelatom_chat_file_message_time)
  private val fileNameTextView: DyteLabelAtom =
    itemView.findViewById(R.id.dytelabelatom_chat_file_message_file_name)
  private val fileDetailsTextView: DyteLabelAtom =
    itemView.findViewById(R.id.dytelabelatom_chat_file_message_file_details)
  private val downloadFileButton: DyteImageButton =
    itemView.findViewById(R.id.dyteimagebutton_chat_file_message_download)
  private val fileDetailsContainerView: View =
    itemView.findViewById(R.id.constraintlayout_chat_file_message_file_details)

  init {
    applyUiTokens(tokens)
  }

  override fun bind(message: DyteFileMessage) {
    senderNameTextView.text = message.displayName
    messageTimeTextView.text = message.time
    fileNameTextView.text = message.name

    messageTimeTextView.textSize = 7.px.toFloat()
    senderNameTextView.textSize = 8.px.toFloat()

    fileDetailsTextView.text =
      "${message.extension.uppercase()} | ${FileUtils.getHumanReadableSize(message.size)}"

    downloadFileButton.setOnClickListener {
      onDownloadFileClick(message.name, message.link)
    }
  }

  private fun applyUiTokens(uiTokens: DyteUITokens) {
    messageTimeTextView.setTextColor(uiTokens.colors.text.onBackground.shade1000)
    fileNameTextView.setTextColor(uiTokens.colors.text.onBackground.shade1000)
    fileDetailsTextView.setTextColor(uiTokens.colors.text.onBackground.shade700)
    fileDetailsContainerView.backgroundTintList =
      ColorStateList.valueOf(uiTokens.colors.background.shade900)
    downloadFileButton.backgroundTintList =
      ColorStateList.valueOf(uiTokens.colors.background.shade800)
    downloadFileButton.imageTintList =
      ColorStateList.valueOf(uiTokens.colors.text.onBackground.shade1000)
  }

  companion object {
    private val DyteFileMessage.extension: String
      get() {
        return name.substringAfterLast('.', "")
          .substringBefore('?')
      }
  }
}
