package dyte.io.uikit.screens.meetingoptions

import android.graphics.drawable.GradientDrawable
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.fragment.app.DialogFragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleObserver
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.screens.chat.DyteChatBottomSheet
import dyte.io.uikit.screens.participant.DyteParticipantsFragment
import dyte.io.uikit.screens.plugins.DytePluginsBottomSheet
import dyte.io.uikit.screens.polls.DytePollsBottomSheet
import dyte.io.uikit.screens.settings.DyteSettingsFragment
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.Utils.showToast
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener

class DyteMeetingOptionBottomSheet : BottomSheetDialogFragment() {
  private lateinit var viewParticipants: View
  private lateinit var viewSettings: View
  private lateinit var viewPlugins: View
  private lateinit var viewChat: View
  private lateinit var viewRecording: View
  private lateinit var viewMuteAll: View
  private lateinit var tvRecording: TextView

  private lateinit var viewNewChat: View
  private lateinit var viewNewPoll: View

  private val chatEventsListener = object : DyteChatEventsListener {
    override fun onChatUpdates(messages: List<DyteChatMessage>) {
      super.onChatUpdates(messages)
      viewNewChat.visible()
    }
  }

  private val pollEventsListener = object : DytePollEventsListener {
    override fun onNewPoll(poll: DytePollMessage) {
      super.onNewPoll(poll)
      viewNewPoll.visible()
    }
  }

  private val meetingEventListener = object : DyteRecordingEventsListener {
    override fun onMeetingRecordingEnded() {
      super.onMeetingRecordingEnded()
      if (!isAdded) {
        return
      }
      refreshRecordingState()
    }

    override fun onMeetingRecordingStarted() {
      super.onMeetingRecordingStarted()
      if (!isAdded) {
        return
      }
      refreshRecordingState()
    }

    override fun onMeetingRecordingStopError(e: Exception) {
      super.onMeetingRecordingStopError(e)
      if (!isAdded) {
        return
      }
      requireContext().showToast(requireView(), "Failed to stop recording, Please try again later")
      refreshRecordingState()
    }
  }

  private val meeting by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }

  override fun onCreate(savedInstanceState: Bundle?) {
    setStyle(DialogFragment.STYLE_NORMAL, R.style.AppBottomSheetDialogTheme)
    super.onCreate(savedInstanceState)
  }

  override fun onCreateView(
    inflater: LayoutInflater, container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View {
    return inflater.inflate(R.layout.meeting_option, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)

    val meetingOptionsContainer = view.findViewById<View>(R.id.rlMeetingOptions)
    val shape = GradientDrawable()
    shape.setColor(tokens.colors.background.shade900)
    val radius = tokens.borderRadius.getRadius(
      BorderRadiusSize.TWO,
      requireContext().resources.displayMetrics.density
    )
    shape.cornerRadii = floatArrayOf(radius, radius, radius, radius, 0f, 0f, 0f, 0f)
    meetingOptionsContainer.background = shape

    // Participants option
    viewParticipants = view.findViewById(R.id.rlParticipant)
    viewParticipants.setOnClickListener {
      val dyteParticipantsFragment = DyteParticipantsFragment()
      dyteParticipantsFragment.show(
        childFragmentManager,
        DyteMeetingOptionBottomSheet::class.java.simpleName
      )
    }

    // On Poll option clicked
    val pollsView = view.findViewById<RelativeLayout>(R.id.rlPoll)
    viewNewPoll = view.findViewById<RelativeLayout>(R.id.newPoll)
    if (meeting.localUser.permissions.polls.canCreate || meeting.localUser.permissions.polls.canView) {
      pollsView.visible()
      pollsView.setOnClickListener {
        viewNewPoll.gone()
        val dytePollsBottomSheet = DytePollsBottomSheet()
        dytePollsBottomSheet.show(childFragmentManager, DytePollsBottomSheet::class.java.simpleName)
      }
    } else {
      pollsView.gone()
    }

    // Settings option
    viewSettings = view.findViewById(R.id.rlSettings)
    viewSettings.setOnClickListener {
      val settingsFragment = DyteSettingsFragment()
      settingsFragment.lifecycle.addObserver(LifecycleEventObserver { _, event ->
        if (event === Lifecycle.Event.ON_DESTROY) {

        }
      } as LifecycleObserver)
      settingsFragment.show(
        childFragmentManager,
        DyteMeetingOptionBottomSheet::class.java.simpleName
      )
    }

    // Plugins option
    viewPlugins = view.findViewById(R.id.rlPlugins)
    if (meeting.localUser.permissions.plugins.canLaunch) {
      viewPlugins.visible()
      viewPlugins.setOnClickListener {
        val dytePluginsBottomSheet = DytePluginsBottomSheet()
        dytePluginsBottomSheet.show(
          childFragmentManager,
          DyteMeetingOptionBottomSheet::class.java.simpleName
        )
      }
    } else {
      viewPlugins.gone()
    }

    // Recording option
    viewRecording = view.findViewById(R.id.rlRecording)
    tvRecording = view.findViewById(R.id.tvRecording)
    if (meeting.localUser.permissions.host.canTriggerRecording) {
      viewRecording.visible()
      refreshRecordingState()
      meeting.addRecordingEventsListener(meetingEventListener)
    } else {
      viewRecording.gone()
    }

    // On Chat option clicked
    viewChat = view.findViewById<RelativeLayout>(R.id.rlChat)
    viewNewChat = view.findViewById<RelativeLayout>(R.id.newChat)
    viewChat.setOnClickListener {
      viewNewChat.gone()
      val dyteChatBottomSheet = DyteChatBottomSheet()
      dyteChatBottomSheet.show(
        childFragmentManager,
        DyteMeetingOptionBottomSheet::class.java.simpleName
      )
    }
    /*if (DyteUIKitBuilder.dyteUIKit.isNewChatPresent()) {
      viewNewChat.visible()
    } else {
      viewNewChat.gone()
    }*/

    viewMuteAll = view.findViewById(R.id.rlMuteAll)
    if (meeting.localUser.permissions.host.canMuteAudio) {
      viewMuteAll.visible()
      viewMuteAll.setOnClickListener {
        meeting.participants.disableAllAudio()
        dismiss()
      }
    } else {
      viewMuteAll.gone()
    }

    meeting.addChatEventsListener(chatEventsListener)
    meeting.addPollEventsListener(pollEventsListener)

    if (DyteUIKitBuilder.dyteUIKit.isNewChatAvailable) {
      viewNewChat.visible()
    } else {
      viewNewChat.gone()
    }

    if (DyteUIKitBuilder.dyteUIKit.isNewPollAvailable) {
      viewNewPoll.visible()
    } else {
      viewNewPoll.gone()
    }
  }

  private fun refreshRecordingState() {
    when (meeting.recording.recordingState) {
      IDLE -> {
        tvRecording.text = "Start recording"
        viewRecording.setOnClickListener {
          meeting.recording.start()
        }
      }
      STARTING -> {
        tvRecording.text = "Starting"
        viewRecording.setOnClickListener {
          // no-op
        }
      }
      STOPPING -> {
        tvRecording.text = "Stopping"
        viewRecording.setOnClickListener {
          // no-op
        }
      }
      RECORDING -> {
        tvRecording.text = "Stop recording"
        viewRecording.setOnClickListener {
          meeting.recording.stop()
        }
      }
    }
  }

  override fun onDestroyView() {
    super.onDestroyView()
    meeting.removeRecordingEventsListener(meetingEventListener)
    meeting.removeChatEventsListener(chatEventsListener)
  }
}