package dyte.io.uikit.screens.polls

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.CheckBox
import android.widget.LinearLayout
import android.widget.TextView
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView.ViewHolder
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.atoms.DyteCheckBoxAtom
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.ViewUtils.setBackground
import dyte.io.uikit.wiptoken.DyteUITokens
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import kotlinx.coroutines.launch

class PollsListAdapter :
  ListAdapter<DytePollMessage, PollsListAdapter.PollViewHolder>(PollsListDiffCallback()) {

  private val meeting by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }
  override fun onCreateViewHolder(
    parent: ViewGroup,
    viewType: Int,
  ): PollViewHolder {
    val view = LayoutInflater.from(parent.context)
      .inflate(R.layout.list_item_poll, parent, false)
    return PollViewHolder(view)
  }

  override fun onBindViewHolder(
    holder: PollViewHolder,
    position: Int,
  ) {
    holder.bindView(getItem(position))
  }

  inner class PollViewHolder(itemView: View) : ViewHolder(itemView) {
    private val pollCreatorTextView: TextView = itemView.findViewById(R.id.poll_text_view_creator)
    private val pollQuestionTextView: TextView = itemView.findViewById(R.id.poll_text_view_question)
    private val pollOptionsContainer: LinearLayout =
      itemView.findViewById(R.id.poll_options_container)
    private val pollContainer: LinearLayout =
      itemView.findViewById(R.id.llPollContainer)

    fun bindView(dytePollMessage: DytePollMessage) {
      pollCreatorTextView.text = "Poll by ${dytePollMessage.createdBy}"
      pollQuestionTextView.text = dytePollMessage.quesion
      pollQuestionTextView.setTextColor(tokens.colors.text.onBackground.shade600)

      pollContainer.setBackground(tokens.borderRadius, tokens.colors.background.shade1000)

      // Remove the pre-added option views from the PollView being recycled
      if (pollOptionsContainer.childCount != 0) {
        pollOptionsContainer.removeAllViews()
      }

      // Find index of poll option selected by User
      val userSelectedOptionIndex = findUserSelectedOptionIndex(
        meeting.localUser.userId,
        dytePollMessage.options
      )
      val userHasVoted = userSelectedOptionIndex != -1

      dytePollMessage.options.forEachIndexed { optionIndex, dytePollOption ->
        // Prepare the poll option view
        val pollOptionView = LayoutInflater.from(itemView.context)
          .inflate(R.layout.list_item_poll_option, pollOptionsContainer, false)
        pollOptionView.setBackground(tokens.borderRadius, tokens.colors.background.shade900)

        val pollOptionCheckBox = pollOptionView.findViewById<DyteCheckBoxAtom>(R.id.poll_option_checkbox)

        // Set poll option text
        pollOptionView.findViewById<TextView>(R.id.poll_option_text_view).text = dytePollOption.text

        val voteCountTextView =
          pollOptionView.findViewById<TextView>(R.id.poll_option_text_view_vote_count)

        // Set user's vote state in checkbox
        if (userHasVoted) {
          // Check if this option is the selected one. If yes, mark it as checked.
          if (optionIndex == userSelectedOptionIndex) {
            pollOptionCheckBox.isChecked = true
          }
          // Since user has already voted, disable the checkboxes.
          pollOptionCheckBox.isClickable = false
        } else { // If User has not voted, set listeners to trigger vote
          pollOptionCheckBox.setOnCheckedChangeListener { _, isChecked ->
            if (isChecked) {
              try {
                  meeting.polls.vote(dytePollMessage, dytePollOption)
                  // Note: Can check for operation error to revert the checkbox to un-checked state
              } catch (e: Exception) {
                e.printStackTrace()
              }
            }
          }
        }

        // Show result only if
        // - "hide result before voting" is disabled or
        // - "hide result before voting" is enabled but user has already voted
        val shouldShowResult = !dytePollMessage.hideVotes || userHasVoted
        if (shouldShowResult) {
          voteCountTextView.text = "(" + dytePollOption.count + ")"
        } else {
          voteCountTextView.text = ""
        }

        pollOptionCheckBox.isEnabled = meeting.localUser.permissions.polls.canVote
        pollOptionsContainer.addView(pollOptionView)
      }
    }

    private fun findUserSelectedOptionIndex(
      userId: String,
      pollOptions: List<DytePollOption>,
    ): Int {
      pollOptions.forEachIndexed { index, dytePollOption ->
        dytePollOption.votes.forEach { vote ->
          if (vote.id == userId) return index
        }
      }
      return -1
    }
  }
}