package dyte.io.uikit.utils

import android.annotation.SuppressLint
import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.Rect
import android.text.style.LineHeightSpan.WithDensity
import android.util.Log
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import androidx.core.content.ContextCompat
import androidx.fragment.app.DialogFragment
import com.google.android.material.snackbar.Snackbar
import dyte.io.uikit.R
import java.text.DateFormat
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.TimeZone
import kotlin.math.abs

object Utils {

  val Int.px: Int
    get() = (this * Resources.getSystem().displayMetrics.density).toInt()

  fun getInitialsFromName(name: String): String {
    val trimmedName = name.trim()

    if (trimmedName.isBlank()) {
      return "NN"
    }

    if (trimmedName.length < 2) {
      return trimmedName.first().toString().uppercase(Locale.getDefault())
    }

    val subNames = trimmedName.split(" ")
    return if (subNames.size >= 2) {
      "${subNames.first().first()}${subNames.last().first()}".uppercase(Locale.getDefault())
    } else {
      subNames.first().substring(0, 2).uppercase(Locale.getDefault())
    }
  }

  // "YYYY-MM-DDThh:mm:ss.sTZD"
  @SuppressLint("SimpleDateFormat")
  fun getUtcTimeNow(): String {
    val m_ISO8601Local: DateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    m_ISO8601Local.timeZone = TimeZone.getTimeZone("UTC")
    return m_ISO8601Local.format(Date())
  }

  // "yyyy-MM-dd'T'HH:mm:ss'Z'"
  @SuppressLint("SimpleDateFormat")
  fun getDiff(startTime: String, endTime: String): String {
    val dates = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    val date1 = requireNotNull(dates.parse(startTime))
    val date2 = requireNotNull(dates.parse(endTime))
    val differenceInMillis = abs(date1.time - date2.time).toInt()

    val seconds: Int = differenceInMillis / 1000 % 60
    val minutes: Int = (differenceInMillis - seconds) / 1000 / 60

    var newSeconds = seconds.toString()
    if (seconds < 10) {
      newSeconds = "0$newSeconds"
    }

    var newMinutes = minutes.toString()
    var newHours = ""

    if (minutes < 10) {
      newMinutes = "0$newMinutes"
    } else if(minutes >= 60) {
      val hourCount = minutes/60
      newHours = if(hourCount < 10) {
        "0${hourCount}"
      } else {
        return "$hourCount"
      }

      var newMinutesAfterHours = minutes-(hourCount*60)
      newMinutes = if(newMinutesAfterHours<10) {
        "0$newMinutesAfterHours"
      } else {
        "$newMinutesAfterHours"
      }
    }

    var time = "$newMinutes:$newSeconds"
    if (newHours.isNotEmpty()) {
      time = "$newHours:$time"
    }
    return time
  }

  /**
   * Call this method (in onActivityCreated or later) to set
   * the width of the dialog to a percentage of the current
   * screen width.
   */
  fun DialogFragment.setWidthPercent(percentage: Int) {
    val percent = percentage.toFloat() / 100
    val dm = Resources.getSystem().displayMetrics
    val rect = dm.run { Rect(0, 0, widthPixels, heightPixels) }
    val percentWidth = rect.width() * percent
    dialog?.window?.setLayout(percentWidth.toInt(), ViewGroup.LayoutParams.WRAP_CONTENT)
  }

  fun Context.showToast(view: View, message: String) {
    val snackbar = Snackbar.make(view, message, Snackbar.LENGTH_SHORT)
    snackbar.view.setBackgroundColor(ContextCompat.getColor(this, R.color.defaultBgDark))
    val tv = snackbar.view.findViewById<TextView>(com.google.android.material.R.id.snackbar_text)
    tv.setTextColor(Color.WHITE)
    snackbar.show()
  }
}

/*
To be discussed later:
https://github.com/android/android-ktx/issues/132
https://code.luasoftware.com/tutorials/android/android-convert-dp-to-pixel/

Although trying with both
Resources.getSystem().displayMetrics.density &
applicationContext.resources.displayMetrics.density
gave same displayDensity
 */
internal fun Float.dpToPx(displayDensity: Float = 1f): Float {
  return this * displayDensity
}

internal val View.displayDensity: Float
  get() = context.resources.displayMetrics.density