package dyte.io.uikit.view

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.drawable.Drawable
import android.graphics.drawable.GradientDrawable
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.View
import android.widget.Space
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.appcompat.widget.LinearLayoutCompat
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import dyte.io.uikit.R
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.displayDensity
import dyte.io.uikit.utils.dpToPx
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import dyte.io.uikit.wiptoken.DyteUITokens

open class DyteControlBarButton : LinearLayoutCompat {
  enum class Variant {
    BUTTON,
    HORIZONTAL
  }

  protected lateinit var iconImageView: AppCompatImageView
  protected lateinit var labelTextView: AppCompatTextView
  protected lateinit var iconSpace: Space

  private var iconSpacing: Float = 0f

  protected var variant: Variant = Variant.BUTTON

  protected var showLabel: Boolean = true

  protected open val defaultLabelResId: Int = ResourcesCompat.ID_NULL
  protected open val defaultIconResId: Int = ResourcesCompat.ID_NULL

  constructor(context: Context) : super(context) {
    init()
  }

  constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
    init(attrs)
  }

  constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
    context,
    attrs,
    defStyleAttr
  ) {
    init(attrs)
  }

  private fun init(attrs: AttributeSet? = null) {
    inflate(context, R.layout.view_dyte_controlbar_button, this)
    iconImageView = findViewById(R.id.imageview_dyte_controlbar_button_icon)
    labelTextView = findViewById(R.id.textview_dyte_controlbar_button_label)
    iconSpace = findViewById(R.id.space_dyte_controlbar_button_icon)

    val attributes = context.obtainStyledAttributes(
      attrs,
      R.styleable.DyteControlBarButton,
      0,
      0
    )

    val iconDrawable = attributes.getDrawable(R.styleable.DyteControlBarButton_dyte_cbb_icon)
      ?: try {
        ContextCompat.getDrawable(context, defaultIconResId)
      } catch (e: Exception) {
        null
      }
    val labelText = attributes.getString(R.styleable.DyteControlBarButton_dyte_cbb_label)
      ?: try {
        resources.getString(defaultLabelResId)
      } catch (e: Exception) {
        null
      }
    variant = attributes.getInt(R.styleable.DyteControlBarButton_dyte_cbb_variant, 0).toVariant()
    showLabel = attributes.getBoolean(R.styleable.DyteControlBarButton_dyte_cbb_showLabel, true)
    attributes.recycle()

    if (showLabel) {
      showLabel()
    } else {
      hideLabel()
    }

    applyVariant()
    applyIconSpacing()
    applyUiTokens() // Called only for design preview
    setIconDrawable(iconDrawable)
    setLabel(labelText)
  }

  private fun showLabel() {
    labelTextView.visibility = View.VISIBLE
    iconSpace.visibility = View.VISIBLE
  }

  private fun hideLabel() {
    labelTextView.visibility = View.GONE
    iconSpace.visibility = View.GONE
  }

  private fun applyVariant() {
    when (variant) {
      Variant.BUTTON -> {
        orientation = VERTICAL
        gravity = Gravity.CENTER
        iconSpacing = DEFAULT_ICON_SPACING_DP_BUTTON.dpToPx(displayDensity)
      }
      Variant.HORIZONTAL -> {
        orientation = HORIZONTAL
        gravity = Gravity.CENTER_VERTICAL
        iconSpacing = DEFAULT_ICON_SPACING_DP_HORIZONTAL.dpToPx(displayDensity)
      }
    }
  }

  private fun applyIconSpacing() {
    when (variant) {
      Variant.BUTTON -> {
        iconSpace.layoutParams = LinearLayoutCompat.LayoutParams(0, iconSpacing.toInt())
      }
      Variant.HORIZONTAL -> {
        iconSpace.layoutParams = LinearLayoutCompat.LayoutParams(iconSpacing.toInt(), 0)
      }
    }
  }

  fun applyUiTokens(uiTokens: DyteUITokens = tokens) {
    background = createBackgroundDrawable(uiTokens)
    iconImageView.imageTintList = ColorStateList.valueOf(uiTokens.colors.text.onBackground.shade1000)
    labelTextView.setTextColor(uiTokens.colors.text.onBackground.shade1000)
  }

  protected open fun createBackgroundDrawable(uiTokens: DyteUITokens): GradientDrawable {
    val backgroundDrawable = GradientDrawable().apply {
      shape = GradientDrawable.RECTANGLE
      setColor(uiTokens.colors.background.shade900)
      cornerRadius = uiTokens.borderRadius.getRadius(BorderRadiusSize.ONE, displayDensity)
    }
    return backgroundDrawable
  }

  fun setIconDrawable(drawable: Drawable?) {
    iconImageView.setImageDrawable(drawable)
  }

  fun setIconTint(color: Int) {
    iconImageView.imageTintList = ColorStateList.valueOf(color)
  }

  fun setLabel(label: String?) {
    labelTextView.text = label
  }

  companion object {
    private const val DEFAULT_ICON_SPACING_DP_BUTTON = 2f
    private const val DEFAULT_ICON_SPACING_DP_HORIZONTAL = 12f

    private fun Int.toVariant(): DyteControlBarButton.Variant {
      return when (this) {
        1 -> Variant.HORIZONTAL
        else -> Variant.BUTTON
      }
    }
  }
}