package dyte.io.uikit.molecules.dytegrid

import android.annotation.SuppressLint
import android.content.Context
import android.util.AttributeSet
import android.widget.LinearLayout
import androidx.viewpager2.widget.ViewPager2
import com.google.android.material.tabs.TabLayout
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.molecules.dytegrid.GridPage.GridAutoPage
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DytePlugin
import io.dyte.core.listeners.DytePluginEventsListener

class DyteGridView : LinearLayout {

  private lateinit var vpPages: ViewPager2
  private lateinit var tabLayout: TabLayout

  private lateinit var adapter: GridPagesAdapter

  private val meeting by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }

  private val pageHelper = GridHelper()

  private val pluginEventsListener = object : DytePluginEventsListener {
    override fun onPluginFileRequest(plugin: DytePlugin) {
      // TODO : what do we do here
    }

    override fun onPluginActivated(plugin: DytePlugin) {
      super.onPluginActivated(plugin)
      refreshGrid()
    }

    override fun onPluginDeactivated(plugin: DytePlugin) {
      super.onPluginDeactivated(plugin)
      refreshGrid()
    }
  }

  private val participantEventsListener = object : DyteParticipantEventsListener {
    override fun onScreenSharesUpdated() {
      super.onScreenSharesUpdated()
      refreshGrid(true)
      refreshTabIndicators()
    }

    override fun onParticipantPinned(participant: DyteMeetingParticipant) {
      super.onParticipantPinned(participant)
      refreshActiveAndPinnedParticipant()
    }

    override fun onParticipantUnpinned() {
      super.onParticipantUnpinned()
      refreshActiveAndPinnedParticipant()
    }

    override fun onActiveParticipantsChanged(active: List<DyteMeetingParticipant>) {
      super.onActiveParticipantsChanged(active)
      refreshGrid()
    }
  }

  private fun refreshActiveAndPinnedParticipant() {
    println("DyteMobileClient | GridView refreshActiveAndPinnedParticipant ")
    adapter.notifyItemChanged(0)
  }

  constructor(context: Context) : super(context) {
    init(context)
  }

  constructor(
    context: Context,
    attrs: AttributeSet?
  ) : super(context, attrs) {
    init(context)
  }

  constructor(
    context: Context,
    attrs: AttributeSet?,
    defStyleAttr: Int
  ) : super(
    context,
    attrs,
    defStyleAttr
  ) {
    init(context)
  }

  private fun refreshTabIndicators() {
    val isScreenShared = meeting.participants.screenshares.isNotEmpty()
    val isPluginActive = meeting.plugins.active.isNotEmpty()

    println("DyteMobileClient | DyteGridView refreshTabIndicators pageCount: ${meeting.participants.pageCount} isScreenShared: $isScreenShared isPluginActive: $isPluginActive")

    if (meeting.participants.pageCount == 1 && (isScreenShared || isPluginActive).not()) {
      tabLayout.gone()
      return
    }

    tabLayout.removeAllTabs()
    tabLayout.visible()

    var pageCount = meeting.participants.pageCount
    if(isScreenShared || isPluginActive) {
      pageCount++
    }

    for (i in 0 until pageCount) {
      val tab = tabLayout.newTab()
      tabLayout.addTab(tab)
      if (i == vpPages.currentItem) {
        tab.select()
      }
    }
  }

  @SuppressLint("ClickableViewAccessibility")
  private fun init(context: Context) {
    inflate(context, R.layout.view_grid, this)

    vpPages = findViewById(R.id.rvPeers)
    tabLayout = findViewById(R.id.tab_layout)

    vpPages.registerOnPageChangeCallback(object : ViewPager2.OnPageChangeCallback() {
      override fun onPageSelected(position: Int) {
        println("DyteMobileClient | GridView onPageSelected $position")
        super.onPageSelected(position)
        refreshTabIndicators()
        adapter.notifyItemChanged(position)
      }
    })

    refreshTabIndicators()
  }

  fun refresh() {
    println("DyteMobileClient | GridView refresh ")
    refreshGrid()
    refreshTabIndicators()

    meeting.addParticipantEventsListener(participantEventsListener)
    meeting.addPluginEventsListener(pluginEventsListener)
  }

  private fun refreshGrid(newAdapter: Boolean = false) {
    println("DyteMobileClient | GridView refreshGrid $newAdapter")

    if (vpPages.adapter == null || newAdapter) {
      adapter = GridPagesAdapter()
      vpPages.adapter = adapter
    }
    adapter.submitList(pageHelper.getPages())
  }

  fun refreshSelf() {
    println("DyteMobileClient | GridView refreshSelf ")
    adapter.notifyDataSetChanged()
  }
}
