package dyte.io.uikit.screens

import android.os.Bundle
import android.view.View
import android.widget.FrameLayout
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import dyte.io.uikit.R.id
import dyte.io.uikit.R.layout
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Error
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.GroupCall
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Left
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.LiveStream
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Loading
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.MeetingNotification
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Removed
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Setup
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.WaitingRoom
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.WaitingRoomEntryRejected
import dyte.io.uikit.screens.livestream.DyteLiveStreamFragment
import dyte.io.uikit.screens.meeting.DyteGroupCallFragment
import dyte.io.uikit.screens.setup.DyteSetupFragment
import dyte.io.uikit.utils.Utils.showToast
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.view.FullScreenErrorView
import dyte.io.uikit.view.FullScreenLoaderView

class DyteMeetingActivity : AppCompatActivity() {
  private lateinit var clLoader: FullScreenLoaderView
  private lateinit var clError: FullScreenErrorView
  private lateinit var flFragmentContainer: FrameLayout

  private lateinit var clWaitListContainer: View
  private lateinit var clWaitingContainer: View
  private lateinit var clWaitingRejectedContainer: View
  private lateinit var btnExit: View

  private lateinit var viewModel: DyteMeetingViewModel

  private val fragmentManager by lazy {
    supportFragmentManager
  }

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)
    setContentView(layout.activity_meeting_room)

    clLoader = findViewById(id.clLoader)
    clError = findViewById(id.clError)
    flFragmentContainer = findViewById(id.fragmentContainer)

    clWaitListContainer = findViewById(id.clWaitListContainer)
    clWaitingContainer = findViewById(id.clWaitingContainer)
    clWaitingRejectedContainer = findViewById(id.clRejectedContainer)
    btnExit = findViewById(id.btnExit)

    viewModel = ViewModelProvider(this)[DyteMeetingViewModel::class.java]
    viewModel.stateLiveData.observe(this) { meetingState ->
      when (meetingState) {
        is Error -> {
          showError(meetingState.exception)
        }
        GroupCall -> {
          showMeetingsFragment()
        }
        LiveStream -> {
          showLiveStreamFragment()
        }
        Loading -> {
          showLoading()
        }
        Setup -> {
          showSetupScreen()
        }
        WaitingRoom -> {
          showWaiting()
        }
        WaitingRoomEntryRejected -> {
          showWaitListRejected()
        }
        Removed -> {
          finish()
        }
        Left -> {
          finish()
        }
        is MeetingNotification -> {
          showToast(flFragmentContainer, meetingState.notificationText)
        }
      }
    }
    viewModel.start()
  }

  private fun showSetupScreen() {
    clLoader.gone()
    fragmentManager.beginTransaction()
      .add(id.fragmentContainer, DyteSetupFragment(), DyteMeetingActivity::class.java.simpleName)
      .commit()
    flFragmentContainer.visible()
  }

  private fun showMeetingsFragment() {
    println("DyteMobileClient | DyteMeetingActivity showMeetingsFragment ")
    clLoader.gone()
    fragmentManager.beginTransaction()
      .replace(
        id.fragmentContainer,
        DyteGroupCallFragment(),
        DyteMeetingActivity::class.java.simpleName
      )
      .commitNow()
    clWaitListContainer.gone()
    flFragmentContainer.visible()
  }

  private fun showLiveStreamFragment() {
    println("DyteMobileClient | DyteMeetingActivity showLiveStreamFragment ")
    clLoader.gone()
    fragmentManager.beginTransaction()
      .add(
        id.fragmentContainer,
        DyteLiveStreamFragment(),
        DyteMeetingActivity::class.java.simpleName
      )
      .commit()
    clWaitListContainer.gone()
    flFragmentContainer.visible()
  }

  private fun showError(e: Exception) {
    flFragmentContainer.gone()
    clLoader.gone()
    clError.visible()
    clError.refresh(e) {
      println("DyteMobileClient | DyteMeetingActivity showError doing init")
      finish()
    }
  }

  private fun showLoading() {
    flFragmentContainer.gone()
    clError.gone()
    clLoader.visible()
  }

  private fun showWaiting() {
    clLoader.gone()
    clError.gone()
    flFragmentContainer.gone()

    clWaitListContainer.visible()
    clWaitingContainer.visible()
    clWaitingRejectedContainer.gone()
  }

  private fun showWaitListRejected() {
    clLoader.gone()
    clError.gone()
    flFragmentContainer.gone()

    clWaitListContainer.visible()
    clWaitingContainer.gone()
    clWaitingRejectedContainer.visible()
    btnExit.setOnClickListener {
      finish()
    }
  }
}

