package dyte.io.uikit.screens

import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Error
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.GroupCall
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Left
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.LiveStream
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Loading
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.MeetingNotification
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Removed
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.Setup
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.WaitingRoom
import dyte.io.uikit.screens.DyteMeetingViewModel.DyteMeetingState.WaitingRoomEntryRejected
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DyteMessageType.FILE
import io.dyte.core.feat.DyteMessageType.IMAGE
import io.dyte.core.feat.DyteMessageType.TEXT
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.WaitListStatus
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.NONE
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.models.WaitListStatus.WAITING

class DyteMeetingViewModel : ViewModel() {
  sealed class DyteMeetingState {
    object Loading : DyteMeetingState()
    object Setup : DyteMeetingState()
    object GroupCall : DyteMeetingState()
    object LiveStream : DyteMeetingState()
    class Error(val errorMessage: String, val exception: Exception) : DyteMeetingState()
    object WaitingRoom : DyteMeetingState()
    object WaitingRoomEntryRejected : DyteMeetingState()
    object Removed : DyteMeetingState()
    object Left : DyteMeetingState()
    class MeetingNotification(val notificationText: String) : DyteMeetingState()
  }

  val stateLiveData = MutableLiveData<DyteMeetingState>()

  private val uiKit = DyteUIKitBuilder.dyteUIKit
  private val meeting by lazy {
    uiKit.meeting
  }

  private val dyteMeetingEventsListener = object : DyteMeetingRoomEventsListener {

    override fun onMeetingInitCompleted() {
      super.onMeetingInitCompleted()
      if (meeting.localUser.shouldShowSetupScreen()) {
        stateLiveData.value = Setup
      } else {
        meeting.joinRoom()
      }
    }

    override fun onMeetingInitFailed(exception: Exception) {
      super.onMeetingInitFailed(exception)
      stateLiveData.value = DyteMeetingState.Error(
        exception.localizedMessage ?: "Something went wrong, Please try again later", exception
      )
    }

    override fun onMeetingInitStarted() {
      super.onMeetingInitStarted()
      stateLiveData.value = Loading
    }

    override fun onMeetingRoomJoinStarted() {
      super.onMeetingRoomJoinStarted()
      stateLiveData.value = Loading
    }

    override fun onMeetingRoomJoinCompleted() {
      super.onMeetingRoomJoinCompleted()
      if (meeting.meta.roomType == "GROUP_CALL") {
        println("DyteMobileClient | DyteMeetingViewModel onMeetingRoomJoined group call")
        stateLiveData.value = GroupCall
      } else {
        println("DyteMobileClient | DyteMeetingViewModel onMeetingRoomJoined live stream")
        stateLiveData.value = LiveStream
      }
    }

    override fun onMeetingRoomJoinFailed(exception: Exception) {
      super.onMeetingRoomJoinFailed(exception)
      stateLiveData.value = DyteMeetingState.Error(
        exception.localizedMessage ?: "Something went wrong, Please try again later", exception
      )
    }

    override fun onMeetingRoomLeaveCompleted() {
      super.onMeetingRoomLeaveCompleted()
      stateLiveData.value = Left
    }
  }

  private val chatEventsListener = object : DyteChatEventsListener {
    override fun onNewChatMessage(message: DyteChatMessage) {
      super.onNewChatMessage(message)
      val notificationText = when (message.type) {
        TEXT -> {
          "${message.displayName}: ${(message as DyteTextMessage).message}"
        }
        IMAGE -> {
          "${message.displayName}: Shared new image"
        }
        FILE -> {
          "${message.displayName}: Shared new file"
        }
      }
      stateLiveData.value = MeetingNotification(notificationText)
    }
  }

  private val dyteLocalUserEventsListener = object : DyteSelfEventsListener {


    override fun onRemovedFromMeeting() {
      super.onRemovedFromMeeting()
      stateLiveData.value = Removed
    }

    override fun onWaitListStatusUpdate(waitListStatus: WaitListStatus) {
      super.onWaitListStatusUpdate(waitListStatus)
      when (waitListStatus) {
        NONE, ACCEPTED -> {
          if (meeting.meta.roomType == "GROUP_CALL") {
            println("DyteMobileClient | DyteMeetingViewModel onWaitListStatusUpdate group call")
            stateLiveData.value = GroupCall
          } else {
            println("DyteMobileClient | DyteMeetingViewModel onWaitListStatusUpdate live stream")
            stateLiveData.value = LiveStream
          }
        }
        WAITING -> {
          stateLiveData.value = WaitingRoom
        }
        REJECTED -> {
          stateLiveData.value = WaitingRoomEntryRejected
        }
      }
    }

    /*override fun onMeetingRoomDisconnected() {
      super.onMeetingRoomDisconnected()
      stateLiveData.value = Error(
        "You are disconnected from the meeting",
        Exception("You are disconnected from the meeting")
      )
    }*/
  }

  private val dyteParticipantEventsListener = object : DyteParticipantEventsListener {
    override fun onParticipantJoin(participant: DyteMeetingParticipant) {
      super.onParticipantJoin(participant)
      if (participant.id != meeting.localUser.id) {
        stateLiveData.value = MeetingNotification("${participant.name} just joined")
      }
    }

    override fun onParticipantLeave(participant: DyteMeetingParticipant) {
      super.onParticipantLeave(participant)
      if (participant.id != meeting.localUser.id) {
        stateLiveData.value = MeetingNotification("${participant.name} left")
      }
    }
  }

  fun start() {
    meeting.addMeetingRoomEventsListener(dyteMeetingEventsListener)
    meeting.addSelfEventsListener(dyteLocalUserEventsListener)
    meeting.addParticipantEventsListener(dyteParticipantEventsListener)
    meeting.addChatEventsListener(chatEventsListener)
    uiKit.meetingInfo?.let {
      uiKit.meeting.init(it)
    }
    uiKit.meetingInfoV2?.let {
      uiKit.meeting.init(it)
    }
  }
}