package dyte.io.uikit.screens.participant

import android.content.res.ColorStateList
import android.graphics.drawable.GradientDrawable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.EditText
import android.widget.ImageView
import android.widget.TextView
import androidx.core.widget.doOnTextChanged
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.atoms.DyteParticipantAudioIndicator
import dyte.io.uikit.atoms.DyteParticipantVideoIndicator
import dyte.io.uikit.screens.participant.DyteParticipantsAdapter.BaseViewHolder
import dyte.io.uikit.screens.participant.UiParticipantPlank.AcceptAll
import dyte.io.uikit.screens.participant.UiParticipantPlank.Joined
import dyte.io.uikit.screens.participant.UiParticipantPlank.Lable
import dyte.io.uikit.screens.participant.UiParticipantPlank.SearchBox
import dyte.io.uikit.screens.participant.UiParticipantPlank.WaitListed
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.Utils
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import io.dyte.core.feat.DyteMeetingParticipant
import kotlinx.coroutines.launch

class DyteParticipantsAdapter(
  private val onHostControlsClicked: (participant: DyteMeetingParticipant) -> Unit,
  private val onSearchTextChanged: (searchText: String) -> Unit
) :
  ListAdapter<UiParticipantPlank, BaseViewHolder>(ParticipantListDiffCallback) {

  override fun getItemViewType(position: Int): Int {
    return getItem(position).viewType
  }

  override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder {
    when (viewType) {
      0 -> {
        // label view holder
        return LableViewHolder(
          LayoutInflater.from(parent.context)
            .inflate(R.layout.participant_lable_item, parent, false)
        )
      }

      1 -> {
        // waitlisted participant
        return WaitListedParticipantViewHolder(
          LayoutInflater.from(parent.context)
            .inflate(R.layout.participant_waitlisted_item, parent, false)
        )
      }

      2 -> {
        // accept all button
        return AcceptAllViewHolder(
          LayoutInflater.from(parent.context)
            .inflate(R.layout.plank_participant_accept_all, parent, false)
        )
      }

      3 -> {
        // joined participant
        return JoinedParticipantViewHolder(
          LayoutInflater.from(parent.context).inflate(R.layout.participant_item, parent, false)
        )
      }

      4 -> {
        return SearchBoxViewHolder(
          LayoutInflater.from(parent.context)
            .inflate(R.layout.participant_search_box, parent, false)
        )
      }
    }
    throw IllegalArgumentException("PlankType $viewType not supported")
  }

  override fun onBindViewHolder(holder: BaseViewHolder, position: Int) {
    holder.bind(getItem(position))
  }

  inner class SearchBoxViewHolder(view: View) :
    BaseViewHolder(view) {
    private var etSearchBox: EditText = view.findViewById(R.id.etSearchBox)

    override fun bind(plank: UiParticipantPlank) {
      plank as SearchBox

      val shape = GradientDrawable()
      // Set color
      shape.setColor(tokens.colors.background.shade900)
      // Set corner radius
      shape.cornerRadius = tokens.borderRadius.getRadius(
        BorderRadiusSize.ONE,
        itemView.context.resources.displayMetrics.density
      )
      etSearchBox.background = shape
      etSearchBox.setTextColor(tokens.colors.text.onBackground.shade1000)
      etSearchBox.setHintTextColor(tokens.colors.text.onBackground.shade800)

      etSearchBox.doOnTextChanged { text, _, _, _ ->
        onSearchTextChanged.invoke(text.toString())
      }
    }
  }

  inner class JoinedParticipantViewHolder(view: View) :
    BaseViewHolder(view) {
    private var tvName: TextView = view.findViewById(R.id.tvParticipantName)
    private var ivPhoto: TextView = view.findViewById(R.id.tvInitials)
    private var ivMic: DyteParticipantAudioIndicator = view.findViewById(R.id.ivMic)
    private var ivCamera: DyteParticipantVideoIndicator = view.findViewById(R.id.ivCameraIndicator)
    private var divider: View = view.findViewById(R.id.divider)
    private var ivParticipantHostControls: ImageView =
      view.findViewById(R.id.ivParticipantHostControls)

    private val meeting by lazy {
      DyteUIKitBuilder.dyteUIKit.meeting
    }

    override fun bind(plank: UiParticipantPlank) {
      plank as Joined
      if (meeting.localUser.canDoParticipantHostControls()) {
        ivParticipantHostControls.visible()
        ivParticipantHostControls.setOnClickListener {
          onHostControlsClicked.invoke(plank.participant.dyteMeetingParticipant)
        }
        itemView.setOnClickListener {
          onHostControlsClicked.invoke(plank.participant.dyteMeetingParticipant)
        }
      } else {
        ivParticipantHostControls.gone()
      }

      ivMic.activate(plank.participant.dyteMeetingParticipant)
      ivCamera.activate(plank.participant.dyteMeetingParticipant)

      tvName.text = plank.participant.name
      ivPhoto.text = Utils.getInitialsFromName(plank.participant.name)

      ivCamera.imageTintList = ColorStateList.valueOf(tokens.colors.text.onBackground.shade1000)
      ivParticipantHostControls.imageTintList =
        ColorStateList.valueOf(tokens.colors.text.onBackground.shade1000)
      tvName.setTextColor(tokens.colors.text.onBackground.shade1000)

      if (plank.participant.isLast) {
        divider.gone()
      } else {
        divider.visible()
      }
    }
  }

  inner class WaitListedParticipantViewHolder(view: View) :
    BaseViewHolder(view) {
    private var tvName: TextView = view.findViewById(R.id.tvParticipantName)
    private var ivPhoto: TextView = view.findViewById(R.id.tvInitials)
    private var ivAccept: ImageView = view.findViewById(R.id.ivAccept)
    private var ivReject: ImageView = view.findViewById(R.id.ivReject)
    private var divider: View = view.findViewById(R.id.view1)

    override fun bind(plank: UiParticipantPlank) {
      plank as UiParticipantPlank.WaitListed
      ivAccept.setOnClickListener {
        plank.participant.dyteMeetingParticipant.acceptWaitListedRequest()
      }
      ivReject.setOnClickListener {
        plank.participant.dyteMeetingParticipant.rejectWaitListedRequest()
      }

      val shape = GradientDrawable()
      // Set color
      shape.setColor(tokens.colors.background.shade800)
      // Set corner radius
      shape.cornerRadius = tokens.borderRadius.getRadius(
        BorderRadiusSize.ONE,
        itemView.context.resources.displayMetrics.density
      )

      ivAccept.background = shape
      ivReject.background = shape

      ivAccept.imageTintList = ColorStateList.valueOf(tokens.colors.status.success)
      ivReject.imageTintList = ColorStateList.valueOf(tokens.colors.status.danger)

      tvName.text = plank.participant.name
      ivPhoto.text = Utils.getInitialsFromName(plank.participant.name)

      if (plank.participant.isLast) {
        divider.gone()
      } else {
        divider.visible()
      }
    }
  }

  inner class LableViewHolder(view: View) :
    BaseViewHolder(view) {
    private var tvLable: TextView = view.findViewById(R.id.tvLable)

    override fun bind(plank: UiParticipantPlank) {
      plank as UiParticipantPlank.Lable
      tvLable.text = plank.label
    }
  }

  inner class AcceptAllViewHolder(view: View) :
    BaseViewHolder(view) {
    private var btnAcceptAll: Button = view.findViewById(R.id.btnAcceptAll)

    private val meeting by lazy {
      DyteUIKitBuilder.dyteUIKit.meeting
    }

    override fun bind(plank: UiParticipantPlank) {
      plank as UiParticipantPlank.AcceptAll

      val shape = GradientDrawable()
      // Set color
      shape.setColor(tokens.colors.background.shade800)
      // Set corner radius
      shape.cornerRadius = tokens.borderRadius.getRadius(
        BorderRadiusSize.ONE,
        itemView.context.resources.displayMetrics.density
      )
      btnAcceptAll.background = shape
      btnAcceptAll.setTextColor(tokens.colors.status.success)

      btnAcceptAll.setOnClickListener {
        meeting.participants.acceptAllWaitingRequests()
      }
    }
  }

  abstract inner class BaseViewHolder(view: View) : RecyclerView.ViewHolder(view) {
    abstract fun bind(plank: UiParticipantPlank)
  }

  companion object {
    private val ParticipantListDiffCallback =
      object : DiffUtil.ItemCallback<UiParticipantPlank>() {
        override fun areItemsTheSame(
          oldItem: UiParticipantPlank,
          newItem: UiParticipantPlank
        ): Boolean {
          var areItemsTheSame = false
          if (oldItem.viewType == newItem.viewType) {
            when (oldItem) {
              AcceptAll -> {
                areItemsTheSame = true
              }
              is Joined -> {
                val newParticipant = newItem as Joined
                areItemsTheSame = newParticipant.participant.id == oldItem.participant.id
              }
              is Lable -> {
                newItem as Lable
                areItemsTheSame = newItem.label == oldItem.label
              }
              is WaitListed -> {
                val newParticipant = newItem as WaitListed
                areItemsTheSame = newParticipant.participant.id == oldItem.participant.id
              }
              SearchBox -> {
                areItemsTheSame = true
              }
            }
          }
          return areItemsTheSame
        }

        override fun areContentsTheSame(
          oldItem: UiParticipantPlank,
          newItem: UiParticipantPlank
        ): Boolean {
          var areContentsTheSame = false
          if (oldItem.viewType == newItem.viewType) {
            when (oldItem) {
              AcceptAll -> {
                areContentsTheSame = true
              }
              is Joined -> {
                val newParticipant = newItem as Joined
                areContentsTheSame =
                  newParticipant.participant.audioEnabled == oldItem.participant.audioEnabled
                    && newParticipant.participant.videoEnabled == oldItem.participant.videoEnabled
              }
              is Lable -> {
                newItem as Lable
                areContentsTheSame = newItem.label == oldItem.label
              }
              is WaitListed -> {
                val newParticipant = newItem as WaitListed
                areContentsTheSame =
                  newParticipant.participant.audioEnabled == oldItem.participant.audioEnabled
                    && newParticipant.participant.videoEnabled == oldItem.participant.videoEnabled
              }
              SearchBox -> {
                areContentsTheSame = true
              }
            }
          }
          return areContentsTheSame
        }
      }
  }
}




