package dyte.io.uikit.screens.participant

import android.app.Dialog
import android.content.res.ColorStateList
import android.graphics.drawable.GradientDrawable
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.DialogFragment
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.R.layout
import dyte.io.uikit.tokens
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.models.DyteRoomParticipants

class DyteParticipantsFragment : BottomSheetDialogFragment() {

  private lateinit var rvParticipant: RecyclerView
  private lateinit var dyteParticipantsAdapter: DyteParticipantsAdapter

  private lateinit var rlParticipantContainer: ConstraintLayout

  private lateinit var dismiss: ImageView

  private val meeting by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }

  override fun onCreate(savedInstanceState: Bundle?) {
    setStyle(DialogFragment.STYLE_NORMAL, R.style.AppBottomSheetDialogTheme)
    super.onCreate(savedInstanceState)
  }

  override fun onCreateView(
    inflater: LayoutInflater, container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View {
    return inflater.inflate(layout.fragment_participant_list, container, false)
  }

  override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
    val dialog = super.onCreateDialog(savedInstanceState)

    dialog.setOnShowListener {
      val bottomSheet: FrameLayout = (dialog as BottomSheetDialog)
        .findViewById(com.google.android.material.R.id.design_bottom_sheet)
        ?: return@setOnShowListener

      val behavior = BottomSheetBehavior.from(bottomSheet)
      behavior.state = BottomSheetBehavior.STATE_EXPANDED
      behavior.skipCollapsed = true
    }
    return dialog
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)

    rlParticipantContainer = view.findViewById(R.id.rlParticipantContainer)

    val shape = GradientDrawable()
    shape.setColor(tokens.colors.background.shade900)
    val radius = tokens.borderRadius.getRadius(BorderRadiusSize.TWO, requireContext().resources.displayMetrics.density)
    shape.cornerRadii = floatArrayOf(radius, radius, radius, radius, 0f, 0f, 0f, 0f)
    rlParticipantContainer.background = shape

    dismiss = view.findViewById(R.id.header_button_close)
    dismiss.setOnClickListener { dismiss() }

    rvParticipant = view.findViewById(R.id.rvParticipantView)
    dyteParticipantsAdapter = DyteParticipantsAdapter({ dyteMeetingParticipant ->
      val hostControlsBottomsheet = ParticipantHostControlsBottomsheet(dyteMeetingParticipant)
      hostControlsBottomsheet.show(childFragmentManager, DyteParticipantsFragment::class.java.simpleName)
    }, { searchText ->
      refreshList(searchText)
    })
    rvParticipant.adapter = dyteParticipantsAdapter
    rvParticipant.layoutManager = LinearLayoutManager(requireContext())

    view.findViewById<TextView>(R.id.header_text_view_title).text = "Participants"

    refreshList()

    meeting.addParticipantEventsListener(object : DyteParticipantEventsListener {
      override fun onUpdate(participants: DyteRoomParticipants) {
        super.onUpdate(participants)
        refreshList()
      }
    })
  }

  private fun refreshList(searchText: String = "") {
    val participantPlanks = arrayListOf<UiParticipantPlank>()

    val waitListedParticipants = meeting.participants.waitlisted
    if (waitListedParticipants.isNotEmpty()) {
      // add waitlisted title
      participantPlanks.add(UiParticipantPlank.Lable("Waiting (${waitListedParticipants.size})"))
      // add waitlisted participants in list
      waitListedParticipants.forEachIndexed { index, dyteMeetingParticipant ->
        participantPlanks.add(UiParticipantPlank.WaitListed(UiParticipant(dyteMeetingParticipant.id, dyteMeetingParticipant.name, dyteMeetingParticipant.picture, dyteMeetingParticipant.audioEnabled, dyteMeetingParticipant.videoEnabled, dyteMeetingParticipant, index == waitListedParticipants.size-1)))
      }
      // add accept all button
      if (waitListedParticipants.size > 1) {
        participantPlanks.add(UiParticipantPlank.AcceptAll)
      }
    }

    val joinedParticipants = meeting.participants.joined.filter { it.name.contains(searchText, true) }
    if (waitListedParticipants.isNotEmpty()) {
      participantPlanks.add(UiParticipantPlank.Lable("In Call (${joinedParticipants.size})"))
    }
    if (joinedParticipants.size > 8) {
      participantPlanks.add(UiParticipantPlank.SearchBox)
    }
    joinedParticipants.forEachIndexed { index, dyteMeetingParticipant ->
      participantPlanks.add(UiParticipantPlank.Joined(UiParticipant(dyteMeetingParticipant.id, dyteMeetingParticipant.name, dyteMeetingParticipant.picture, dyteMeetingParticipant.audioEnabled, dyteMeetingParticipant.videoEnabled, dyteMeetingParticipant, index == joinedParticipants.size-1)))
    }
    dyteParticipantsAdapter.submitList(participantPlanks)
  }

}
