package dyte.io.uikit.screens.polls

import android.app.Dialog
import android.content.res.Resources
import android.graphics.Rect
import android.graphics.drawable.GradientDrawable
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.DialogFragment
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize
import dyte.io.uikit.wiptoken.DyteUITokens
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.listeners.DytePollEventsListener

class DytePollsBottomSheet : BottomSheetDialogFragment() {
  private lateinit var pollsRecyclerView: RecyclerView
  private lateinit var createPollButton: Button
  private lateinit var pollsAdapter: PollsListAdapter
  private lateinit var noActivePollsTextView: TextView

  private val meetingRoom by lazy {
    DyteUIKitBuilder.dyteUIKit.meeting
  }

  private val pollUpdateListener = object : DytePollEventsListener {
    override fun onPollUpdates(pollMessages: List<DytePollMessage>) {
      super.onPollUpdates(pollMessages)
      refreshPolls(pollMessages)
    }

    override fun onNewPoll(poll: DytePollMessage) {
      super.onNewPoll(poll)
      refreshPolls(meetingRoom.polls.polls)
    }
  }

  override fun onCreate(savedInstanceState: Bundle?) {
    setStyle(DialogFragment.STYLE_NORMAL, R.style.AppBottomSheetDialogTheme)
    super.onCreate(savedInstanceState)
  }

  override fun onCreateView(
    inflater: LayoutInflater,
    container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View? {
    return inflater.inflate(R.layout.bottom_sheet_polls, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)
    (requireDialog() as BottomSheetDialog).dismissWithAnimation = true

    val shape = GradientDrawable()
    shape.setColor(tokens.colors.background.shade900)
    val radius = tokens.borderRadius.getRadius(BorderRadiusSize.TWO, requireContext().resources.displayMetrics.density)
    shape.cornerRadii = floatArrayOf(radius, radius, radius, radius, 0f, 0f, 0f, 0f)
    view.findViewById<View>(R.id.clPollsContainer).background = shape

    setUpHeader(view)
    meetingRoom.addPollEventsListener(pollUpdateListener)
    setUpChildViews(view)
    refreshPolls(meetingRoom.polls.polls)

    DyteUIKitBuilder.dyteUIKit.onPollScreenOpened()
  }

  override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
    val dialog = super.onCreateDialog(savedInstanceState)

    dialog.setOnShowListener {
      val bottomSheet: FrameLayout = (dialog as BottomSheetDialog)
        .findViewById(com.google.android.material.R.id.design_bottom_sheet)
        ?: return@setOnShowListener

      val behavior = BottomSheetBehavior.from(bottomSheet)
      behavior.state = BottomSheetBehavior.STATE_EXPANDED
      behavior.skipCollapsed = true
    }
    return dialog
  }

  override fun onDestroy() {
    super.onDestroy()
    meetingRoom.removePollEventsListener(pollUpdateListener)
  }

  private fun setUpHeader(view: View) {
    view.findViewById<TextView>(R.id.header_text_view_title).text = "Polls"
    view.findViewById<ImageView>(R.id.header_button_close).setOnClickListener {
      dismiss()
    }
  }

  private fun setUpChildViews(view: View) {
    pollsRecyclerView = view.findViewById(R.id.polls_list)
    createPollButton = view.findViewById(R.id.polls_button_create_poll)
    noActivePollsTextView = view.findViewById(R.id.polls_text_view_no_polls)

    pollsAdapter = PollsListAdapter()
    pollsRecyclerView.adapter = pollsAdapter
    pollsRecyclerView.addItemDecoration(
      PollItemDecoration(resources.getDimensionPixelSize(R.dimen.dyte_vertical_spacing_polls))
    )

    if (meetingRoom.localUser.permissions.polls.canCreate) {
      createPollButton.visible()
      createPollButton.setOnClickListener {
        showCreatePollScreen()
      }
    } else {
      createPollButton.gone()
      createPollButton.isEnabled = false
    }
  }

  private fun refreshPolls(polls: List<DytePollMessage>) {
    if (polls.isNotEmpty()) {
      noActivePollsTextView.visibility = View.GONE
      pollsRecyclerView.visibility = View.VISIBLE
      pollsAdapter = PollsListAdapter()
      pollsRecyclerView.adapter = pollsAdapter
      pollsAdapter.submitList(polls)
    } else {
      noActivePollsTextView.visibility = View.VISIBLE
      pollsRecyclerView.visibility = View.GONE
    }
  }

  private fun showCreatePollScreen() {
    val createPollBottomSheet = CreatePollBottomSheet()
    createPollBottomSheet.show(childFragmentManager, CreatePollBottomSheet::class.java.simpleName)
  }
}

internal class PollItemDecoration(private val spacing: Int) : RecyclerView.ItemDecoration() {
  override fun getItemOffsets(
    outRect: Rect, view: View,
    parent: RecyclerView, state: RecyclerView.State
  ) {
    with(outRect) {
      if (parent.getChildAdapterPosition(view) == 0) {
        top = spacing
      }
      bottom = spacing
    }
  }
}