package dyte.io.uikit.screens.settings

import android.content.res.ColorStateList
import android.graphics.Color
import android.graphics.drawable.Drawable
import android.graphics.drawable.GradientDrawable
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver.OnGlobalLayoutListener
import android.widget.AdapterView
import android.widget.AdapterView.OnItemSelectedListener
import android.widget.ArrayAdapter
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.widget.AppCompatSpinner
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.DialogFragment
import androidx.lifecycle.ViewModelProvider
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.DyteUIKitBuilder.meeting
import dyte.io.uikit.R
import dyte.io.uikit.molecules.DyteVideoPeer
import dyte.io.uikit.screens.setup.DyteSetupViewModel
import dyte.io.uikit.states.AudioState
import dyte.io.uikit.states.AudioState.AudioBlocked
import dyte.io.uikit.states.AudioState.AudioDisabled
import dyte.io.uikit.states.AudioState.AudioEnabled
import dyte.io.uikit.states.VideoState.VideoBlocked
import dyte.io.uikit.states.VideoState.VideoDisabled
import dyte.io.uikit.states.VideoState.VideoEnabled
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.ViewUtils.gone
import dyte.io.uikit.utils.ViewUtils.visible
import dyte.io.uikit.utils.displayDensity
import dyte.io.uikit.wiptoken.BorderRadiusToken.BorderRadiusSize

class DyteSettingsFragment : BottomSheetDialogFragment() {

  private lateinit var ivClose: ImageView
  private lateinit var selfPreviewMolecule: DyteVideoPeer

  private lateinit var videoDeviceSpinner: AppCompatSpinner
  private lateinit var audioDeviceSpinner: AppCompatSpinner

  private var shouldIgnoreVideoSelection = true
  private var shouldIgnoreAudioSelection = true

  private lateinit var settingsContainer: ConstraintLayout
  private lateinit var titleTextView: TextView
  private lateinit var cameraTextView: TextView
  private lateinit var speakerTextView: TextView
  private lateinit var cameraSpinnerContainer: ConstraintLayout
  private lateinit var speakerSpinnerContainer: ConstraintLayout

  private lateinit var viewModel : DyteSettingsViewModel

  override fun onCreate(savedInstanceState: Bundle?) {
    setStyle(DialogFragment.STYLE_NORMAL, R.style.AppBottomSheetDialogTheme)
    super.onCreate(savedInstanceState)
  }

  override fun onCreateView(
    inflater: LayoutInflater,
    container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View {
    return inflater.inflate(R.layout.fragment_settings, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)

    viewModel = ViewModelProvider(this)[DyteSettingsViewModel::class.java]

    view.viewTreeObserver.addOnGlobalLayoutListener(object : OnGlobalLayoutListener {
      override fun onGlobalLayout() {
        view.viewTreeObserver.removeOnGlobalLayoutListener(this)
        val dialog = dialog as BottomSheetDialog
        val bottomSheet =
          dialog.findViewById<View>(com.google.android.material.R.id.design_bottom_sheet) as FrameLayout
        val behavior = BottomSheetBehavior.from(bottomSheet)
        behavior.skipCollapsed = true
        behavior.state = BottomSheetBehavior.STATE_EXPANDED
      }
    })

    settingsContainer = view.findViewById(R.id.containerSettings)
    settingsContainer.setBackgroundColor(tokens.colors.background.shade1000)

    titleTextView = view.findViewById(R.id.tvTopBar)
    titleTextView.setTextColor(tokens.colors.text.onBackground.shade600)

    ivClose = view.findViewById(R.id.ivTopBar)
    ivClose.setOnClickListener {
      dismiss()
    }

    cameraTextView = view.findViewById(R.id.tvCameraLable)
    cameraTextView.setTextColor(tokens.colors.text.onBackground.shade800)

    speakerTextView = view.findViewById(R.id.tvAudioLable)
    speakerTextView.setTextColor(tokens.colors.text.onBackground.shade800)

    cameraSpinnerContainer = view.findViewById(R.id.clVideoSpinner)
    cameraSpinnerContainer.background = createSpinnerBackground(cameraSpinnerContainer)

    speakerSpinnerContainer = view.findViewById(R.id.clAudioSpinner)
    speakerSpinnerContainer.background = createSpinnerBackground(speakerSpinnerContainer)

    selfPreviewMolecule = view.findViewById(R.id.selfView)
    selfPreviewMolecule.refresh(meeting.localUser)

    audioDeviceSpinner = view.findViewById(R.id.spAudioSource)
    videoDeviceSpinner = view.findViewById(R.id.spVideoSource)

    viewModel.videoStateLiveData.observe(viewLifecycleOwner) { videoState ->
      when (videoState) {
        VideoBlocked -> {
          cameraTextView.gone()
          cameraSpinnerContainer.gone()
        }
        is VideoEnabled, VideoDisabled -> {
          refreshVideoDeviceSelection()
        }
      }
    }

    viewModel.audioStateLiveData.observe(viewLifecycleOwner) { audioState ->
      when (audioState) {
        AudioBlocked -> {
          speakerTextView.gone()
          speakerSpinnerContainer.gone()
        }
        is AudioEnabled, AudioDisabled -> {
          refreshAudioDeviceSelection()
        }
      }
    }

    viewModel.start()
  }

  private fun refreshVideoDeviceSelection() {
    val videoDevices = viewModel.videoDevices
    val videoAdapter = ArrayAdapter(
      requireContext(),
      R.layout.dyte_spinner_item,
      videoDevices.map { it.type.displayName })
    videoAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item)
    val selectedVideoDevice = viewModel.selectedVideoDevice
    val selectedPosition = videoDevices.indexOfFirst { it.type == selectedVideoDevice.type }
    videoDeviceSpinner.adapter = videoAdapter
    videoDeviceSpinner.onItemSelectedListener = object : OnItemSelectedListener {
      override fun onItemSelected(parent: AdapterView<*>?, view: View?, position: Int, id: Long) {
        if (shouldIgnoreVideoSelection) {
          shouldIgnoreVideoSelection = false
          return
        }
        val device = videoDevices[position]
        viewModel.setVideoDevice(device)
      }

      override fun onNothingSelected(parent: AdapterView<*>?) {
      }
    }
    videoDeviceSpinner.setSelection(selectedPosition, false)
  }

  private fun refreshAudioDeviceSelection() {
    val audioDevices = viewModel.audioDevices
    val audioAdapter = ArrayAdapter(
      requireContext(),
      R.layout.dyte_spinner_item,
      audioDevices.map { it.type.displayName })
    audioAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item)
    val selectedAudioDevice = viewModel.selectedAudioDevice
    val selectedAudioPosition = audioDevices.indexOfFirst { it.type == selectedAudioDevice.type }
    audioDeviceSpinner.adapter = audioAdapter
    audioDeviceSpinner.onItemSelectedListener = object : OnItemSelectedListener {
      override fun onItemSelected(parent: AdapterView<*>?, view: View?, position: Int, id: Long) {
        if (shouldIgnoreAudioSelection) {
          shouldIgnoreAudioSelection = false
          return
        }
        val device = audioDevices[position]
        viewModel.setAudioDevice(device)
      }

      override fun onNothingSelected(parent: AdapterView<*>?) {
      }
    }
    audioDeviceSpinner.setSelection(selectedAudioPosition, false)
  }

  private fun createSpinnerBackground(view: View): Drawable {
    val shape = GradientDrawable()
    // Set color
    shape.setColor(tokens.colors.background.shade900)
    // Set corner radius
    shape.cornerRadius = tokens.borderRadius.getRadius(BorderRadiusSize.ONE, view.displayDensity)

    return shape
  }
}