package dyte.io.uikit.screens.setup

import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.fragment.app.Fragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.ViewModelProvider
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.R
import dyte.io.uikit.atoms.DyteButton
import dyte.io.uikit.atoms.DyteImageButton
import dyte.io.uikit.atoms.DyteInputFieldAtom
import dyte.io.uikit.atoms.DyteLabelAtom
import dyte.io.uikit.molecules.DyteVideoPeer
import dyte.io.uikit.screens.settings.DyteSettingsFragment
import dyte.io.uikit.states.AudioState.AudioBlocked
import dyte.io.uikit.states.AudioState.AudioDisabled
import dyte.io.uikit.states.AudioState.AudioEnabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SettingsState.Disabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SettingsState.Enabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SetupState.Setup
import dyte.io.uikit.states.VideoState.VideoBlocked
import dyte.io.uikit.states.VideoState.VideoDisabled
import dyte.io.uikit.states.VideoState.VideoEnabled
import dyte.io.uikit.tokens
import dyte.io.uikit.utils.ViewUtils.invisible
import dyte.io.uikit.utils.ViewUtils.setBackground
import dyte.io.uikit.utils.ViewUtils.visible
import kotlinx.coroutines.launch

class DyteSetupFragment : Fragment() {

  private lateinit var viewModel: DyteSetupViewModel

  private lateinit var dataContainer: View

  private lateinit var nameAtom: DyteInputFieldAtom
  private lateinit var selfPreviewMolecule: DyteVideoPeer
  private lateinit var cameraButton: DyteImageButton
  private lateinit var micButton: DyteImageButton
  private lateinit var settingsButton: DyteImageButton
  private lateinit var joinButton: DyteButton

  override fun onCreateView(
    inflater: LayoutInflater, container: ViewGroup?,
    savedInstanceState: Bundle?
  ): View {
    return inflater.inflate(R.layout.fragment_setup, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)

    viewModel = ViewModelProvider(this)[DyteSetupViewModel::class.java]

    selfPreviewMolecule = view.findViewById(R.id.selfView)

    view.findViewById<View>(R.id.rooView)
      .setBackground(tokens.borderRadius, tokens.colors.background.shade900)
    view.findViewById<DyteLabelAtom>(R.id.dyteLabelView)
      .setTextColor(tokens.colors.text.onBackground.shade600)

    nameAtom = view.findViewById(R.id.textInputEditText)
    nameAtom.setBackground(tokens.borderRadius, tokens.colors.background.shade900)

    cameraButton = view.findViewById(R.id.cameraButton)
    micButton = view.findViewById(R.id.micButton)
    settingsButton = view.findViewById(R.id.settingsButton)

    dataContainer = view.findViewById(R.id.rooView)

    joinButton = view.findViewById(R.id.joinButton)

    nameAtom.addTextChangedListener(object : TextWatcher {
      override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
        // no-op
      }

      override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        // no-op
      }

      override fun afterTextChanged(s: Editable?) {
        viewModel.setName(s.toString())
        selfPreviewMolecule.refreshName()
      }
    })

    settingsButton.setOnClickListener {
      val settingsFragment = DyteSettingsFragment()
      settingsFragment.show(childFragmentManager, DyteSetupFragment::class.java.simpleName)
      settingsFragment.lifecycle.addObserver(LifecycleEventObserver { _, event ->
        if (event === Lifecycle.Event.ON_DESTROY) {
          selfPreviewMolecule.refresh(viewModel.getSelf())
        }
      } as LifecycleObserver)
    }

    joinButton.setOnClickListener {
      DyteUIKitBuilder.dyteUIKit.meeting.joinRoom()
    }

    cameraButton.setOnClickListener {
      if (viewModel.getSelf().videoEnabled) {
        viewModel.disableVideo()
      } else {
        viewModel.enableVideo()
      }
    }

    micButton.setOnClickListener {
      if (viewModel.getSelf().audioEnabled) {
        viewModel.disableAudio()
      } else {
        viewModel.enableAudio()
      }
    }

    dataContainer.setBackgroundColor(tokens.colors.background.shade1000)

    viewModel.audioStateLiveData.observe(viewLifecycleOwner) { audioState ->
      when (audioState) {
        AudioBlocked -> {
          micButton.invisible()
        }
        AudioDisabled, is AudioEnabled -> {
          refreshMicButton()
        }
      }
    }

    viewModel.videoStateLiveData.observe(viewLifecycleOwner) { videoState ->
      when (videoState) {
        VideoBlocked -> {
          cameraButton.invisible()
        }
        is VideoEnabled, VideoDisabled,  -> {
          selfPreviewMolecule.refresh(viewModel.getSelf())
          refreshCameraButton()
        }
      }
    }

    viewModel.settingsStateLiveData.observe(viewLifecycleOwner) { settingsState ->
      when (settingsState) {
        Enabled -> {
          settingsButton.visible()
        }
        Disabled -> {
          settingsButton.invisible()
        }
      }
    }

    viewModel.setupStateLiveData.observe(viewLifecycleOwner) { setupState ->
      when (setupState) {
        is Setup -> {
          selfPreviewMolecule.refresh(viewModel.getSelf())
          nameAtom.setText(viewModel.getSelf().name)
        }
      }
    }

    viewModel.start()
  }

  override fun onResume() {
    super.onResume()
    selfPreviewMolecule.refresh(viewModel.getSelf())
  }

  private fun refreshMicButton() {
    val isAudioEnabled = viewModel.getSelf().audioEnabled
    if (isAudioEnabled) {
      micButton.setImageResource(R.drawable.ic_mic)
    } else {
      micButton.setImageResource(R.drawable.ic_mic_off)
    }
  }

  private fun refreshCameraButton() {
    val isVideoEnabled = viewModel.getSelf().videoEnabled
    if (isVideoEnabled) {
      cameraButton.setImageResource(R.drawable.ic_video)
    } else {
      cameraButton.setImageResource(R.drawable.ic_video_off)
    }
  }
}