package dyte.io.uikit.screens.setup

import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dyte.io.uikit.DyteUIKitBuilder
import dyte.io.uikit.states.AudioState.AudioBlocked
import dyte.io.uikit.states.AudioState.AudioDisabled
import dyte.io.uikit.states.AudioState.AudioEnabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SettingsState.Disabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SettingsState.Enabled
import dyte.io.uikit.screens.setup.DyteSetupViewModel.SetupState.Setup
import dyte.io.uikit.states.VideoState.VideoBlocked
import dyte.io.uikit.states.VideoState.VideoDisabled
import dyte.io.uikit.states.VideoState.VideoEnabled
import dyte.io.uikit.states.AudioState
import dyte.io.uikit.states.VideoState
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteSelfParticipant
import kotlinx.coroutines.launch

class DyteSetupViewModel : ViewModel() {

  sealed class SettingsState {
    object Enabled: SettingsState()
    object Disabled: SettingsState()
  }

  sealed class SetupState {
    class Setup(val self: DyteSelfParticipant): SetupState()
  }

  val audioStateLiveData = MutableLiveData<AudioState>()
  val videoStateLiveData = MutableLiveData<VideoState>()
  val settingsStateLiveData = MutableLiveData<SettingsState>()
  val setupStateLiveData = MutableLiveData<SetupState>()

  private val meeting = DyteUIKitBuilder.meeting

  private val selfEventsListener = object : DyteSelfEventsListener {
    override fun onAudioUpdate(audioEnabled: Boolean) {
      super.onAudioUpdate(audioEnabled)
      println("DyteMobileClient | DyteSetupViewModel onAudioUpdate $audioEnabled")
      val state = if (audioEnabled) {
        AudioEnabled(meeting.localUser.getSelectedAudioDevice())
      } else {
        AudioDisabled
      }
      audioStateLiveData.value = state
    }

    override fun onVideoUpdate(videoEnabled: Boolean) {
      super.onVideoUpdate(videoEnabled)
      println("DyteMobileClient | DyteSetupViewModel onVideoUpdate $videoEnabled")
      val state = if (videoEnabled) {
        VideoEnabled(meeting.localUser.getSelectedVideoDevice())
      } else {
        VideoDisabled
      }
      videoStateLiveData.value = state
    }
  }

  fun start() {
    meeting.addSelfEventsListener(selfEventsListener)
    setupStateLiveData.value = Setup(meeting.localUser)
    val permissions = meeting.localUser.permissions
    if (permissions.media.canPublishAudio.not() || meeting.localUser.isMicrophonePermissionGranted.not()) {
      audioStateLiveData.value = AudioBlocked
    }
    if (permissions.media.canPublishVideo.not() || meeting.localUser.isCameraPermissionGranted.not()) {
      videoStateLiveData.value = VideoBlocked
    }
    if (audioStateLiveData.value == AudioBlocked && videoStateLiveData.value == VideoBlocked) {
      settingsStateLiveData.value = Disabled
    } else {
      settingsStateLiveData.value = Enabled
    }
  }

  fun enableAudio() {
    if (audioStateLiveData.value == AudioBlocked) {
      return
    }
    viewModelScope.launch {
      meeting.localUser.enableAudio()
    }
  }

  fun disableAudio() {
    if (audioStateLiveData.value == AudioBlocked) {
      return
    }
    viewModelScope.launch {
      meeting.localUser.disableAudio()
    }
  }

  fun enableVideo() {
    if (videoStateLiveData.value == VideoBlocked) {
      return
    }
    viewModelScope.launch {
      meeting.localUser.enableVideo()
    }
  }

  fun disableVideo() {
    if (videoStateLiveData.value == VideoBlocked) {
      return
    }
    viewModelScope.launch {
      meeting.localUser.disableVideo()
    }
  }

  fun setName(name: String) {
    meeting.localUser.setDisplayName(name)
  }

  fun getSelf() : DyteSelfParticipant {
    return meeting.localUser
  }

  override fun onCleared() {
    super.onCleared()
  }
}