package dyte.io.uikit.utils

import android.content.ContentUris
import android.content.Context
import android.net.Uri
import android.os.Environment
import android.provider.DocumentsContract
import android.provider.MediaStore
import java.io.File
import java.text.DecimalFormat
import kotlin.math.log
import kotlin.math.pow

internal object FileUtils {
  private val SIZE_UNITS = listOf("B", "KB", "MB", "GB", "TB")
  private val DECIMAL_FORMAT_TWO_DIGITS = DecimalFormat("#.##")
  private const val BASE_SIZE = 1024.0
  private const val DATA_COLUMN = "_data"

  /**
   * Returns [ImageFile.name] & [ImageFile.path] of image located at given [Uri].
   * @param context for ContentResolver
   * @param imageUri content Uri of Image
   */
  fun getImageNameAndPath(context: Context, imageUri: Uri): ImageFile? {
    val projection = arrayOf(
      MediaStore.Images.Media.DISPLAY_NAME,
      MediaStore.Images.Media.DATA
    )
    context.applicationContext.contentResolver.query(
      imageUri,
      projection,
      null,
      null,
      null
    )?.use { cursor ->
      val displayNameColumn = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DISPLAY_NAME)
      val dataColumn: Int = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA)
      cursor.moveToFirst()
      return ImageFile(
        name = cursor.getString(displayNameColumn),
        path = cursor.getString(dataColumn)
      )
    }
    return null
  }

  fun getHumanReadableSize(sizeInBytes: Long): String {
    val size = sizeInBytes.toDouble()
    val i = log(size, BASE_SIZE).toInt()
    val humanReadableSize = size / BASE_SIZE.pow(i)
    return "${DECIMAL_FORMAT_TWO_DIGITS.format(humanReadableSize)} ${SIZE_UNITS[i]}"
  }

  internal data class ImageFile(
    val name: String,
    val path: String
  )
}