package dyte.io.uikit.wiptoken

import dyte.io.uikit.utils.dpToPx

/*
Can we move the implementation of getWidth inside BorderWidthToken class
and make only the thicknessFactor abstract and protected?
In that case, the  sub-classes None, Thin and Fat will provide value for the thicknessFactor.

For now, the calculation part is inside the subclasses. We can change it if needed after the
remaining parts of our Design system are implemented in the coming days.
*/
sealed class BorderRadiusToken {
  enum class BorderRadiusSize {
    ZERO,
    ONE,
    TWO,
    THREE
  }

  /**
   * Returns border radius in PX, based on the [BorderRadiusSize].
   *
   * @param size border radius size
   * @param displayDensity of the device, needed for calculating radius in PX
   *
   * @return border radius in PX unit
   */
  // NOTE: Border radius is first calculated in DP as it is mapped with our Figma Design.
  internal abstract fun getRadius(size: BorderRadiusSize, displayDensity: Float = 1f): Float


  object Sharp : BorderRadiusToken() {
    override fun getRadius(size: BorderRadiusSize, displayDensity: Float): Float {
      return 0f
    }
  }

  object Rounded : BorderRadiusToken() {
    private const val roundnessFactor = 4.0f

    override fun getRadius(size: BorderRadiusSize, displayDensity: Float): Float {
      val radius = when (size) {
        BorderRadiusSize.ZERO -> 0f * roundnessFactor
        BorderRadiusSize.ONE -> 1f * roundnessFactor
        BorderRadiusSize.TWO -> 2f * roundnessFactor
        BorderRadiusSize.THREE -> 3f * roundnessFactor
      }

      return radius.dpToPx(displayDensity)
    }
  }

  object ExtraRounded : BorderRadiusToken() {
    private const val roundnessFactor = 8.0f

    override fun getRadius(size: BorderRadiusSize, displayDensity: Float): Float {
      val radius = when (size) {
        BorderRadiusSize.ZERO -> 0f * roundnessFactor
        BorderRadiusSize.ONE -> 1f * roundnessFactor
        BorderRadiusSize.TWO -> 2f * roundnessFactor
        BorderRadiusSize.THREE -> 3f * roundnessFactor
      }

      return radius.dpToPx(displayDensity)
    }
  }

  object Circular : BorderRadiusToken() {
    private const val roundnessFactor = 12.0f

    override fun getRadius(size: BorderRadiusSize, displayDensity: Float): Float {
      if (size == BorderRadiusSize.ONE) return 9999f

      val radius = when (size) {
        BorderRadiusSize.ZERO -> 0f * roundnessFactor
        BorderRadiusSize.ONE -> 1f * roundnessFactor
        BorderRadiusSize.TWO -> 2f * roundnessFactor
        BorderRadiusSize.THREE -> 3f * roundnessFactor
      }

      return radius.dpToPx(displayDensity)
    }
  }
}