package dyte.io.uikit.wiptoken

import dyte.io.uikit.utils.dpToPx

/*
Can we move the implementation of getWidth inside BorderWidthToken class
and make only the thicknessFactor abstract and protected?
In that case, the  sub-classes None, Thin and Fat will provide value for the thicknessFactor.

For now, the calculation part is inside the subclasses. We can change it if needed after the
remaining parts of our Design system are implemented in the coming days.
*/
sealed class BorderWidthToken {
  internal enum class BorderWidthSize {
    ZERO,
    ONE,
    TWO,
    THREE
  }

  /**
   * Returns border width in PX, based on the [BorderWidthSize].
   *
   * @param size border width size
   * @param displayDensity of the device, needed for calculating width in PX
   *
   * @return border width in PX unit
   */
  // NOTE: Border width is first calculated in DP as it is mapped with our Figma Design.
  internal abstract fun getWidth(size: BorderWidthSize, displayDensity: Float = 1f): Float


  object None : BorderWidthToken() {
    override fun getWidth(size: BorderWidthSize, displayDensity: Float): Float {
      return 0f
    }
  }

  object Thin : BorderWidthToken() {
    private const val thicknessFactor = 1.0f

    override fun getWidth(size: BorderWidthSize, displayDensity: Float): Float {
      val width = when (size) {
        BorderWidthSize.ZERO -> 0f * thicknessFactor
        BorderWidthSize.ONE -> 1f * thicknessFactor
        BorderWidthSize.TWO -> 2f * thicknessFactor
        BorderWidthSize.THREE -> 4f * thicknessFactor
      }

      return width.dpToPx(displayDensity)
    }
  }

  object Fat : BorderWidthToken() {
    private const val thicknessFactor = 2.0f

    override fun getWidth(size: BorderWidthSize, displayDensity: Float): Float {
      val width = when (size) {
        BorderWidthSize.ZERO -> 0f * thicknessFactor
        BorderWidthSize.ONE -> 1f * thicknessFactor
        BorderWidthSize.TWO -> 2f * thicknessFactor
        BorderWidthSize.THREE -> 4f * thicknessFactor
      }

      return width.dpToPx(displayDensity)
    }
  }
}
