package io.embrace.android.embracesdk

import android.app.Activity
import android.app.Application.ActivityLifecycleCallbacks
import android.os.Build
import android.os.Bundle
import androidx.annotation.RequiresApi
import com.google.gson.annotations.SerializedName
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_CREATE
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_DESTROY
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_PAUSE
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_RESUME
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_SAVE_INSTANCE_STATE
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_START
import io.embrace.android.embracesdk.ActivityLifecycleState.ON_STOP
import java.util.Queue
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ConcurrentLinkedQueue

/**
 * The possible Activity lifecycle states that we care about capturing
 */
internal enum class ActivityLifecycleState {
    ON_CREATE,
    ON_START,
    ON_RESUME,
    ON_PAUSE,
    ON_STOP,
    ON_DESTROY,
    ON_SAVE_INSTANCE_STATE,
}

internal data class ActivityLifecycleData(
    @SerializedName("a")
    internal val activity: String?,

    @SerializedName("d")
    internal val data: List<ActivityLifecycleBreadcrumb>?
)

/**
 * The data for the activity lifecycle breadcrumb. Note that this does not have the same structure
 * as the Breadcrumb.java interface.
 */
internal data class ActivityLifecycleBreadcrumb(

    @Transient
    internal val activity: String?,

    @SerializedName("s")
    internal val state: ActivityLifecycleState,

    @SerializedName("st")
    internal val start: Long?,

    @SerializedName("b")
    internal var bundlePresent: Boolean? = false,

    @SerializedName("en")
    internal var end: Long? = -1,
)

// TODO future: decide on a configurable limit?
private const val LIMIT = 80

/**
 * Captures activity lifecycle breadcrumbs whenever the system alters the lifecycle of any
 * Activity in the app.
 *
 * Breadcrumbs are captured separately for each activity and the duration of each activity
 * lifecycle is also collected. This allows in principle for aggregate metrics on
 * abnormally long lifecycle to be detected and shown to the user.
 */
@RequiresApi(Build.VERSION_CODES.Q)
internal class ActivityLifecycleBreadcrumbCollector(
    private val configService: ConfigService,
    private val clock: Clock
) : ActivityLifecycleCallbacks, SessionEndListener {

    // store breadcrumbs in a map with the activity hash code as the key
    private val crumbs = ConcurrentHashMap<String, Queue<ActivityLifecycleBreadcrumb>>()

    // onCreate()
    override fun onActivityPreCreated(activity: Activity, savedInstanceState: Bundle?) =
        createBreadcrumb(activity, ON_CREATE, savedInstanceState != null)

    override fun onActivityPostCreated(activity: Activity, savedInstanceState: Bundle?) =
        endBreadcrumb(activity)

    // onStart()
    override fun onActivityPreStarted(activity: Activity) = createBreadcrumb(activity, ON_START)
    override fun onActivityPostStarted(activity: Activity) = endBreadcrumb(activity)

    // onResume()
    override fun onActivityPreResumed(activity: Activity) = createBreadcrumb(activity, ON_RESUME)
    override fun onActivityPostResumed(activity: Activity) = endBreadcrumb(activity)

    // onPause()
    override fun onActivityPrePaused(activity: Activity) = createBreadcrumb(activity, ON_PAUSE)
    override fun onActivityPostPaused(activity: Activity) = endBreadcrumb(activity)

    // onStop()
    override fun onActivityPreStopped(activity: Activity) = createBreadcrumb(activity, ON_STOP)
    override fun onActivityPostStopped(activity: Activity) = endBreadcrumb(activity)

    // onDestroy()
    override fun onActivityPreDestroyed(activity: Activity) = createBreadcrumb(activity, ON_DESTROY)
    override fun onActivityPostDestroyed(activity: Activity) = endBreadcrumb(activity)

    // onSaveInstanceState()
    override fun onActivityPreSaveInstanceState(activity: Activity, outState: Bundle) =
        createBreadcrumb(activity, ON_SAVE_INSTANCE_STATE)

    override fun onActivityPostSaveInstanceState(activity: Activity, outState: Bundle) =
        endBreadcrumb(activity)

    // no-ops
    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {}
    override fun onActivityStarted(activity: Activity) {}
    override fun onActivityResumed(activity: Activity) {}
    override fun onActivityPaused(activity: Activity) {}
    override fun onActivityStopped(activity: Activity) {}
    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {}
    override fun onActivityDestroyed(activity: Activity) {}

    /**
     * Creates a breadcrumb for the upcoming state change in the Activity lifecycle.
     */
    private fun createBreadcrumb(
        activity: Activity,
        state: ActivityLifecycleState,
        bundlePresent: Boolean? = false
    ) {
        val name = activity.javaClass.simpleName
        val queue = crumbs.getOrPut(name) { ConcurrentLinkedQueue() }
        val crumb = ActivityLifecycleBreadcrumb(
            name,
            state,
            clock.now(),
            bundlePresent
        )
        queue.add(crumb)

        while (queue.size > LIMIT) {
            queue.poll()
        }
    }

    private fun endBreadcrumb(activity: Activity) {
        val name = activity.javaClass.simpleName
        val queue = crumbs[name]
        val crumb = queue?.lastOrNull() ?: return
        crumb.end = clock.now()
    }

    /**
     * Makes a copy of the breadcrumbs captured at the current point in time as an
     * ordered list.
     */
    fun collectBreadcrumbs() = transformToSessionData(crumbs.values)

    override fun onSessionEnd(builder: Session.Builder) {
        if (configService.isBetaFeaturesEnabled) {
            val sessionData = transformToSessionData(crumbs.values)

            builder.withBetaFeatures {
                it.activityLifecycleBreadcrumbs = sessionData
            }
        }
    }

    private fun transformToSessionData(data: Collection<Queue<ActivityLifecycleBreadcrumb>>) = data
        .filter { it.isNotEmpty() }
        .map { entry ->
            val copy = entry.toList()
            val name = copy.firstOrNull()?.activity
            ActivityLifecycleData(name, copy)
        }
}
