package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;

import java.io.Closeable;
import java.util.List;

/**
 * Service which detects when the application is not responding.
 */
interface AnrService extends Closeable {
    /**
     * Gets the intervals during which the application was not responding (ANR).
     *
     * @param startTime the time to search from
     * @param endTime the time to search until
     * @return the list of ANR intervals
     */
    List<AnrInterval> getAnrIntervals(long startTime, long endTime);

    /**
     * Forces ANR tracking stop by closing the monitoring thread when a crash is
     * handled by the {@link EmbraceCrashService}.
     */
    void forceANRTrackingStopOnCrash();

    /**
     * Finishes initialization of the AnrService so that it can react appropriately to
     * lifecycle changes and capture the correct data according to the config. This is necessary
     * as the service can be initialized before the rest of the SDK.
     *
     * @param memoryCleanerService the memoryCleanerService
     * @param activityService the activityService
     * @param configService the configService
     */
    void finishInitialization(@NonNull MemoryCleanerService memoryCleanerService,
                              @NonNull ActivityService activityService,
                              @NonNull ConfigService configService);

    /**
     * Adds a listener which is invoked when the thread becomes blocked/unblocked.
     */
    void addBlockedThreadListener(BlockedThreadListener listener);
}
