package io.embrace.android.embracesdk

import androidx.annotation.VisibleForTesting
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

/**
 * This class is responsible for tracking the state of the ANR service itself, such as whether an
 * ANR is in progress, whether the process is crashing, etc.
 *
 * It also invokes any delegates which are set. This internal API allows callers to
 * add custom behaviour in response to an ANR.
 */
internal class AnrServiceState @JvmOverloads constructor(
    clock: Clock,
    @VisibleForTesting internal val listeners: CopyOnWriteArrayList<BlockedThreadListener> = CopyOnWriteArrayList()
) :
    BlockedThreadListener {

    @JvmField
    val started = AtomicBoolean(false)

    @Volatile
    var lastTargetThreadResponseMs: Long = clock.now()

    @Volatile
    var lastMonitorThreadResponseMs: Long = clock.now()

    @Volatile
    var anrInProgress = false

    @Volatile
    var isProcessCrashing = false

    override fun onThreadBlocked(thread: Thread, timestamp: Long) {
        anrInProgress = true

        listeners.forEach {
            it.onThreadBlocked(thread, timestamp)
        }
    }

    override fun onThreadBlockedInterval(thread: Thread, timestamp: Long) {
        listeners.forEach {
            it.onThreadBlockedInterval(thread, timestamp)
        }
    }

    override fun onThreadUnblocked(thread: Thread, timestamp: Long) {
        anrInProgress = false

        listeners.forEach {
            it.onThreadUnblocked(thread, timestamp)
        }
    }

    fun addListeners(listeners: List<BlockedThreadListener>) {
        this.listeners.addAllAbsent(listeners)
    }
}
