package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.regex.Pattern;

/**
 * This class is responsible for tracking the state of JVM stacktraces sampled during an ANR.
 */
class AnrStacktraceState implements BlockedThreadListener, MemoryCleanerListener {

    private final Thread targetThread;
    final NavigableMap<Long, AnrInterval> anrIntervals = new ConcurrentSkipListMap<>();
    final Map<Long, Integer> currentStacktraceStates = new HashMap<>();
    AnrStacktraces stacktraces = new AnrStacktraces();

    private ConfigService configService;
    private long lastUnblockedMs = 0;

    AnrStacktraceState(ConfigService configService, Thread targetThread) {
        this.configService = configService;
        this.targetThread = targetThread;
    }

    void setConfigService(ConfigService configService) {
        this.configService = configService;
    }

    @Override
    public void onThreadBlocked(@NonNull Thread thread, long timestamp) {
        currentStacktraceStates.clear();
        lastUnblockedMs = timestamp;
    }

    @Override
    public void onThreadBlockedInterval(@NonNull Thread thread, long timestamp) {
        AnrTick anrTick = new AnrTick(timestamp);

        for (ThreadInfo threadInfo : getAllowedThreads()) {
            // Compares every thread with the last known thread state via hashcode. If hashcode changed
            // it should be added to the anrInfo list and also the currentAnrInfoState must be updated.
            Integer threadHash = currentStacktraceStates.get(threadInfo.getThreadId());
            if (threadHash != null) {
                if (threadHash != threadInfo.hashCode()) {
                    updateThread(threadInfo, anrTick);
                }
            } else {
                updateThread(threadInfo, anrTick);
            }
        }
        stacktraces.add(anrTick);
    }

    @Override
    public void onThreadUnblocked(@NonNull Thread thread, long timestamp) {
        // Finalize AnrInterval
        AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
        long responseMs = lastUnblockedMs;
        anrIntervalBuilder
                .withStartTime(responseMs)
                .withLastKnownTime(timestamp)
                .withEndTime(timestamp)
                .withType(AnrInterval.Type.UI);

        long duration = timestamp - responseMs;
        boolean shouldCaptureStacktrace = configService.isAnrCaptureEnabled() && isMinimumCaptureDurationExceeded(duration);
        if (shouldCaptureStacktrace) {
            anrIntervalBuilder.withStacktraces(stacktraces);
            if (reachedAnrCaptureLimit()) {
                Objects.requireNonNull(anrIntervals.lastEntry()).getValue().removeStacktraces();
            }
        }
        anrIntervals.put(timestamp, anrIntervalBuilder.build());

        // reset state
        lastUnblockedMs = timestamp;
        currentStacktraceStates.clear();
        stacktraces = new AnrStacktraces();
    }

    boolean isMinimumCaptureDurationExceeded(long duration) {
        return duration >= configService.getConfig().getAnrStacktraceMinimumDuration();
    }

    @VisibleForTesting
    boolean reachedAnrCaptureLimit() {
        int limit = configService.getConfig().getMaxAnrCapturedIntervalsPerSession();
        return anrIntervals.size() >= limit;
    }

    @Override
    public void cleanCollections() {
        anrIntervals.clear();
    }

    @VisibleForTesting
    void updateThread(ThreadInfo threadInfo, AnrTick anrTick) {
        currentStacktraceStates.put(threadInfo.getThreadId(), threadInfo.hashCode());
        anrTick.add(threadInfo);
    }

    /**
     * Filter the thread list based on allow/block list get by config.
     *
     * @return filtered threads
     */
    @VisibleForTesting
    Set<ThreadInfo> getAllowedThreads() {
        Set<ThreadInfo> allowed = new HashSet<>();
        List<Pattern> blockList = configService.getConfig().getAnrBlockPatternList();
        List<Pattern> allowList = configService.getConfig().getAnrAllowPatternList();
        int anrStacktracesMaxLength = configService.getConfig().getAnrStacktracesMaxDepth();
        int priority = configService.getConfig().getAnrThreadCapturePriority();

        if (configService.getConfig().captureMainThreadOnly()) {
            ThreadInfo threadInfo = ThreadInfo.ofThread(
                    targetThread,
                    targetThread.getStackTrace(),
                    anrStacktracesMaxLength);

            allowed.add(threadInfo);
        } else {
            for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
                ThreadInfo threadInfo = ThreadInfo.ofThread(
                        entry.getKey(),
                        entry.getValue(),
                        anrStacktracesMaxLength);

                if (allowList != null && !allowList.isEmpty()) {
                    if (isAllowedByList(allowList, threadInfo) &&
                            isAllowedByPriority(priority, threadInfo)) {
                        allowed.add(threadInfo);
                    }
                } else if (blockList != null && !blockList.isEmpty()) {
                    if (!isAllowedByList(blockList, threadInfo)
                            && isAllowedByPriority(priority, threadInfo)) {
                        allowed.add(threadInfo);
                    }
                } else {
                    if (isAllowedByPriority(priority, threadInfo)) {
                        allowed.add(threadInfo);
                    }
                }
            }
        }
        return allowed;
    }

    private boolean isAllowedByList(List<Pattern> allowed, ThreadInfo threadInfo) {
        for (Pattern pattern : allowed) {
            if (pattern.matcher(threadInfo.getName()).find()) {
                return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    boolean isAllowedByPriority(int priority, ThreadInfo threadInfo) {
        if (priority != 0) {
            return threadInfo.getPriority() >= priority;
        }
        return true;
    }
}
