package io.embrace.android.embracesdk;

import android.content.Context;
import android.os.Debug;

import androidx.annotation.NonNull;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.nio.charset.Charset;
import java.util.List;
import java.util.concurrent.Future;
import java.util.zip.GZIPOutputStream;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.networking.EmbraceConnection;
import io.embrace.android.embracesdk.networking.EmbraceUrl;
import io.embrace.android.embracesdk.utils.BiConsumer;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.exceptions.Unchecked;
import kotlin.Lazy;
import kotlin.LazyKt;

/**
 * Client for calling the Embrace API.
 */
class ApiClient implements Closeable {

    /**
     * The version of the API message format.
     */
    static final int MESSAGE_VERSION = 13;

    private static final int CONFIG_API_VERSION = 2;

    private static final int API_VERSION = 1;

    private final String screenshotUrl;

    private final String configBaseUrl;
    private final String appId;
    private final String operatingSystemCode = MetadataUtils.getOperatingSystemVersionCode() + ".0.0";
    private final BackgroundWorker worker = BackgroundWorker.ofSingleThread("API Client");
    private final Lazy<Gson> gson = LazyKt.lazy(Gson::new);
    private final ScheduledWorker retryWorker = ScheduledWorker.ofSingleThread("API Retry");
    private final LocalConfig localConfig;
    private final MetadataService metadataService;
    private final CacheService cacheService;
    private final boolean enableIntegrationTesting;
    private ApiClientRetryWorker<SessionMessage> sessionRetryWorker;
    private ApiClientRetryWorker<EventMessage> eventRetryWorker;
    private String coreBaseUrl;
    private String appVersion;
    private InternalEmbraceLogger logger;

    /**
     * Creates an instance of the API client. This service handles all calls to the Embrace API.
     * <p>
     * Sessions can be sent to either the production or development endpoint. The development
     * endpoint shows sessions on the 'integration testing' screen within the dashboard, whereas
     * the production endpoint sends sessions to 'recent sessions'.
     * <p>
     * The development endpoint is only used if the build is a debug build, and if integration
     * testing is enabled when calling {@link Embrace#start(Context, boolean)} )}.
     *
     * @param metadataService          the metadata service
     * @param cacheService             the cache service
     * @param enableIntegrationTesting true if sessions should be sent to the integration testing
     *                                 screen in the dashboard for debug builds, false if they
     *                                 should be sent to 'recent sessions'
     */
    ApiClient(
                LocalConfig localConfig,
                MetadataService metadataService,
                CacheService cacheService,
                boolean enableIntegrationTesting,
                InternalEmbraceLogger logger) {

        this.localConfig = Preconditions.checkNotNull(localConfig, "localConfig must not be null");
        this.metadataService = Preconditions.checkNotNull(metadataService, "metadataService must not be null");
        this.cacheService = Preconditions.checkNotNull(cacheService, "cacheService must not be null");

        this.enableIntegrationTesting = enableIntegrationTesting;

        LocalConfig.SdkConfigs.BaseUrls baseUrls = localConfig.getConfigurations().getBaseUrls();
        this.configBaseUrl = buildUrl(baseUrls.getConfig(), CONFIG_API_VERSION, "config");
        this.screenshotUrl = buildUrl(baseUrls.getImages(), API_VERSION, "screenshot");

        this.appId = metadataService.getAppId();

        this.logger = logger;
    }

    @NonNull
    private String buildUrl(String config, int configApiVersion, String path) {
        return config + "/v" + configApiVersion + "/" + path;
    }

    /**
     * This method purpose is to lazy load the SessionRetryWorker upon request
     *
     * @return the SessionRetryWorker
     */
    private synchronized ApiClientRetryWorker<SessionMessage> getSessionRetryWorker() {
        if (sessionRetryWorker == null) {
            sessionRetryWorker = new ApiClientRetryWorker<>(SessionMessage.class, cacheService,
                        this, retryWorker);
        }
        return sessionRetryWorker;
    }

    /**
     * This method purpose is to lazy load the EventRetryWorker upon request
     *
     * @return the EventRetryWorker
     */
    private synchronized ApiClientRetryWorker<EventMessage> getEventRetryWorker() {
        if (eventRetryWorker == null) {
            eventRetryWorker = new ApiClientRetryWorker<>(EventMessage.class, cacheService,
                        this, retryWorker);
        }
        return eventRetryWorker;
    }

    /**
     * This method purpose is to lazy load the coreBaseUrl upon request
     *
     * @return the core base url
     */
    private synchronized String getCoreBaseUrl() {
        if (coreBaseUrl == null) {
            coreBaseUrl = metadataService.isDebug()
                        && enableIntegrationTesting
                        && (Debug.isDebuggerConnected() || Debug.waitingForDebugger()) ?
                        localConfig.getConfigurations().getBaseUrls().getDataDev() :
                        localConfig.getConfigurations().getBaseUrls().getData();
        }

        logger.logDeveloper("ApiClient", "getCoreBaseUrl - coreBaseUrl: " + coreBaseUrl);
        return coreBaseUrl;
    }

    /**
     * This method purpose is to lazy load the appVersion upon request
     *
     * @return the app version
     */
    private synchronized String getAppVersion() {
        if (appVersion == null) {
            appVersion = metadataService.getAppVersionName();
        }

        logger.logDeveloper("ApiClient", "appVersion: " + appVersion);
        return appVersion;
    }

    /**
     * Asynchronously gets the app's SDK configuration.
     * <p>
     * These settings define app-specific settings, such as disabled log patterns, whether
     * screenshots are enabled, as well as limits and thresholds.
     *
     * @return a future containing the configuration.
     */
    Future<Config> getConfig() {
        logger.logDeveloper("ApiClient", "getConfig");
        String url = getConfigUrl();
        final EmbraceConnection connection = ApiRequest.newBuilder()
                    .withHttpMethod(HttpMethod.GET)
                    .withUrl(Unchecked.wrap(() -> EmbraceUrl.getUrl(url)))
                    .build()
                    .toConnection();
        return worker.submit(() -> {
            connection.connect();
            logger.logDeveloper("ApiClient", "getConfig - connection.connect()");
            return httpCall(connection, Config.class);
        });

    }

    @NonNull
    private String getConfigUrl() {
        return configBaseUrl + "?appId=" + appId + "&osVersion=" +
                    operatingSystemCode + "&appVersion=" + getAppVersion() + "&deviceId=" + metadataService.getDeviceId();
    }

    /**
     * Asynchronously sends a screenshot corresponding to a log entry.
     *
     * @param screenshot the screenshot payload
     * @param logId      the ID of the corresponding log
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendLogScreenshot(byte[] screenshot, String logId) {
        logger.logDeveloper("ApiClient", "sendLogScreenshot");
        EmbraceUrl url = Unchecked.wrap(() ->
                    EmbraceUrl.getUrl(getLogScreenshotUrl(logId)));
        ApiRequest request = screenshotBuilder(url)
                    .withLogId(logId)
                    .build();
        return rawPost(request, screenshot);
    }

    @NonNull
    private String getLogScreenshotUrl(String logId) {
        logger.logDeveloper("ApiClient", "getLogScreenshotUrl - logId: " + logId);
        return screenshotUrl + "/" + appId + "/" + "logs" + "/" + logId + ".jpg";
    }

    /**
     * Sends a screenshot corresponding to a moment.
     *
     * @param screenshot the screenshot payload
     * @param eventId    the ID of the moment
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendMomentScreenshot(byte[] screenshot, String eventId) {
        logger.logDeveloper("ApiClient", "sendMomentScreenshot");
        EmbraceUrl url = Unchecked.wrap(() ->
                    EmbraceUrl.getUrl(getMomentScreenshotUrl(eventId)));
        ApiRequest request = screenshotBuilder(url)
                    .withEventId(eventId)
                    .build();
        return rawPost(request, screenshot);
    }

    @NonNull
    private String getMomentScreenshotUrl(String eventId) {
        return screenshotUrl + "/" + appId + "/" + "moments" + "/" + eventId + ".jpg";
    }

    /**
     * Sends a session message to the API.
     *
     * @param sessionMessage the session to send
     * @return a future containing the response body from the server
     */
    Future<String> sendSession(SessionMessage sessionMessage) {
        logger.logDeveloper("ApiClient", "sendSession");
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(getEmbraceUrlWithSuffix("sessions")));
        ApiRequest request = eventBuilder(url)
                    .withDeviceId(metadataService.getDeviceId())
                    .withAppId(appId)
                    .withUrl(url)
                    .withHttpMethod(HttpMethod.POST)
                    .withContentEncoding("gzip")
                    .build();
        return jsonPost(request, sessionMessage, SessionMessage.class, getSessionRetryWorker()::addFailedCall);
    }

    /**
     * Sends a log message to the API.
     *
     * @param eventMessage the event message containing the log entry
     * @return a future containing the response body from the server
     */
    Future<String> sendLogs(EventMessage eventMessage) {
        logger.logDeveloper("ApiClient", "sendLogs");
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getEventId(), "event ID must be set");
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(getEmbraceUrlWithSuffix("logging")));
        String abbreviation = event.getType().getAbbreviation();
        String logIdentifier = abbreviation + ":" + event.getMessageId();
        ApiRequest request = eventBuilder(url)
                    .withLogId(logIdentifier)
                    .build();
        return sendEvent(eventMessage, request);
    }

    /**
     * Sends an event to the API.
     *
     * @param eventMessage the event message containing the event
     * @return a future containing the response body from the server
     */
    Future<String> sendEvent(EventMessage eventMessage) {
        logger.logDeveloper("ApiClient", "sendEvent");
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        logger.logDeveloper("ApiClient", "sendEvent - event: " + event.getName());
        logger.logDeveloper("ApiClient", "sendEvent - event: " + event.getType());
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getEventId(), "event ID must be set");
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(getEmbraceUrlWithSuffix("events")));
        String abbreviation = event.getType().getAbbreviation();
        String eventIdentifier;
        if (event.getType().equals(EmbraceEvent.Type.CRASH)) {
            eventIdentifier = createCrashActiveEventsHeader(abbreviation, event.getActiveEventIds());
        } else {
            eventIdentifier = abbreviation + ":" + event.getEventId();
        }
        ApiRequest request = eventBuilder(url)
                    .withEventId(eventIdentifier)
                    .build();
        return sendEvent(eventMessage, request);
    }

    @NonNull
    private String getEmbraceUrlWithSuffix(String suffix) {
        logger.logDeveloper("ApiClient", "getEmbraceUrlWithSuffix - suffix: " + suffix);
        return getCoreBaseUrl() + "/v" + API_VERSION + "/log/" + suffix;
    }

    /**
     * Posts a JSON payload to the API.
     * <p>
     * If the request fails for any reason, such as a non-200 response, or an exception being thrown
     * during the connection, the failer handler will be invoked. This can be used to re-attempt
     * sending the message.
     *
     * @param request        the details of the API request
     * @param payload        the body of the API request
     * @param clazz          the class of the payload
     * @param failureHandler callback if the request fails
     * @param <T>            the type of the payload
     * @return a future containing the response body from the server
     */
    <T> Future<String> jsonPost(
                ApiRequest request,
                T payload,
                Class<T> clazz,
                BiConsumer<ApiRequest, T> failureHandler) {

        logger.logDeveloper("ApiClient", "jsonPost");

        return worker.submit(() -> {
            try {
                final byte[] bytes = gson.getValue().toJson(payload, clazz.getGenericSuperclass()).getBytes(Charset.forName("UTF-8"));
                logger.logDeveloper("ApiClient", "jsonPost - sendBytes");
                return sendBytes(request, gzip(bytes));
            } catch (Exception ex) {
                logger.logDebug("Failed to post Embrace API call. Will retry.", ex);
                failureHandler.accept(request, payload);
                throw Unchecked.propagate(ex);
            }
        });
    }

    private Future<String> sendEvent(EventMessage eventMessage, ApiRequest request) {
        return jsonPost(request, eventMessage, EventMessage.class, getEventRetryWorker()::addFailedCall);
    }

    private Future<String> rawPost(ApiRequest request, byte[] payload) {
        return worker.submit(() -> sendBytes(request, payload));
    }

    private String sendBytes(ApiRequest request, byte[] payload) {
        logger.logDeveloper("ApiClient", "sendBytes");
        return Unchecked.wrap(() -> {
            EmbraceConnection connection = request.toConnection();
            if (payload != null) {
                logger.logDeveloper("ApiClient", "sendBytes - payload " + payload.length);
                OutputStream outputStream = connection.getOutputStream();
                outputStream.write(payload);
                connection.connect();
            }
            return httpCall(connection);
        });
    }

    /**
     * Compresses a given byte array using the GZIP compression algorithm.
     *
     * @param bytes the byte array to compress
     * @return the compressed byte array
     */
    private static byte[] gzip(byte[] bytes) {
        ByteArrayOutputStream baos = null;
        GZIPOutputStream os = null;
        try {
            baos = new ByteArrayOutputStream();
            os = new GZIPOutputStream(baos);
            os.write(bytes);
            os.finish();
            return baos.toByteArray();
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (baos != null) {
                    baos.close();
                }
                if (os != null) {
                    os.close();
                }
            } catch (IOException ex) {
                InternalStaticEmbraceLogger.logDebug("Failed to close streams when gzipping payload", ex);
            }
        }
    }

    /**
     * Executes a HTTP call using the specified connection, returning the JSON response from the
     * server as an object of the specified type.
     *
     * @param connection the HTTP connection to use
     * @param clazz      the class representing the type of response
     * @param <T>        the type of the object being returned
     * @return the Java representation of the JSON object
     */
    private <T> T httpCall(EmbraceConnection connection, Class<T> clazz) {
        handleHttpResponseCode(connection);
        return readJsonObject(connection, clazz);
    }

    /**
     * Executes a HTTP call using the specified connection, returning the response from the server
     * as a string.
     *
     * @param connection the HTTP connection to use
     * @return a string containing the response from the server to the request
     */
    private String httpCall(EmbraceConnection connection) {
        logger.logDeveloper("ApiClient", "httpCall");
        handleHttpResponseCode(connection);
        return Unchecked.wrap(() -> readString(connection.getInputStream()));
    }

    /**
     * Checks that the HTTP response was 200, otherwise throws an exception.
     * <p>
     * Attempts to read the error message payload from the response, if present, and writes it to
     * the logs at warning level.
     *
     * @param connection the connection containing the HTTP response
     */
    private void handleHttpResponseCode(EmbraceConnection connection) {
        Integer responseCode = null;
        try {
            responseCode = connection.getResponseCode();
        } catch (IOException ex) {
            logger.logDeveloper("ApiClient", "Connection failed or unexpected response code");
        }

        logger.logDeveloper("ApiClient", "handleHttpResponseCode - responseCode: " + responseCode);
        if (responseCode == null || responseCode != HttpURLConnection.HTTP_OK) {
            if (responseCode != null) {
                String errorMessage = readString(connection.getErrorStream());
                logger.logDeveloper("ApiClient", "Embrace API request failed. HTTP response code: " + responseCode + ", message: " + errorMessage);
            }
            throw new IllegalStateException("Failed to retrieve from Embrace server.");
        }
    }

    /**
     * Reads the JSON response payload of a {@link HttpURLConnection} and converts it to a Java
     * object of the specified type.
     *
     * @param connection the connection to read
     * @param clazz      the class representing the type
     * @param <T>        the type of object to return
     * @return the Java representation of the JSON object
     */
    private <T> T readJsonObject(EmbraceConnection connection, Class<T> clazz) {
        logger.logDeveloper("ApiClient", "Failed to close HTTP reader");
        InputStreamReader inputStreamReader = null;
        JsonReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(connection.getInputStream());
            reader = new JsonReader(inputStreamReader);
            return gson.getValue().fromJson(reader, clazz);
        } catch (IOException ex) {
            logger.logDeveloper("ApiClient", "readJsonObject - IOException");
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                logger.logDeveloper("ApiClient", "Failed to close HTTP reader", ex);
            }
        }
    }

    /**
     * Reads an {@link InputStream} into a String.
     *
     * @param inputStream the input stream to read
     * @return the string representation
     */
    private String readString(InputStream inputStream) {
        logger.logDeveloper("ApiClient", "readString");
        InputStreamReader inputStreamReader = null;
        BufferedReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(inputStream);
            reader = new BufferedReader(inputStreamReader);
            char[] buffer = new char[4096];
            StringBuilder sb = new StringBuilder();
            for (int len; (len = reader.read(buffer)) > 0; )
                sb.append(buffer, 0, len);
            return sb.toString();
        } catch (IOException ex) {
            logger.logDeveloper("ApiClient", "readString - IOException");
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                logger.logDeveloper("ApiClient", "Failed to close HTTP reader", ex);
            }
        }
    }

    private ApiRequest.Builder screenshotBuilder(EmbraceUrl url) {
        logger.logDeveloper("ApiClient", "screenshotBuilder");
        return ApiRequest.newBuilder()
                    .withContentType("application/octet-stream")
                    .withHttpMethod(HttpMethod.POST)
                    .withAppId(appId)
                    .withDeviceId(metadataService.getDeviceId())
                    .withUrl(url);
    }

    private ApiRequest.Builder eventBuilder(EmbraceUrl url) {
        logger.logDeveloper("ApiClient", "eventBuilder");
        return ApiRequest.newBuilder()
                    .withUrl(url)
                    .withHttpMethod(HttpMethod.POST)
                    .withAppId(appId)
                    .withDeviceId(metadataService.getDeviceId())
                    .withContentEncoding("gzip");
    }

    @Override
    public void close() {
        logger.logDeveloper("ApiClient", "Shutting down ApiClient");
        retryWorker.close();
        worker.close();
    }

    /**
     * Crashes are sent with a header containing the list of active stories.
     *
     * @param abbreviation the abbreviation for the event type
     * @param eventIds     the list of story IDs
     * @return the header
     */
    private String createCrashActiveEventsHeader(String abbreviation, List<String> eventIds) {
        logger.logDeveloper("ApiClient", "createCrashActiveEventsHeader");
        String stories = "";
        if (eventIds != null) {
            stories = StreamUtilsKt.join(",", eventIds);
        }
        return abbreviation + ":" + stories;
    }
}