package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;

import io.embrace.android.embracesdk.utils.optional.Optional;

/**
 * Specifies the application ID and build ID.
 */
@InternalApi
public final class BuildInfo {

    static final String BUILD_INFO_BUILD_ID = "emb_build_id";
    static final String BUILD_INFO_BUILD_TYPE = "emb_build_type";
    static final String BUILD_INFO_BUILD_FLAVOR = "emb_build_flavor";

    private static final String RES_TYPE_STRING = "string";

    /**
     * The ID of the particular build, generated at compile-time.
     */
    private final String buildId;

    /**
     * The BuildType name of the particular build, extracted at compile-time.
     */
    private final String buildType;

    /**
     * The Flavor name of the particular build, extracted at compile-time.
     */
    private final String buildFlavor;

    BuildInfo(String buildId, String buildType, String buildFlavor) {
        this.buildId = buildId;
        this.buildType = buildType;
        this.buildFlavor = Optional.fromNullable(buildFlavor).orNull();
    }

    /**
     * Loads the build information from resources provided by the config file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */
    static BuildInfo fromResources(Context context) {
        return new BuildInfo(
                getBuildResource(context, BUILD_INFO_BUILD_ID, RES_TYPE_STRING),
                getBuildResource(context, BUILD_INFO_BUILD_TYPE, RES_TYPE_STRING),
                getBuildResource(context, BUILD_INFO_BUILD_FLAVOR, RES_TYPE_STRING));
    }

    /**
     * Given a build property name and a build property type, retrieves the embrace build resource value.
     */
    private static String getBuildResource(Context context, String buildProperty, String type) {
        try {
            Resources resources = context.getResources();
            int resourceId = resources.getIdentifier(buildProperty, type, context.getPackageName());

            // Flavor value is optional, so we should not hard fail if doesn't exists.
            if (buildProperty.equals(BUILD_INFO_BUILD_FLAVOR) && resourceId == 0) {
                return null;
            }

            return resources.getString(resourceId);
        } catch (NullPointerException | Resources.NotFoundException ex) {
            throw new IllegalArgumentException("No resource found for " + buildProperty + " property. Failed to create build info.", ex);
        }
    }


    String getBuildId() {
        return buildId;
    }

    String getBuildType() {
        return buildType;
    }

    String getBuildFlavor() {
        return buildFlavor;
    }
}
