package io.embrace.android.embracesdk;

import androidx.annotation.Nullable;

import io.embrace.android.embracesdk.config.KillSwitchConfig;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.config.AnrConfig;
import io.embrace.android.embracesdk.utils.optional.Optional;
import com.google.gson.annotations.SerializedName;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Configuration of the SDK set by the Embrace API.
 */
@InternalApi
public class Config {

    /**
     * The default percentage of devices for which the SDK is enabled.
     */
    private static final int DEFAULT_THRESHOLD = 100;

    /**
     * The default percentage offset of devices for which the SDK is enabled.
     */
    private static final int DEFAULT_OFFSET = 0;

    /**
     * Exception capture is enabled by default.
     */
    private static final Boolean DEFAULT_INTERNAL_EXCEPTION_CAPTURE = true;

    /**
     * The percentage of devices which should have beta features initialized.
     *
     * The range of allowed values is 0.0f to 100.0f, and the default is 1.0f (1% of devices).
     */
    private static final float DEFAULT_BETA_FEATURES_PCT = 1.0f;

    private static final Boolean SCREENSHOT_ENABLED_DEFAULT = false;

    private Boolean stacktraceSamplingEnabled = null;

    private Boolean bgAnrCaptureEnabled = null;

    /**
     * Used to determine whether or not the SDK should be activated for this device. The threshold
     * identifies the percentage of devices for which the SDK is enabled. A threshold of 100 means
     * that the SDK is enabled for all devices, whilst 0 means it is disabled for all devices.
     */
    @SerializedName("threshold")
    private final Integer threshold;

    /**
     * Used to shift the offset of devices for which the SDK is enabled/disabled.
     */
    @SerializedName("offset")
    private final Integer offset;

    /**
     * Whether screenshots should be taken for slow moments or error logs.
     */
    @SerializedName("screenshots_enabled")
    private final Boolean screenshotsEnabled;

    /**
     * The time in milliseconds after which a particular event ID is considered 'late'.
     */
    @SerializedName("event_limits")
    private final Map<String, Long> eventLimits;

    /**
     * The list of {@link MessageType} which are disabled.
     */
    @SerializedName("disabled_message_types")
    private final Set<String> disabledMessageTypes;

    /**
     * List of regular expressions matching event names and log messages which should be disabled.
     */
    @SerializedName("disabled_event_and_log_patterns")
    private final Set<String> disabledEventAndLogPatterns;

    /**
     * List of regular expressions of URLs which should not be logged.
     */
    @SerializedName("disabled_url_patterns")
    private final Set<String> disabledUrlPatterns;

    /**
     * List of regular expressions matching event names for which screenshots should be disabled.
     */
    @SerializedName("disabled_screenshot_patterns")
    private final Set<String> disabledScreenshotPatterns;

    /**
     * Settings relating to the user interface, such as the breadcrumb limits.
     */
    @SerializedName("ui")
    private final UiConfig uiConfig;

    /**
     * Settings defining the capture limits for network calls.
     */
    @SerializedName("network")
    private final NetworkConfig networkConfig;

    /**
     * Settings defining session control is enabled or not
     */
    @SerializedName("session_control")
    SessionControl sessionControl;

    /**
     * Settings defining the log configuration.
     */
    @SerializedName("logs")
    private final LogConfig logConfig;

    /**
     * Settings defining if internal exception capture is enabled or not
     */
    @SerializedName("internal_exception_capture_enabled")
    private final Boolean internalExceptionCaptureEnabled;

    @SerializedName("anr")
    private final AnrConfig anrConfig;

    @SerializedName("killswitch")
    private final KillSwitchConfig killSwitchConfig;

    @SerializedName("startup_sampling")
    private final StartupSamplingConfig startupSamplingConfig;

    @SerializedName("pct_beta_features_enabled")
    private final Float pctBetaFeaturesEnabled;

    private final transient Random random;

    Config(
            Integer threshold,
            Integer offset,
            Boolean screenshotsEnabled,
            Map<String, Long> eventLimits,
            Set<String> disabledMessageTypes,
            Set<String> disabledEventAndLogPatterns,
            Set<String> disabledUrlPatterns,
            Set<String> disabledScreenshotPatterns,
            UiConfig uiConfig,
            NetworkConfig networkConfig,
            SessionControl sessionControl,
            LogConfig logConfig,
            AnrConfig anrConfig,
            KillSwitchConfig killSwitchConfig,
            StartupSamplingConfig startupSamplingConfig,
            Boolean internalExceptionCaptureEnabled,
            Float pctBetaFeaturesEnabled,
            Random random
    ) {

        this.threshold = threshold;
        this.offset = offset;
        this.screenshotsEnabled = screenshotsEnabled;
        this.eventLimits = eventLimits;
        this.disabledMessageTypes = disabledMessageTypes;
        this.disabledEventAndLogPatterns = disabledEventAndLogPatterns;
        this.disabledUrlPatterns = disabledUrlPatterns;
        this.disabledScreenshotPatterns = disabledScreenshotPatterns;
        this.uiConfig = uiConfig;
        this.networkConfig = networkConfig;
        this.sessionControl = sessionControl;
        this.logConfig = logConfig;
        this.anrConfig = anrConfig;
        this.killSwitchConfig = killSwitchConfig;
        this.startupSamplingConfig = startupSamplingConfig;
        this.internalExceptionCaptureEnabled = internalExceptionCaptureEnabled;
        this.pctBetaFeaturesEnabled = pctBetaFeaturesEnabled;
        this.random = random;
    }

    /**
     * Creates SDK configuration based on default settings (threshold of 100, offset of 0, log limit of 10, with
     * screenshots enabled).
     *
     * @return the default SDK configuration
     */
    static Config ofDefault() {
        return ofDefault(AnrConfig.ofDefault(), StartupSamplingConfig.ofDefault(), new Random());
    }

    /**
     * Creates SDK configuration based on default settings (threshold of 100, offset of 0, log limit of 10, with
     * screenshots enabled).
     *
     * @return the default SDK configuration
     * @param anrConfig the anr config
     */
    static Config ofDefault(AnrConfig anrConfig, StartupSamplingConfig startupSamplingConfig, Random random) {
        return new Config(
                DEFAULT_THRESHOLD,
                DEFAULT_OFFSET,
                SCREENSHOT_ENABLED_DEFAULT,
                Collections.emptyMap(),
                Collections.emptySet(),
                Collections.emptySet(),
                Collections.emptySet(),
                Collections.emptySet(),
                new UiConfig(null, null, null, null, null),
                new NetworkConfig(null, null),
                new SessionControl(true, false, null, null),
                new LogConfig(null, null, null, null),
                anrConfig,
                new KillSwitchConfig(),
                startupSamplingConfig,
                DEFAULT_INTERNAL_EXCEPTION_CAPTURE,
                DEFAULT_BETA_FEATURES_PCT,
                random
        );
    }

    Integer getThreshold() {
        return threshold != null ? threshold : DEFAULT_THRESHOLD;
    }

    Integer getOffset() {
        return offset != null ? offset : DEFAULT_OFFSET;
    }

    Boolean getScreenshotsEnabled() {
        return screenshotsEnabled != null ? screenshotsEnabled : SCREENSHOT_ENABLED_DEFAULT;
    }

    @Deprecated
    Boolean getSignalStrengthEnabled() {
        InternalStaticEmbraceLogger.logWarning("Warning: signal strength enabled config usage detected. " +
                "The signal quality service is deprecated.");
        return false;
    }

    Map<String, Long> getEventLimits() {
        return eventLimits != null ? eventLimits : new HashMap<>();
    }

    Set<String> getDisabledMessageTypes() {
        return disabledMessageTypes != null ? disabledMessageTypes : new HashSet<>();
    }

    Set<String> getDisabledEventAndLogPatterns() {
        return disabledEventAndLogPatterns != null ? disabledEventAndLogPatterns : new HashSet<>();
    }

    Set<String> getDisabledUrlPatterns() {
        return disabledUrlPatterns != null ? disabledUrlPatterns : new HashSet<>();
    }

    Set<String> getDisabledScreenshotPatterns() {
        return disabledScreenshotPatterns != null ? disabledScreenshotPatterns : new HashSet<>();
    }

    Optional<Integer> getFragmentBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getFragments();
    }

    Optional<Integer> getViewBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getViews();
    }

    Optional<Integer> getTapBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getTaps();
    }

    Optional<Integer> getWebViewBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getWebViews();
    }

    Optional<Integer> getDefaultNetworkCallLimit() {
        return networkConfig == null ? Optional.absent() : networkConfig.getDefaultCaptureLimit();
    }

    Map<String, Integer> getNetworkCallLimitsPerDomain() {
        return networkConfig == null ? new HashMap<>() : networkConfig.getDomainLimits();
    }

    Optional<Integer> getInfoLogLimit() {
        return logConfig == null ? Optional.absent() : logConfig.getLogInfoLimit();
    }

    Optional<Integer> getWarnLogLimit() {
        return logConfig == null ? Optional.absent() : logConfig.getLogWarnLimit();
    }

    Optional<Integer> getErrorLogLimit() {
        return logConfig == null ? Optional.absent() : logConfig.getLogErrorLimit();
    }

    Boolean getSessionControl() {
        return sessionControl != null && sessionControl.isEnabled();
    }

    Boolean endSessionInBackgroundThread() {
        return sessionControl != null && sessionControl.endSessionInBackgroundThread();
    }

    @Nullable
    Set<String> getSessionComponents() {
        return sessionControl != null? sessionControl.getSessionComponents() : null;
    }

    Set<String> getFullSessionComponents() {
        return sessionControl != null? sessionControl.getFullSessionEvents() : new HashSet<>();
    }

    Optional<Integer> getLogMessageMaximumAllowedLength() {
        return logConfig == null ? Optional.absent() : logConfig.getLogMessageMaximumAllowedLength();
    }

    Boolean getInternalExceptionCaptureEnabled() {
        return internalExceptionCaptureEnabled != null ? internalExceptionCaptureEnabled : DEFAULT_INTERNAL_EXCEPTION_CAPTURE;
    }

    int getCustomBreadcrumbLimit() {
        return uiConfig != null ? uiConfig.getBreadcrumbs(): UiConfig.DEFAULT_BREADCRUMB_LIMIT;
    }

    int getAnrUsersEnabledPercentage() {
        return anrConfig != null ? anrConfig.getPctEnabled() : AnrConfig.DEFAULT_ANR_PCT_ENABLED;
    }

    int getAnrUsersGoogleEnabledPercentage() {
        return anrConfig != null ? anrConfig.getGooglePctEnabled() : AnrConfig.DEFAULT_ANR_GOOGLE_PCT_ENABLED;
    }

    int getAnrBgUsersEnabledPercentage() {
        return anrConfig != null ? anrConfig.getPctBgEnabled() : AnrConfig.DEFAULT_ANR_BG_PCT_ENABLED;
    }

    float getDefaultBetaFeaturesPct() {
        return pctBetaFeaturesEnabled != null ? pctBetaFeaturesEnabled : DEFAULT_BETA_FEATURES_PCT;
    }

    List<Pattern> getAnrAllowPatternList() {
        return anrConfig != null ? anrConfig.getAllowPatternList() : null;
    }

    List<Pattern> getAnrBlockPatternList() {
        return anrConfig != null ? anrConfig.getBlockPatternList() : null;
    }

    int getMaxAnrCapturedIntervalsPerSession() {
        return anrConfig != null ? anrConfig.getMaxAnrCapturedIntervalsPerSession() : AnrConfig.DEFAULT_ANR_MAX_ANR_INTERVALS_PER_SESSION;
    }

    int getStacktracesPerInterval() {
        return anrConfig != null ? anrConfig.getStacktracesPerInterval() : AnrConfig.DEFAULT_ANR_MAX_PER_INTERVAL;
    }

    int getAnrStacktracesMaxDepth() {
        return anrConfig != null ? anrConfig.getStacktraceMaxLength() : AnrConfig.DEFAULT_ANR_MAX_STACK_DEPTH;
    }

    long getCaptureAnrIntervalMs() {
        return anrConfig != null ? anrConfig.getIntervalMs() : AnrConfig.DEFAULT_ANR_INTERVAL_MS;
    }

    boolean captureMainThreadOnly() {
        return anrConfig != null ? anrConfig.isMainThreadOnly() : AnrConfig.DEFAULT_ANR_MAIN_THREAD_ONLY;
    }

    boolean captureEarlyAnrs() {
        return anrConfig != null ? anrConfig.isAllowEarlyCapture() : AnrConfig.DEFAULT_ALLOW_EARLY_CAPTURE;
    }

    int getAnrThreadCapturePriority() {
        return anrConfig != null ? anrConfig.getPriority() : AnrConfig.DEFAULT_ANR_MIN_THREAD_PRIORITY_TO_CAPTURE;
    }

    int getAnrStacktraceMinimumDuration() {
        return anrConfig != null ? anrConfig.getMinDuration() : AnrConfig.DEFAULT_ANR_MIN_CAPTURE_DURATION;
    }

    AnrConfig getAnrConfig() {
        return anrConfig != null ? anrConfig : AnrConfig.ofDefault();
    }

    KillSwitchConfig getKillSwitchConfig() {
        return killSwitchConfig != null ? killSwitchConfig : new KillSwitchConfig();
    }

    StartupSamplingConfig getStartupSamplingConfig() {
        return startupSamplingConfig != null ? startupSamplingConfig : StartupSamplingConfig.ofDefault();
    }

    public boolean isStartupSamplingEnabled() {
        if (stacktraceSamplingEnabled == null) {
            int k = random.nextInt(100) + 1;
            int pctEnabled = getStartupSamplingConfig().getPctEnabled();
            stacktraceSamplingEnabled = k <= pctEnabled;
        }
        return stacktraceSamplingEnabled;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Config config = (Config) o;

        if (threshold != null ? !threshold.equals(config.threshold) : config.threshold != null) {
            return false;
        }
        if (offset != null ? !offset.equals(config.offset) : config.offset != null) {
            return false;
        }
        if (screenshotsEnabled != null ? !screenshotsEnabled.equals(config.screenshotsEnabled) : config.screenshotsEnabled != null) {
            return false;
        }
        if (eventLimits != null ? !eventLimits.equals(config.eventLimits) : config.eventLimits != null) {
            return false;
        }
        if (disabledMessageTypes != null ? !disabledMessageTypes.equals(config.disabledMessageTypes) : config.disabledMessageTypes != null) {
            return false;
        }
        if (disabledEventAndLogPatterns != null ? !disabledEventAndLogPatterns.equals(config.disabledEventAndLogPatterns) : config.disabledEventAndLogPatterns != null) {
            return false;
        }
        if (disabledUrlPatterns != null ? !disabledUrlPatterns.equals(config.disabledUrlPatterns) : config.disabledUrlPatterns != null) {
            return false;
        }
        if (disabledScreenshotPatterns != null ? !disabledScreenshotPatterns.equals(config.disabledScreenshotPatterns) : config.disabledScreenshotPatterns != null) {
            return false;
        }
        if (uiConfig != null ? !uiConfig.equals(config.uiConfig) : config.uiConfig != null) {
            return false;
        }
        if (sessionControl != null ? !sessionControl.equals(config.sessionControl) : config.sessionControl != null) {
            return false;
        }
        if (networkConfig != null ? !networkConfig.equals(config.networkConfig) : config.networkConfig != null) {
            return false;
        }
        if (logConfig != null ? !logConfig.equals(config.logConfig) : config.logConfig != null) {
            return false;
        }
        if (pctBetaFeaturesEnabled != null ? !pctBetaFeaturesEnabled.equals(config.pctBetaFeaturesEnabled) : config.pctBetaFeaturesEnabled != null) {
            return false;
        }
        return anrConfig != null ? anrConfig.equals(config.anrConfig) : config.anrConfig == null;
    }

    @Override
    public int hashCode() {
        int result = threshold != null ? threshold.hashCode() : 0;
        result = 31 * result + (offset != null ? offset.hashCode() : 0);
        result = 31 * result + (screenshotsEnabled != null ? screenshotsEnabled.hashCode() : 0);
        result = 31 * result + (eventLimits != null ? eventLimits.hashCode() : 0);
        result = 31 * result + (disabledMessageTypes != null ? disabledMessageTypes.hashCode() : 0);
        result = 31 * result + (disabledEventAndLogPatterns != null ? disabledEventAndLogPatterns.hashCode() : 0);
        result = 31 * result + (disabledUrlPatterns != null ? disabledUrlPatterns.hashCode() : 0);
        result = 31 * result + (disabledScreenshotPatterns != null ? disabledScreenshotPatterns.hashCode() : 0);
        result = 31 * result + (uiConfig != null ? uiConfig.hashCode() : 0);
        result = 31 * result + (networkConfig != null ? networkConfig.hashCode() : 0);
        result = 31 * result + (logConfig != null ? logConfig.hashCode() : 0);
        result = 31 * result + (sessionControl != null ? sessionControl.hashCode() : 0);
        result = 31 * result + (logConfig != null ? logConfig.hashCode() : 0);
        result = 31 * result + (anrConfig != null ? anrConfig.hashCode() : 0);
        result = 31 * result + (pctBetaFeaturesEnabled != null ? pctBetaFeaturesEnabled.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "Config{" +
                "threshold=" + threshold +
                ", offset=" + offset +
                ", screenshotsEnabled=" + screenshotsEnabled +
                ", eventLimits=" + eventLimits +
                ", disabledMessageTypes=" + disabledMessageTypes +
                ", disabledEventAndLogPatterns=" + disabledEventAndLogPatterns +
                ", disabledUrlPatterns=" + disabledUrlPatterns +
                ", disabledScreenshotPatterns=" + disabledScreenshotPatterns +
                ", uiConfig=" + uiConfig +
                ", networkConfig=" + networkConfig +
                ", logConfig=" + logConfig +
                ", sessionControl=" + sessionControl +
                ", anrConfig" + anrConfig +
                ", pctBetaFeaturesEnabled" + pctBetaFeaturesEnabled +
                '}';
    }

    /**
     * Configuration values relating to the user interface of the app.
     */
    static class UiConfig {

        /**
         * The default breadcrumbs capture limit.
         */
        private static final int DEFAULT_BREADCRUMB_LIMIT = 100;

        /**
         * The maximum number of custom breadcrumbs to send per session.
         */
        private final Integer breadcrumbs;

        private final Integer fragments;

        private final Integer taps;

        private final Integer views;

        @SerializedName("web_views")
        private final Integer webViews;

        UiConfig(Integer breadcrumbs, Integer taps, Integer views, Integer webViews, Integer fragments) {
            this.breadcrumbs = breadcrumbs;
            this.taps = taps;
            this.views = views;
            this.webViews = webViews;
            this.fragments = fragments;
        }

        public int getBreadcrumbs() {
            return breadcrumbs != null? breadcrumbs : DEFAULT_BREADCRUMB_LIMIT;
        }

        public Optional<Integer> getFragments() {
            return Optional.fromNullable(fragments);
        }

        public Optional<Integer> getTaps() {
            return Optional.fromNullable(taps);
        }

        public Optional<Integer> getViews() {
            return Optional.fromNullable(views);
        }

        public Optional<Integer> getWebViews() {
            return Optional.fromNullable(webViews);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            UiConfig uiConfig = (UiConfig) o;

            if (breadcrumbs != null ? !breadcrumbs.equals(uiConfig.breadcrumbs) : uiConfig.breadcrumbs != null) {
                return false;
            }
            if (fragments != null ? !fragments.equals(uiConfig.fragments) : uiConfig.fragments != null) {
                return false;
            }
            if (taps != null ? !taps.equals(uiConfig.taps) : uiConfig.taps != null) {
                return false;
            }
            if (views != null ? !views.equals(uiConfig.views) : uiConfig.views != null) {
                return false;
            }
            return webViews != null ? webViews.equals(uiConfig.webViews) : uiConfig.webViews == null;
        }

        @Override
        public int hashCode() {
            int result = breadcrumbs != null ? breadcrumbs.hashCode() : 0;
            result = 31 * result + (fragments != null ? fragments.hashCode() : 0);
            result = 31 * result + (taps != null ? taps.hashCode() : 0);
            result = 31 * result + (views != null ? views.hashCode() : 0);
            result = 31 * result + (webViews != null ? webViews.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "UiConfig{" +
                    "breadcrumbs=" + breadcrumbs +
                    ", fragments=" + fragments +
                    ", taps=" + taps +
                    ", views=" + views +
                    ", webViews=" + webViews +
                    '}';
        }
    }

    /**
     * Configures limit of number of requests for network calls per domain.
     * <p>
     * If the default capture limit is specified as zero, then the config operates in allow-list
     * mode, meaning only specified domains will be tracked.
     */
    static class NetworkConfig {
        /**
         * The default request capture limit for non-specified domains.
         */
        private final Integer defaultCaptureLimit;

        /**
         * Map of domain suffix to maximum number of requests.
         */
        @SerializedName("domains")
        private final Map<String, Integer> domainLimits;

        NetworkConfig(Integer defaultCaptureLimit, Map<String, Integer> domainLimits) {
            this.defaultCaptureLimit = defaultCaptureLimit;
            this.domainLimits = domainLimits;
        }

        Map<String, Integer> getDomainLimits() {
            return domainLimits == null ? new HashMap<>() : domainLimits;
        }

        Optional<Integer> getDefaultCaptureLimit() {
            return Optional.fromNullable(defaultCaptureLimit);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            NetworkConfig that = (NetworkConfig) o;

            if (defaultCaptureLimit != null ? !defaultCaptureLimit.equals(that.defaultCaptureLimit) : that.defaultCaptureLimit != null) {
                return false;
            }
            return domainLimits != null ? domainLimits.equals(that.domainLimits) : that.domainLimits == null;
        }

        @Override
        public int hashCode() {
            int result = defaultCaptureLimit != null ? defaultCaptureLimit.hashCode() : 0;
            result = 31 * result + (domainLimits != null ? domainLimits.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "NetworkConfig{" +
                    "defaultCaptureLimit=" + defaultCaptureLimit +
                    ", domainLimits=" + domainLimits +
                    '}';
        }
    }

    /**
     * Configuration values relating to the logs of the app.
     */
    static class LogConfig {

        /**
         * Used to truncate log messages.
         */
        @SerializedName("max_length")
        private final Integer logMessageMaximumAllowedLength;

        /**
         * Limit of info logs that user is able to send.
         */
        @SerializedName("info_limit")
        private final Integer logInfoLimit;

        /**
         * Limit of warning logs that user is able to send.
         */
        @SerializedName("warn_limit")
        private final Integer logWarnLimit;

        /**
         * Limit of error logs that user is able to send.
         */
        @SerializedName("error_limit")
        private final Integer logErrorLimit;

        LogConfig(Integer logMessageMaximumAllowedLength, Integer logInfoLimit, Integer logWarnLimit, Integer logErrorLimit) {
            this.logMessageMaximumAllowedLength = logMessageMaximumAllowedLength;
            this.logInfoLimit = logInfoLimit;
            this.logWarnLimit = logWarnLimit;
            this.logErrorLimit = logErrorLimit;
        }

        Optional<Integer> getLogMessageMaximumAllowedLength() {
            return Optional.fromNullable(logMessageMaximumAllowedLength);
        }

        Optional<Integer> getLogInfoLimit() {
            return Optional.fromNullable(logInfoLimit);
        }

        Optional<Integer> getLogWarnLimit() {
            return Optional.fromNullable(logWarnLimit);
        }

        Optional<Integer> getLogErrorLimit() {
            return Optional.fromNullable(logErrorLimit);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            LogConfig that = (LogConfig) o;

            if (logMessageMaximumAllowedLength != null ? !logMessageMaximumAllowedLength.equals(that.logMessageMaximumAllowedLength) : that.logMessageMaximumAllowedLength != null) {
                return false;
            }
            if (logInfoLimit != null ? !logInfoLimit.equals(that.logInfoLimit) : that.logInfoLimit != null) {
                return false;
            }
            if (logWarnLimit != null ? !logWarnLimit.equals(that.logWarnLimit) : that.logWarnLimit != null) {
                return false;
            }
            return logErrorLimit != null ? logErrorLimit.equals(that.logErrorLimit) : that.logErrorLimit == null;
        }

        @Override
        public int hashCode() {
            int result = logMessageMaximumAllowedLength != null ? logMessageMaximumAllowedLength.hashCode() : 0;
            result = 31 * result + (logInfoLimit != null ? logInfoLimit.hashCode() : 0);
            result = 31 * result + (logWarnLimit != null ? logWarnLimit.hashCode() : 0);
            result = 31 * result + (logErrorLimit != null ? logErrorLimit.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "LogConfig{" +
                    ", logMessageMaximumAllowedLength=" + logMessageMaximumAllowedLength +
                    ", logInfoLimit=" + logInfoLimit +
                    ", logWarnLimit=" + logWarnLimit +
                    ", logErrorLimit=" + logErrorLimit +
                    '}';
        }
    }

    /**
     * It serves as a session controller components. It determines if session may be ended in
     * the background. It also determines which components will be sent as part of the
     * session payload. This feature may be enabled/disabled.
     */
    static class SessionControl {

        @SerializedName("enable")
        boolean enabled;

        @SerializedName("async_end")
        boolean endAsync;

        /**
         * A list of session components (i.e. Breadcrumbs, Session properties, etc) that will be
         * included in the session payload. If components list exists, the services should restrict
         * the data that is provided to the session.
         */
        @SerializedName("components")
        Set<String> components;

        /**
         * A list of session components allowed to send a full session payload (only if "components"
         * exists)
         */
        @SerializedName("send_full_for")
        Set<String> fullSessionEvents;

        SessionControl(boolean enabled, boolean endAsync, Set<String> components, Set<String> fullSessionEvents) {
            this.enabled = enabled;
            this.endAsync = endAsync;
            this.components = components;
            this.fullSessionEvents = fullSessionEvents;
        }

        boolean isEnabled() {
            return enabled;
        }

        public boolean endSessionInBackgroundThread() {
            return endAsync;
        }

        /**
         * Determines if the gating feature is enabled based on the presence of the session
         * components list property.
         *
         * @return true if the gating feature is enabled
         */
        public boolean isGatingFeatureEnabled() {
            return components != null;
        }

        /**
         * The whitelist of session components that should be included in the session payload.
         *
         * @return the whitelist of session components
         */
        public Set<String> getSessionComponents() {
            return components;
        }

        /**
         * The whitelist of events (crashes, errors) that should send a full session payload even
         * if the gating feature is enabled.
         *
         * @return a whitelist of events allowed to send full session payloads
         */
        public Set<String> getFullSessionEvents() {
            return fullSessionEvents == null? new HashSet<>(): fullSessionEvents;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            SessionControl that = (SessionControl) o;

            if(enabled != that.enabled) {
                return false;
            }

            if(endAsync != that.endAsync) {
                return false;
            }

            if (components != null ? !components.equals(that.components) : that.components != null) {
                return false;
            }
            return fullSessionEvents != null ? fullSessionEvents.equals(that.fullSessionEvents) : that.fullSessionEvents == null;
        }

        @Override
        public int hashCode() {
            int result = enabled ? 1 : 0;
            result = 31 * result + (endAsync ? 1 : 0);
            result = 31 * result + (components != null ? components.hashCode() : 0);
            result = 31 * result + (fullSessionEvents != null ? fullSessionEvents.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "SessionControlConfig{" +
                    ", enabled=" + enabled +
                    ", endAsync=" + endAsync +
                    ", components=" + components +
                    ", fullSessionEnabled=" + fullSessionEvents +
                    '}';
        }
    }


    static class StartupSamplingConfig {
        static final int DEFAULT_PCT_ENABLED = 100;
        static final int DEFAULT_STARTUP_SAMPLE_GRANULARITY_MS = 100;
        static final int DEFAULT_STARTUP_SAMPLE_INTERVAL_MS = 1000;
        static final int DEFAULT_STARTUP_SAMPLING_DURATION_MS = 10000;
        static final int DEFAULT_STARTUP_SAMPLING_MAX_STACKTRACE_LENGTH = 100;

        static final String PCT_ENABLED = "pct_enabled";
        static final String SAMPLE_GRANULARITY = "sample_granularity";
        static final String SAMPLE_INTERVAL = "sample_interval";
        static final String SAMPLING_DURATION = "sampling_duration";
        static final String STACKTRACE_LENGTH = "stacktrace_length";

        /**
         * Percentage of users for which ANR stack trace capture is enabled.
         */
        @SerializedName(PCT_ENABLED)
        private final Integer pctEnabled;

        @SerializedName(SAMPLE_GRANULARITY)
        private final Integer sampleGranularity;

        @SerializedName(SAMPLE_INTERVAL)
        private final Integer sampleInterval;

        @SerializedName(SAMPLING_DURATION)
        private final Integer samplingDuration;

        @SerializedName(STACKTRACE_LENGTH)
        private final Integer maxStacktraceLength;


        StartupSamplingConfig(Integer pctEnabled, Integer sampleInterval, Integer sampleGranularity,
                              Integer samplingDuration, Integer maxStacktraceLength) {
            this.maxStacktraceLength = maxStacktraceLength;
            this.pctEnabled = pctEnabled;
            this.sampleGranularity = sampleGranularity;
            this.sampleInterval = sampleInterval;
            this.samplingDuration = samplingDuration;
        }

        static StartupSamplingConfig ofDefault() {
            return new StartupSamplingConfig(null, null, null, null, null);
        }

        public int getMaxStacktraceLength() {
            return Optional.fromNullable(maxStacktraceLength).or(DEFAULT_STARTUP_SAMPLING_MAX_STACKTRACE_LENGTH);
        }

        public int getPctEnabled() {
            return Optional.fromNullable(pctEnabled).or(DEFAULT_PCT_ENABLED);
        }

        public int getSampleGranularity() {
            return Optional.fromNullable(sampleGranularity).or(DEFAULT_STARTUP_SAMPLE_GRANULARITY_MS);
        }

        public int getSamplingDuration() {
            return Optional.fromNullable(samplingDuration).or(DEFAULT_STARTUP_SAMPLING_DURATION_MS);
        }

        public int getSampleInterval() {
            return Optional.fromNullable(sampleInterval).or(DEFAULT_STARTUP_SAMPLE_INTERVAL_MS);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            StartupSamplingConfig startupSamplingConfig = (StartupSamplingConfig) o;

            if (pctEnabled != null ? !pctEnabled.equals(startupSamplingConfig.pctEnabled) : startupSamplingConfig.pctEnabled != null) {
                return false;
            }
            if (sampleGranularity != null ? !sampleGranularity.equals(startupSamplingConfig.sampleGranularity) : startupSamplingConfig.sampleGranularity != null) {
                return false;
            }
            if (sampleInterval != null ? !sampleInterval.equals(startupSamplingConfig.sampleInterval) : startupSamplingConfig.sampleInterval != null) {
                return false;
            }
            if (samplingDuration != null ? !samplingDuration.equals(startupSamplingConfig.samplingDuration) : startupSamplingConfig.samplingDuration != null) {
                return false;
            }
            return maxStacktraceLength != null ? maxStacktraceLength.equals(startupSamplingConfig.maxStacktraceLength) : startupSamplingConfig.maxStacktraceLength == null;
        }

        @Override
        public int hashCode() {
            int result = maxStacktraceLength != null ? maxStacktraceLength.hashCode() : 0;
            result = 31 * result + (pctEnabled != null ? pctEnabled.hashCode() : 0);
            result = 31 * result + (sampleGranularity != null ? sampleGranularity.hashCode() : 0);
            result = 31 * result + (sampleInterval != null ? sampleInterval.hashCode() : 0);
            result = 31 * result + (samplingDuration != null ? samplingDuration.hashCode() : 0);
            return result;
        }
    }
}
