package io.embrace.android.embracesdk;

import android.text.TextUtils;

import androidx.annotation.VisibleForTesting;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;

/**
 * Base class for creating custom domain-specific flows that are essentially convenience wrappers around existing SDK
 * functionality.
 */
abstract class CustomFlow {

    protected static final String PROP_MOMENT_ID = "moment-id";
    protected static final String PROP_MOMENT_NAME = "moment-name";
    protected static final String PROP_MESSAGE = "message";

    private static final String ERROR_BLANK_NAME = "Moment name is null or blank.";

    protected final Map<String, Map<String, Moment>> customMoments = new HashMap<>();

    @VisibleForTesting
    Set<String> getCustomMomentInstances(String customMomentName) {
        if (customMoments.get(customMomentName) != null) {
            return Objects.requireNonNull(customMoments.get(customMomentName)).keySet();
        } else {
            return null;
        }
    }


    private final EventService eventService;
    private final EmbraceRemoteLogger embraceRemoteLogger;

    protected CustomFlow() {
        this.eventService = Embrace.getInstance().getEventService();
        this.embraceRemoteLogger = Embrace.getInstance().getRemoteLogger();
    }

    /**
     * Starts a custom moment.
     *
     * @param momentName          The name of the moment.
     * @param doesAllowScreenshot If true, a screenshot will be taken if the moment exceeds the late threshold. If this
     *                            value is false, a screenshot will be not be taken regardless of the moment duration.
     * @param properties          A map of Strings to Objects that represent additional properties to associate with the moment.
     *                            This value is optional. A maximum of 10 properties may be set.
     * @return A moment identifier that uniquely identifies the newly started moment instance.
     */
    public String momentStart(String momentName, boolean doesAllowScreenshot, Map<String, Object> properties) {

        if (!validateMomentName(momentName))
            return null;

        if (customMoments.get(momentName) == null) {
            customMoments.put(momentName, new HashMap<>());
        }
        String momentId = Uuid.getEmbUuid();
        Map<String, Moment> moments = this.customMoments.get(momentName);

        sendMomentStartEvent(momentName, momentId, doesAllowScreenshot, PropertyUtils.sanitizeProperties(properties));

        // Register the custom moment instance so that it can be accessed later.
        moments.put(momentId, new Moment(momentName, momentId, doesAllowScreenshot, properties));
        return momentId;
    }

    private boolean validateMomentName(String momentName) {
        if (TextUtils.isEmpty(momentName)) {
            sendLogError(ERROR_BLANK_NAME, false, null);
            return false;
        } else if (momentName.startsWith("_")) {
            sendLogError("Moment name may not start with '_'.", false, null);
            return false;
        }

        return true;
    }

    /**
     * Completes all started instances of the specified custom moment.
     * <p>
     * Note that only moment instances managed by this Flow object will be completed. In other words, if another Flow
     * instance starts a moment with the same name, completing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentComplete(String momentName) {
        return momentComplete(momentName, null);
    }

    /**
     * Completes a started instance of the custom moment specified by the moment identifier.
     * <p>
     * Note that only moment instances managed by this Flow object will be completed. In other words, if another Flow
     * instance starts a moment with the same name, completing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentComplete(String momentName, String momentId) {
        return momentComplete(momentName, momentId, null);
    }


    /**
     * Completes a started instance of the custom moment specified by the moment identifier.
     * <p>
     * Note that only moment instances managed by this Flow object will be completed. In other words, if another Flow
     * instance starts a moment with the same name, completing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @param properties Custom properties to pass in with the moment.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentComplete(String momentName, String momentId, Map<String, Object> properties) {

        if (TextUtils.isEmpty(momentName)) {
            sendLogError(ERROR_BLANK_NAME, false, null);
            return false;
        }

        // Get all the moment instances for the requested moment.
        Map<String, Moment> moments = this.customMoments.get(momentName);

        if (moments == null) {
            InternalStaticEmbraceLogger.logError("Cannot fail " + momentName + " because moment name is not recognized.");
            return false;
        }

        if (momentId == null) {
            // Complete all known moment instances for the requested moment.
            for (Map.Entry<String, Moment> entry : moments.entrySet()) {
                sendMomentEndEvent(momentName, entry.getKey(), properties);
            }
            moments.clear();

        } else {
            // Complete only the specific moment instance that was requested.
            Moment moment = moments.get(momentId);
            if (moment == null) {
                InternalStaticEmbraceLogger.logError("Cannot fail " + momentName + " because moment identifier is not recognized.");
                return false;
            }
            sendMomentEndEvent(momentName, momentId, properties);
            moments.remove(momentId);
        }

        if (moments.isEmpty()) {
            // No instances of the moment name is running so remove the entire entry in the custom moment map.
            this.customMoments.remove(momentName);
        }

        return true;
    }

    /**
     * Fails all started instances of the specified custom moment and generates an error log message for each failed
     * moment instance.
     * <p>
     * Note that only moment instances managed by this Flow object will be failed. In other words, if another Flow
     * instance fails a moment with the same name, failing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentFail(String momentName, String msg) {
        return momentFail(momentName, null, msg);
    }

    /**
     * Fails a started instance of the custom moment specified by the moment identifier and sends an error log message for
     * the failed moment instance.
     * <p>
     * Note that only moment instances managed by this Flow object will be failed. In other words, if another Flow
     * instance fails a moment with the same name, failing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @param properties Custom properties to pass in with the moments.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentFail(String momentName, String momentId, String msg, Map<String, Object> properties) {

        if (!isValidMoment(momentName)) {
            return false;
        }

        // Get all the moment instances for the requested moment.
        Map<String, Moment> moments = this.customMoments.get(momentName);

        if (moments == null) {
            InternalStaticEmbraceLogger.logError("Cannot fail " + momentName + " because moment name is not recognized.");
            return false;
        }

        String errorLogMsg = getErrorLogMessage(msg, momentName);

        if (momentId == null) {
            // Fail all known moment instances for the requested moment.
            failsAllMomentInstances(moments, momentName, msg, properties, errorLogMsg);
        } else {

            // Fail all known moment instances for the requested moment.
            if (!isValidMomentId(moments, momentId, momentName)) {
                return false;
            }

            failsSingleMomentInstances(moments, moments.get(momentId), momentName, msg, properties, errorLogMsg);
        }

        if (moments.isEmpty()) {
            // No instances of the moment name is running so remove the entire entry in the custom moment map.
            this.customMoments.remove(momentName);
        }

        return true;
    }

    private boolean isValidMomentId(Map<String, Moment> moments, String momentId, String momentName) {
        if (momentId != null && moments.get(momentId) == null) {
            InternalStaticEmbraceLogger.logError("Cannot fail " + momentName + " because moment identifier is not recognized.");
            return false;
        }

        return true;
    }

    private boolean isValidMoment(String momentName) {
        if (TextUtils.isEmpty(momentName)) {
            sendLogError(ERROR_BLANK_NAME, false, null);
            return false;
        } else return true;
    }


    private String getErrorLogMessage(String msg, String momentName) {
        String errorLogMsg;

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = "A failure occurred during the " + momentName + " moment.";
        } else {
            errorLogMsg = "A failure occurred during the " + momentName + " moment: " + msg;
        }

        return errorLogMsg;
    }


    private void failsAllMomentInstances(Map<String, Moment> moments, String momentName, String msg, Map<String, Object> properties, String errorLogMsg) {
        for (Map.Entry<String, Moment> entry : moments.entrySet()) {
            momentInstanceFail(moments, entry.getValue(), momentName, msg, properties, errorLogMsg);
        }
        moments.clear();
    }

    private void failsSingleMomentInstances(Map<String, Moment> moments, Moment moment, String momentName, String msg, Map<String, Object> properties, String errorLogMsg) {
        momentInstanceFail(moments, moment, momentName, msg, properties, errorLogMsg);
        moments.remove(moment.id);
    }

    // Fail only the specific moment instance that was requested.
    private void momentInstanceFail(Map<String, Moment> moments, Moment moment, String momentName, String msg, Map<String, Object> properties, String errorLogMsg) {
        addMessage(moment, msg);
        sendMomentEndEvent(momentName, moment.id, properties);
        sendLogError(errorLogMsg, moment.doesAllowScreenshot, moment.properties);

    }

    private void addMessage(Moment moment, String msg) {
        if (msg != null) {
            if (moment.properties == null) {
                moment.properties = new HashMap<>();
            }
            moment.properties.put(PROP_MESSAGE, msg);
        }
    }

    /**
     * Fails a started instance of the custom moment specified by the moment identifier and sends an error log message for
     * the failed moment instance.
     * <p>
     * Note that only moment instances managed by this Flow object will be failed. In other words, if another Flow
     * instance fails a moment with the same name, failing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentFail(String momentName, String momentId, String msg) {
        return momentFail(momentName, momentId, msg, null);
    }

    protected void sendMomentStartEvent(String momentName,
                                        String momentId,
                                        boolean doesAllowScreenshot,
                                        Map<String, Object> properties) {
        try {
            if (this.eventService != null) {
                this.eventService.startEvent(momentName, momentId, doesAllowScreenshot, properties);
            } else {
                throw new Exception("Event service is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            InternalStaticEmbraceLogger.logError("An error occurred trying to start moment: " + momentName + " - " + momentId + ".", e);
        }
    }

    protected void sendMomentEndEvent(String momentName,
                                      String momentId,
                                      Map<String, Object> properties) {

        try {
            if (this.eventService != null) {
                this.eventService.endEvent(momentName, momentId, properties);
            } else {
                throw new Exception("Event service is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            InternalStaticEmbraceLogger.logError("An error occurred trying to end moment: " + momentName + " - " + momentId + ".", e);
        }
    }

    protected void sendLogInfo(String msg,
                               Map<String, Object> properties) {

        try {
            if (this.embraceRemoteLogger != null) {
                this.embraceRemoteLogger.log(msg, EmbraceEvent.Type.INFO_LOG, false, properties);
            } else {
                throw new Exception("Remote Logger is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            InternalStaticEmbraceLogger.logError("An error occurred sending log info message: " + msg + ".", e);
        }
    }

    protected void sendLogError(String msg,
                                boolean doesAllowScreenshot,
                                Map<String, Object> properties) {

        try {
            if (this.embraceRemoteLogger != null) {
                this.embraceRemoteLogger.log(msg, EmbraceEvent.Type.ERROR_LOG, doesAllowScreenshot, properties);
            } else {
                throw new Exception("Remote Logger is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            InternalStaticEmbraceLogger.logError("An error occurred sending log error message: " + msg + ".", e);
        }
    }

    private static class Moment {
        final String name;
        final String id;
        final boolean doesAllowScreenshot;
        Map<String, Object> properties;

        Moment(String name, String id, boolean doesAllowScreenshot, Map<String, Object> properties) {
            this.name = name;
            this.id = id;
            this.doesAllowScreenshot = doesAllowScreenshot;
            this.properties = properties != null ? new HashMap<>(properties) : null;
        }
    }
}
