package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.EmbraceEventService.STARTUP_EVENT_NAME;

import android.app.Application;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.preference.PreferenceManager;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import io.embrace.android.embracesdk.logging.AndroidLogger;
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequest;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequestV2;
import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.utils.Preconditions;
import kotlin.Lazy;
import kotlin.LazyKt;

/**
 * Entry point for the SDK. This class is part of the Embrace Public API.
 * <p>
 * Contains a singleton instance of itself, and is used for initializing the SDK.
 */
public final class Embrace {

    private static final String ERROR_USER_UPDATES_DISABLED = "User updates are disabled, ignoring user persona update.";
    /**
     * Singleton instance of the Embrace SDK.
     */
    private static Embrace embrace = new Embrace();

    /**
     * Singleton instance of the AnrService. Only initialized if a user started the ANR service
     * before the rest of the SDK.
     */
    private static volatile EmbraceEarlyAnrInitializer earlyAnrInitializer;

    /**
     * Singleton instance of the startup tracing service. Only initialized if a user started the
     * service manually.
     */
    @Nullable
    private static StartupTracingService startupTracingService;

    /**
     * Whether the Embrace SDK has been started yet.
     */
    private final AtomicBoolean started = new AtomicBoolean(false);
    /**
     * Custom app ID that overrides the one specified at build time
     */
    private static String customAppId;
    /**
     * The application being instrumented by the SDK.
     */
    private volatile Application application;
    /**
     * The application framework. It should be Android Native, Unity or React Native
     */
    private volatile AppFramework appFramework;
    /**
     * The clock used with the SDK.
     */
    private volatile Clock clock;
    /**
     * The object caching service.
     */
    private volatile CacheService cacheService;
    /**
     * The crash handling service.
     */
    private volatile CrashService crashService;
    /**
     * The breadcrumbs service.
     */
    private volatile BreadcrumbService breadcrumbService;
    /**
     * The session handling service.
     */
    private volatile EmbraceSessionService sessionService;
    /**
     * The manager for sessions cache.
     */
    private volatile SessionCacheManager sessionCacheManager;
    /**
     * The device metadata service.
     */
    private volatile MetadataService metadataService;
    /**
     * The device's performance information service.
     */
    private volatile PerformanceInfoService performanceInfoService;
    /**
     * The power service.
     */
    private volatile PowerService powerService;
    /**
     * The memory service;
     */
    private volatile MemoryService memoryService;
    /**
     * The Activity service.
     */
    private volatile EmbraceActivityService activityService;
    /**
     * The Networking service.
     */
    private volatile NetworkConnectivityService networkConnectivityService;
    /**
     * The network call logging service.
     */
    private volatile NetworkLoggingService networkLoggingService;
    /**
     * The Build info data accessor.
     */
    private volatile BuildInfo buildInfo;
    /**
     * The local config info data accessor.
     */
    private volatile LocalConfig localConfig;
    /**
     * The ANR Monitoring service. This is static to allow for customers to initialize it
     * before the main SDK has initialized if they wish to capture early ANRs.
     */
    private volatile AnrService anrService;
    /**
     * The EmbraceRemoteLogger.
     */
    private volatile EmbraceRemoteLogger remoteLogger;
    /**
     * The Configuration service.
     */
    private volatile ConfigService configService;
    /**
     * The Embrace prefences service.
     */
    private volatile PreferencesService preferencesService;
    /**
     * The Embrace screenshot service.
     */
    private volatile ScreenshotService screenshotService;
    /**
     * The Embrace event service.
     */
    private volatile EventService eventService;
    /**
     * The User service.
     */
    private volatile UserService userService;
    /**
     * The Embrace exception class service.
     */
    private volatile EmbraceExceptionService exceptionsService;
    /**
     * The Embrace memory cleaner class service.
     */
    private volatile MemoryCleanerService memoryCleanerService;
    /**
     * The Embrace orientation class service.
     */
    private volatile OrientationService orientationService;
    /**
     * The Embrace NDK class service.
     */
    private volatile NdkService ndkService;
    /**
     * The Embrace gating service.
     */
    private volatile GatingService gatingService;

    /**
     * Tracks the thermal status of the device.
     */
    private volatile EmbraceThermalStatusService thermalStatusService;

    /**
     * The Internal Embrace Logger.
     */
    private final InternalEmbraceLogger internalEmbraceLogger = InternalStaticEmbraceLogger.getLogger();
    /**
     * Worker used by services to execute tasks in background
     */
    private volatile BackgroundWorker bgRegistrationWorker;
    /**
     * Worker used by services to schedule tasks
     */
    private volatile ScheduledWorker scheduledWorker;

    private volatile GoogleAnrTimestampRepository googleAnrTimestampRepository;

    /**
     * Gets the singleton instance of the Embrace SDK.
     *
     * @return the instance of the Embrace SDK
     */
    @NonNull
    public static Embrace getInstance() {
        return embrace;
    }

    static void setInstance(@Nullable Embrace instance) {
        embrace = instance;
    }

    /**
     * Capture stacktraces during startup to better understand what is happening during startup.
     * <p>
     * This API is experimental and subject to change.
     */
    public static void enableStartupTracing(@NonNull Context context) {
        if (embrace.isStarted()) {
            InternalStaticEmbraceLogger.logError("You must enable startup tracing before the SDK is started.");
            return;
        }

        try {
            Clock clock = new NormalizedIntervalClock(new SystemClock());
            InternalEmbraceLogger logger = InternalStaticEmbraceLogger.getLogger();
            Lazy<CacheService> cacheService = LazyKt.lazy(() -> new EmbraceCacheService(context, logger));
            startupTracingService = new EmbraceStartupTracingService(clock, cacheService, logger);
        } catch (Throwable exc) {
            InternalStaticEmbraceLogger.logDebug("Failed to initialize startup tracing service.", exc);
        }
    }

    /**
     * Starts capturing ANRs before the rest of the Embrace SDK has initialized. This allows
     * you to get insight into what is causing ANRs early on in an application.
     * <p>
     * This API is experimental and subject to change.
     */
    public static void enableEarlyAnrCapture(@NonNull Context context) {
        if (embrace.isStarted()) {
            InternalStaticEmbraceLogger.logError("You must enable early ANR capture before the SDK is started.");
            return;
        }
        if (earlyAnrInitializer != null) {
            return; // initialize once if this method is unintentionally called multiple times.
        }
        try {
            earlyAnrInitializer = new EmbraceEarlyAnrInitializer(context);
        } catch (Throwable exc) {
            InternalStaticEmbraceLogger.logWarning("Failed to initialize early " +
                    "ANR capture. Falling back to regular ANR initialization.", exc);
        }
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context an instance of the application context
     */
    public void start(@NonNull Context context) {
        start(context, true, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context                  an instance of context
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(@NonNull Context context, boolean enableIntegrationTesting) {
        start(context, enableIntegrationTesting, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context                  an instance of context
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(@NonNull Context context, boolean enableIntegrationTesting, @NonNull AppFramework appFramework) {
        if (this.application != null) {
            // We don't hard fail if the SDK has been already initialized.
            InternalStaticEmbraceLogger.logWarning("Embrace SDK has already been initialized");
            return;
        }
        if (!(context instanceof Application)) {
            context = context.getApplicationContext();
        }
        this.application = Preconditions.checkNotNull((Application) context, "application must not be null");
        this.appFramework = Preconditions.checkNotNull(appFramework, "Application framework must not be null");
        internalEmbraceLogger.logDeveloper("Embrace", "Starting SDK for framework " + appFramework.name());

        this.clock = new NormalizedIntervalClock(new SystemClock());
        long startTime = this.clock.now();

        // Bring up all services and inject dependencies
        try {
            /* ------------------------------------------------------------------------------------
             *  Device instrumentation (power, memory, CPU, network, preferences, CPU)
             *  */

            bgRegistrationWorker = BackgroundWorker.ofSingleThread("Background Registration Worker");
            scheduledWorker = ScheduledWorker.ofSingleThread("Scheduled Registration Worker");

            this.memoryCleanerService = new EmbraceMemoryCleanerService();
            this.powerService = new EmbracePowerService(context, memoryCleanerService, bgRegistrationWorker, scheduledWorker, clock);

            this.memoryService = EmbraceMemoryService.ofContext(clock, context, memoryCleanerService, scheduledWorker);
            this.orientationService = new EmbraceOrientationService(memoryCleanerService, clock);
            this.activityService = new EmbraceActivityService(application, memoryService, orientationService, clock);
            this.localConfig = LocalConfig.fromResources(context, customAppId);
            // initialize the logger early so that logged exceptions have a good chance of
            // being appended to the exceptions service rather than logcat
            Boolean logStrictMode = localConfig.getConfigurations().sessionConfig.getSessionEnableErrorLogStrictMode();
            this.exceptionsService = new EmbraceExceptionService(activityService, clock, logStrictMode);
            ExceptionServiceLogger action = new ExceptionServiceLogger(exceptionsService, new AndroidLogger(), logStrictMode);
            internalEmbraceLogger.addLoggerAction(action);

            Context finalContext = context;
            Lazy<SharedPreferences> lazyPrefs = LazyKt.lazy(() -> PreferenceManager.getDefaultSharedPreferences(finalContext));
            this.preferencesService = new EmbracePreferencesService(activityService, bgRegistrationWorker, lazyPrefs);
            this.networkConnectivityService = new EmbraceNetworkConnectivityService(context, clock, memoryCleanerService, bgRegistrationWorker, internalEmbraceLogger);
            this.buildInfo = BuildInfo.fromResources(context);

            this.cacheService = new EmbraceCacheService(context, internalEmbraceLogger);
            this.metadataService = EmbraceMetadataService.ofContext(
                    context,
                    buildInfo,
                    localConfig,
                    this.appFramework,
                    preferencesService,
                    activityService,
                    bgRegistrationWorker);

            /* ------------------------------------------------------------------------------------
             *  API services
             *  */
            EmbraceSessionProperties sessionProperties = new EmbraceSessionProperties(
                    preferencesService,
                    internalEmbraceLogger);

            ApiClient apiClient = new ApiClient(
                    localConfig,
                    metadataService,
                    cacheService,
                    enableIntegrationTesting,
                    internalEmbraceLogger);
            this.configService = new EmbraceConfigService(
                    localConfig,
                    apiClient,
                    activityService,
                    cacheService,
                    metadataService,
                    preferencesService,
                    clock,
                    internalEmbraceLogger,
                    bgRegistrationWorker
            );
            if (configService.isSdkDisabled()) {
                internalEmbraceLogger.logDeveloper("Embrace", "the SDK is disabled");
                stop();
                return;
            }
            this.googleAnrTimestampRepository = new GoogleAnrTimestampRepository(internalEmbraceLogger);
            /* *
             * Since onForeground() is called sequential in the order that services registered for it,
             * it is important to initialize the `EmbraceAnrService`, and thus register the `onForeground()
             * listener for it, before the `EmbraceSessionService`.
             * The onForeground() call inside the EmbraceAnrService should be called before the
             * EmbraceSessionService call. This is necessary since the EmbraceAnrService should be able to
             * force a Main thread health check and close the pending ANR intervals that happened on the
             * background before the next session is created.
             * */
            if (earlyAnrInitializer != null) {
                internalEmbraceLogger.logDeveloper("Embrace", "Early ANR initializer started");
                this.anrService = earlyAnrInitializer.getService();
            } else {
                // the customer didn't enable early ANR detection, so construct the service
                // as part of normal initialization.
                internalEmbraceLogger.logDeveloper("Embrace", "Embrace ANR detection started");
                Clock intervalClock = new NormalizedIntervalClock(new SystemClock());
                FilesDelegate filesDelegate = new FilesDelegate();
                this.anrService = new EmbraceAnrService(intervalClock, configService, internalEmbraceLogger,
                        new SharedObjectLoader(),
                        new FindGoogleThread(
                                internalEmbraceLogger,
                                new GetThreadsInCurrentProcess(filesDelegate),
                                new GetThreadCommand(filesDelegate)),
                        new GoogleAnrHandlerNativeDelegate(googleAnrTimestampRepository, internalEmbraceLogger),
                        googleAnrTimestampRepository);
            }

            // set callbacks and pass in non-placeholder config.
            anrService.finishInitialization(
                    memoryCleanerService,
                    activityService,
                    configService
            );

            // install the Unity thread sampler
            SharedObjectLoader sharedObjectLoader = new SharedObjectLoader();
            UnityThreadSamplerService unityThreadSampler = null;

            if (sharedObjectLoader.loadEmbraceNative()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Loaded Embrace Native");
                kotlin.Lazy<Map<String, String>> symbols = LazyKt.lazy(() -> {
                    if (ndkService != null) {
                        internalEmbraceLogger.logDeveloper("Embrace", "Getting NDK symbols");
                        return ndkService.getSymbolsForCurrentArch();
                    } else {
                        internalEmbraceLogger.logDeveloper("Embrace", "NDK Service is null");
                        return null;
                    }
                });
                unityThreadSampler = new EmbraceUnityThreadSamplerService(configService, clock, symbols);
                new UnityThreadSamplerInstaller().install(
                        unityThreadSampler,
                        configService,
                        anrService
                );
            } else {
                internalEmbraceLogger.logDeveloper("Embrace", "Failed to load SO file embrace-native");
            }

            this.gatingService = new EmbraceGatingService(
                    internalEmbraceLogger,
                    localConfig,
                    configService);
            this.exceptionsService.setConfigService(this.configService);
            this.breadcrumbService = new EmbraceBreadcrumbService(
                    clock,
                    configService,
                    localConfig,
                    activityService,
                    memoryCleanerService,
                    internalEmbraceLogger
            );
            this.sessionCacheManager = new SessionCacheManager(
                    localConfig,
                    cacheService,
                    apiClient,
                    internalEmbraceLogger);

            /*
             * If the last session did not end cleanly, we expect to have a cached version of the
             * session saved. We need to move this to a new location for the following reasons:
             *
             * 1. When we start the session service we will start writing the current session to
             *    this location.
             * 2. The NDK service needs access to this cached session when it is started since it
             *    may need to insert a crash ID into it if an NDK crash occurred. the session
             *    needs to be in a predictable place when the NDK service tries to load it.
             * 3. The session service will send this session when the app is foregrounded.
             *
             * Thus we need this move to occur before either the NDK or session services are
             * started.
             */
            sessionCacheManager.stashPreviousSession();

            this.userService = new EmbraceUserService(
                    preferencesService,
                    internalEmbraceLogger);
            this.screenshotService = new EmbraceScreenshotService(
                    activityService,
                    configService,
                    apiClient,
                    internalEmbraceLogger);
            this.networkLoggingService = new EmbraceNetworkLoggingService(
                    configService,
                    localConfig,
                    memoryCleanerService,
                    internalEmbraceLogger);
            this.performanceInfoService = new EmbracePerformanceInfoService(
                    anrService,
                    networkConnectivityService,
                    networkLoggingService,
                    powerService,
                    memoryService,
                    metadataService,
                    googleAnrTimestampRepository);
            this.eventService = new EmbraceEventService(
                    startTime,
                    apiClient,
                    configService,
                    localConfig,
                    metadataService,
                    performanceInfoService,
                    userService,
                    screenshotService,
                    activityService,
                    memoryCleanerService,
                    gatingService,
                    sessionProperties,
                    internalEmbraceLogger,
                    clock);
            this.remoteLogger = new EmbraceRemoteLogger(
                    metadataService,
                    screenshotService,
                    apiClient,
                    userService,
                    configService,
                    memoryCleanerService,
                    sessionProperties,
                    internalEmbraceLogger,
                    clock,
                    gatingService);
            this.ndkService = new EmbraceNdkService(
                    context,
                    metadataService,
                    activityService,
                    configService,
                    localConfig,
                    apiClient,
                    userService,
                    sessionProperties,
                    this.appFramework,
                    sharedObjectLoader,
                    internalEmbraceLogger,
                    new EmbraceNdkServiceRepository(context, internalEmbraceLogger),
                    new NdkDelegateImpl(),
                    BackgroundWorker.ofSingleThread("Native Crash Cleaner"),
                    BackgroundWorker.ofSingleThread("Native Crash Fetch"),
                    BackgroundWorker.ofSingleThread("Native Startup"));

            SessionHandler sessionHandler = new SessionHandler(
                    internalEmbraceLogger,
                    configService,
                    preferencesService,
                    powerService,
                    userService,
                    networkConnectivityService,
                    metadataService,
                    apiClient,
                    gatingService,
                    localConfig,
                    breadcrumbService,
                    activityService,
                    ndkService,
                    eventService,
                    remoteLogger,
                    exceptionsService,
                    startupTracingService,
                    performanceInfoService,
                    memoryCleanerService,
                    sessionCacheManager,
                    clock
            );
            this.sessionService = new EmbraceSessionService(
                    preferencesService,
                    activityService,
                    ndkService,
                    sessionProperties,
                    internalEmbraceLogger,
                    sessionCacheManager,
                    sessionHandler,
                    localConfig.isNdkEnabled()
            );

            Thread.setDefaultUncaughtExceptionHandler(new EmbraceAutomaticVerification.AutomaticVerificationExceptionHandler(Thread.getDefaultUncaughtExceptionHandler()));
            this.crashService = new EmbraceCrashService(
                    localConfig,
                    sessionService,
                    metadataService,
                    apiClient,
                    userService,
                    eventService,
                    anrService,
                    ndkService,
                    gatingService);

            List<SessionStartListener> startListeners = new ArrayList<>();
            List<SessionEndListener> endListeners = new ArrayList<>();
            startListeners.add((EmbraceOrientationService) orientationService);
            endListeners.add((EmbraceOrientationService) orientationService);
            endListeners.add((EmbraceUserService) userService);

            if (unityThreadSampler != null) {
                endListeners.add(unityThreadSampler);
                memoryCleanerService.addListener(unityThreadSampler);
                internalEmbraceLogger.logDeveloper("Embrace", "Added UnityThreadSampler to SessionService end listeners");
            }
            if (configService.isBetaFeaturesEnabled()) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    ActivityLifecycleBreadcrumbCollector collector = new ActivityLifecycleBreadcrumbCollector(configService, clock);
                    application.registerActivityLifecycleCallbacks(collector);
                    endListeners.add(collector);
                    internalEmbraceLogger.logDeveloper("Embrace", "Added ActivityLifecycleBreadcrumbCollector to SessionService end listeners");

                    this.thermalStatusService = new EmbraceThermalStatusService(application,
                            Executors.newSingleThreadExecutor(), clock, internalEmbraceLogger);
                    endListeners.add(thermalStatusService);
                    internalEmbraceLogger.logDeveloper("Embrace", "Added EmbraceThermalStatusService to SessionService end listeners");
                }
            }

            internalEmbraceLogger.logDeveloper("Embrace", "Adding default session start listeners");
            sessionService.addSessionStartListeners(startListeners);

            internalEmbraceLogger.logDeveloper("Embrace", "Adding default session end listeners");
            sessionService.addSessionEndListeners(endListeners);
        } catch (Exception ex) {
            internalEmbraceLogger.logError("Exception occurred while initializing the Embrace SDK. Instrumentation may be disabled.", ex);
            if (enableIntegrationTesting) {
                internalEmbraceLogger.logDeveloper("Embrace", "Integration testing enabled");
                throw ex;
            }
            return;
        }

        String startMsg = "Embrace SDK started. App ID: " + this.localConfig.getAppId() +
                " Version: " + BuildConfig.VERSION_NAME;
        internalEmbraceLogger.logInfo(startMsg);

        if (localConfig.getConfigurations().getNetworking().isNativeNetworkingMonitoringEnabled()) {
            // Intercept Android network calls
            internalEmbraceLogger.logDeveloper("Embrace", "Native Networking Monitoring enabled");
            StreamHandlerFactoryInstaller.registerFactory(localConfig.getConfigurations().getNetworking().getCaptureRequestContentLength());
        }

        started.set(true);
        long startupDuration = this.clock.now() - startTime;
        sessionService.setSdkStartupDuration(startupDuration);
        internalEmbraceLogger.logDeveloper("Embrace", "Startup duration: " + startupDuration + " millis");

        // Attempt to send the startup event if the app is already in the foreground. We registered to send this when
        // we went to the foreground, but if an activity had already gone to the foreground, we may have missed
        // sending this, so to ensure the startup message is sent, we force it to be sent here.
        if (!activityService.isInBackground()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Sending startup moment");
            eventService.sendStartupMoment();
        }
    }

    /**
     * Whether or not the SDK has been started.
     *
     * @return true if the SDK is started, false otherwise
     */
    public boolean isStarted() {
        return started.get();
    }

    /**
     * Sets a custom app ID that overrides the one specified at build time. Must be called before
     * the SDK is started.
     *
     * @param appId custom app ID
     * @return true if the app ID could be set, false otherwise.
     */
    public boolean setAppId(@NonNull String appId) {
        if (isStarted()) {
            internalEmbraceLogger.logError("You must set the custom app ID before the SDK is started.");
            return false;
        }
        if (appId == null || appId.isEmpty()) {
            internalEmbraceLogger.logError("App ID cannot be null or empty.");
            return false;
        }
        if (!MetadataUtils.isValidAppID(appId)) {
            internalEmbraceLogger.logError("Invalid app ID. Must be a 5-character string with " +
                    "characters from the set [A-Za-z0-9], but it was \"" + appId + "\".");
            return false;
        }

        customAppId = appId;
        internalEmbraceLogger.logDeveloper("Embrace", "App Id set");
        return true;
    }

    /**
     * Shuts down the Embrace SDK.
     */
    void stop() {
        if (started.compareAndSet(true, false)) {
            internalEmbraceLogger.logInfo("Shutting down Embrace SDK.");
            try {
                // Keep the config service active to give the application the opportunity to
                // update the disabled flag, in case the SDK has been re-enabled.
                internalEmbraceLogger.logDeveloper("Embrace", "Attempting to close services...");


                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.powerService != null) {
                    this.powerService.close();
                }

                if (this.memoryService != null) {
                    this.memoryService.close();
                }

                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.activityService != null) {
                    this.activityService.close();
                }

                if (this.sessionService != null) {
                    this.sessionService.close();
                }

                if (this.eventService != null) {
                    this.eventService.close();
                }

                if (this.networkConnectivityService != null) {
                    this.networkConnectivityService.close();
                }

                if (this.configService != null) {
                    this.configService.close();
                }

                if (this.thermalStatusService != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    this.thermalStatusService.close();
                }

                this.memoryCleanerService = null;
                this.powerService = null;
                this.memoryService = null;
                this.breadcrumbService = null;
                this.activityService = null;
                this.preferencesService = null;
                this.networkConnectivityService = null;
                this.anrService = null;
                this.buildInfo = null;
                this.localConfig = null;
                this.metadataService = null;
                this.performanceInfoService = null;
                this.cacheService = null;
                this.userService = null;
                this.screenshotService = null;
                this.eventService = null;
                this.remoteLogger = null;
                this.sessionService = null;
                this.crashService = null;
                this.networkLoggingService = null;
                this.configService = null;
                this.application = null;
                this.thermalStatusService = null;

                internalEmbraceLogger.logDeveloper("Embrace", "Services closed");
                closeWorkers();
            } catch (Exception ex) {
                internalEmbraceLogger.logError("Error while shutting down Embrace SDK", ex);
            }
        }
    }

    /**
     * Close all workers used by the services.
     */
    private void closeWorkers() {
        bgRegistrationWorker.close();
        scheduledWorker.close();
        internalEmbraceLogger.logDeveloper("Embrace", "Workers closed");
    }

    /**
     * This method sets a logging level, but this logging level is never used.
     *
     * @param severity the severity
     * @deprecated as the log level is never used. Use {@link EmbraceLogger}.
     */
    @Deprecated
    public void setLogLevel(@NonNull EmbraceLogger.Severity severity) {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * This method enables debug logging.
     */
    public void enableDebugLogging() {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * This method disables debug logging.
     */
    public void disableDebugLogging() {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * Sets the user ID. This would typically be some form of unique identifier such as a UUID or
     * database key for the user.
     *
     * @param userId the unique identifier for the user
     */
    public void setUserIdentifier(@Nullable String userId) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.setUserIdentifier(userId);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (userId != null) {
                internalEmbraceLogger.logDebug("Set user ID to " + userId);
            } else {
                internalEmbraceLogger.logDebug("Cleared user ID by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user identifier");
        }
    }

    /**
     * Clears the currently set user ID. For example, if the user logs out.
     */
    public void clearUserIdentifier() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.clearUserIdentifier();
            internalEmbraceLogger.logDebug("Cleared user ID");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user identifier");
        }
    }

    /**
     * Sets the current user's email address.
     *
     * @param email the email address of the current user
     */
    public void setUserEmail(@Nullable String email) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.setUserEmail(email);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (email != null) {
                internalEmbraceLogger.logDebug("Set email to " + email);
            } else {
                internalEmbraceLogger.logDebug("Cleared email by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user email");
        }
    }

    /**
     * Clears the currently set user's email address.
     */
    public void clearUserEmail() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.clearUserEmail();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            internalEmbraceLogger.logDebug("Cleared email");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user email");
        }
    }

    /**
     * Sets this user as a paying user. This adds a persona to the user's identity.
     */
    public void setUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.setUserAsPayer();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user as payer");
        }
    }

    /**
     * Clears this user as a paying user. This would typically be called if a user is no longer
     * paying for the service and has reverted back to a basic user.
     */
    public void clearUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.clearUserAsPayer();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user as payer");
        }
    }

    /**
     * Sets a custom user persona. A persona is a trait associated with a given user.
     *
     * @param persona the persona to set
     */
    public void setUserPersona(@NonNull String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.setUserPersona(persona);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user persona");
        }
    }

    /**
     * Clears the custom user persona, if it is set.
     *
     * @param persona the persona to clear
     */
    public void clearUserPersona(@NonNull String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearUserPersona(persona);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user persona");
        }
    }

    /**
     * Clears all custom user personas from the user.
     */
    public void clearAllUserPersonas() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearAllUserPersonas();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user personas");
        }
    }

    public boolean addSessionProperty(@NonNull String key, @NonNull String value, boolean permanent) {
        if (isStarted()) {
            return sessionService.addProperty(key, value, permanent);
        }
        internalEmbraceLogger.logSDKNotInitialized("cannot add session property");
        return false;
    }

    public boolean removeSessionProperty(@NonNull String key) {
        if (isStarted()) {
            return sessionService.removeProperty(key);
        }

        internalEmbraceLogger.logSDKNotInitialized("remove session property");
        return false;
    }

    @Nullable
    public Map<String, String> getSessionProperties() {
        if (isStarted()) {
            return sessionService.getProperties();
        }

        internalEmbraceLogger.logSDKNotInitialized("gets session properties");
        return null;
    }

    /**
     * Sets the username of the currently logged in user.
     *
     * @param username the username to set
     */
    public void setUsername(@Nullable String username) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.setUsername(username);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (username != null) {
                internalEmbraceLogger.logDebug("Set username to " + username);
            } else {
                internalEmbraceLogger.logDebug("Cleared username by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set username");
        }
    }

    /**
     * Clears the username of the currently logged in user, for example if the user has logged out.
     */
    public void clearUsername() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.clearUsername();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            internalEmbraceLogger.logDebug("Cleared username");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear username");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name a name identifying the event
     */
    public void startEvent(@NonNull String name) {
        if (isStarted()) {
            eventService.startEvent(name);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     */
    public void startEvent(@NonNull String name, @Nullable String identifier) {
        if (isStarted()) {
            eventService.startEvent(name, identifier);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     */
    public void startEvent(@NonNull String name, @Nullable String identifier, boolean allowScreenshot) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    public void startEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, normalizeProperties(properties));
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     * @param properties      custom key-value pairs to provide with the event
     */
    public void startEvent(@NonNull String name, @Nullable String identifier, boolean allowScreenshot, @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot, normalizeProperties(properties));
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name the name of the event to end
     */
    public void endEvent(@NonNull String name) {
        if (isStarted()) {
            eventService.endEvent(name);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     */
    public void endEvent(@NonNull String name, @Nullable String identifier) {
        if (isStarted()) {
            eventService.endEvent(name, identifier);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param properties custom key-value pairs to provide with the event
     */
    public void endEvent(@NonNull String name, @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.endEvent(name, normalizeProperties(properties));
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    public void endEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.endEvent(name, identifier, normalizeProperties(properties));
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals that the app has completed startup.
     */
    public void endAppStartup() {
        endEvent(STARTUP_EVENT_NAME);
    }

    /**
     * Signals that the app has completed startup.
     *
     * @param properties properties to include as part of the startup moment
     */
    public void endAppStartup(@NonNull Map<String, Object> properties) {
        endEvent(STARTUP_EVENT_NAME, null, properties);
    }

    /**
     * Retrieve the HTTP request header to extract trace ID from.
     *
     * @return the Trace ID header.
     */
    @NonNull
    public String getTraceIdHeader() {
        return isStarted() ? this.localConfig.getConfigurations().getNetworking().getTraceIdHeader() : LocalConfig.SdkConfigs.Networking.CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE;
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the int value of the HTTP method of the network call
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param statusCode    the status code returned by the server
     * @param error         the error returned by the exception
     */
    public void logNetworkRequest(@NonNull String url,
                                  int httpMethod,
                                  long startTime,
                                  long endTime,
                                  int bytesSent,
                                  int bytesReceived,
                                  int statusCode,
                                  @Nullable String error) {
        EmbraceNetworkRequestV2.Builder requestBuilder = EmbraceNetworkRequestV2.newBuilder()
                .withUrl(url)
                .withHttpMethod(httpMethod)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .withBytesIn(bytesReceived)
                .withBytesOut(bytesSent)
                .withResponseCode(statusCode);

        if (error != null && !error.isEmpty()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Log network with error: " + error);
            requestBuilder.withError(new Throwable(error));
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "Log network request without errors");
        }

        logNetworkRequest(requestBuilder.build());
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param request An EmbraceNetworkRequestV2 with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequestV2 request) {
        if (isStarted()) {
            if (request == null) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request is null");
                return;
            }
            if (!request.canSend()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request can't be sent");
                return;
            }
            if (request.getError() != null) {
                networkLoggingService.logNetworkError(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getError().getClass().getCanonicalName(),
                        request.getError().getLocalizedMessage(),
                        request.getTraceId());
            } else {
                networkLoggingService.logNetworkCall(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getResponseCode(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getBytesIn(),
                        request.getBytesOut(),
                        request.getTraceId());
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network request");
        }
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param request An EmbraceNetworkRequest with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequest request) {
        if (isStarted()) {
            if (request == null) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request is null");
                return;
            }
            if (!request.canSend()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request can't be sent");
                return;
            }
            if (request.getError() != null) {
                networkLoggingService.logNetworkError(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getError().getClass().getCanonicalName(),
                        request.getError().getLocalizedMessage(),
                        request.getTraceId());
            } else {
                networkLoggingService.logNetworkCall(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getResponseCode(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getBytesIn(),
                        request.getBytesOut(),
                        request.getTraceId());
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network request");
        }
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     */
    public void logNetworkCall(
            @NonNull String url,
            @NonNull HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived) {
        logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, null);
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param traceId       the optional trace id that can be used to trace a particular request
     */
    public void logNetworkCall(
            @NonNull String url,
            @NonNull HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived,
            @Nullable String traceId) {
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log network call");

            if (configService.isUrlDisabled(url) || localConfig.isUrlDisabled(url)) {
                internalEmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }

            internalEmbraceLogger.logDeveloper("Embrace", "Log network call");
            networkLoggingService.logNetworkCall(
                    url,
                    httpMethod.name(),
                    statusCode,
                    startTime,
                    endTime,
                    bytesSent,
                    bytesReceived,
                    traceId);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network call");
        }
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     */
    public void logNetworkClientError(
            @NonNull String url,
            @NonNull HttpMethod httpMethod,
            long startTime,
            long endTime,
            @NonNull String errorType,
            @NonNull String errorMessage) {
        logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, null);
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     * @param traceId      the optional trace id that can be used to trace a particular request
     */
    public void logNetworkClientError(
            @NonNull String url,
            @NonNull HttpMethod httpMethod,
            long startTime,
            long endTime,
            @NonNull String errorType,
            @NonNull String errorMessage,
            @Nullable String traceId) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log network client error");
        if (isStarted()) {
            if (configService.isUrlDisabled(url) || localConfig.isUrlDisabled(url)) {
                internalEmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }

            internalEmbraceLogger.logDeveloper("Embrace", "Log network client error");
            networkLoggingService.logNetworkError(
                    url,
                    httpMethod.name(),
                    startTime,
                    endTime,
                    errorType,
                    errorMessage,
                    traceId);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network error");
        }
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logInfo(@NonNull String message) {
        logInfo(message, null);
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logInfo(@NonNull String message, @Nullable Map<String, Object> properties) {
        logMessage(EmbraceEvent.Type.INFO_LOG, message, properties, false, null, null, false);
    }


    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logWarning(@NonNull String message) {
        logWarning(message, null, false, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logWarning(@NonNull String message, @Nullable Map<String, Object> properties) {
        logWarning(message, properties, false, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logWarning(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logWarning(message, properties, allowScreenshot, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    public void logWarning(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot, @Nullable String javascriptStackTrace) {
        logMessage(EmbraceEvent.Type.WARNING_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, false);
    }


    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message the message to remotely log
     */
    public void logError(@NonNull String message) {
        logError(message, null, true, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logError(@NonNull String message, @Nullable Map<String, Object> properties) {
        logError(message, properties, true, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logError(message, properties, allowScreenshot, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    public void logError(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot, @Nullable String javascriptStackTrace) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     * @param isException          mark the log as an exception
     */

    public void logError(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot, @Nullable String javascriptStackTrace, boolean isException) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, isException);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e the throwable to remotely log
     */
    public void logError(@NonNull Throwable e) {
        logError(e, null, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e          the throwable to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logError(@NonNull Throwable e, @Nullable Map<String, Object> properties) {
        logError(e, properties, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e               the throwable to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(@NonNull Throwable e, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logError(e, e.getLocalizedMessage(), properties, allowScreenshot);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e               the throwable to remotely log
     * @param message         message to log (overrides getLocalizedMessage() from e)
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(@NonNull Throwable e, @NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, e.getStackTrace(), null, true);
    }

    void logMessage(
            EmbraceEvent.Type type,
            String message,
            Map<String, Object> properties,
            boolean allowScreenshot,
            StackTraceElement[] stackTraceElements,
            String customStackTrace,
            boolean isException) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log message");
        if (isStarted()) {
            try {
                this.remoteLogger.log(message, type, allowScreenshot, isException, normalizeProperties(properties), stackTraceElements, customStackTrace, this.appFramework);
            } catch (Exception ex) {
                internalEmbraceLogger.logDebug("Failed to log message using Embrace SDK.", ex);
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log message");
        }
    }

    /**
     * Logs a breadcrumb.
     * <p>
     * Breadcrumbs track a user's journey through the application and will be shown on the timeline.
     *
     * @param message the name of the breadcrumb to log
     */
    public void logBreadcrumb(@NonNull String message) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log breadcrumb");
        if (isStarted()) {
            this.breadcrumbService.logCustom(message, System.currentTimeMillis());
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log breadcrumb");
        }
    }

    /**
     * Logs a javascript unhandled exception.
     *
     * @param name       name of the exception.
     * @param message    exception message.
     * @param type       error type.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logUnhandledJsException(@NonNull String name, @NonNull String message, @Nullable String type, @Nullable String stacktrace) {
        if (isStarted()) {
            JsException exception = new JsException(name, message, type, stacktrace);
            internalEmbraceLogger.logDeveloper("Embrace", "Log Unhandled JS exception: " + name + " -- stacktrace: " + stacktrace);
            this.crashService.logUnhandledJsException(exception);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log JS exception");
        }
    }

    /**
     * Logs a Unity unhandled exception.
     *
     * @param message    exception message.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logUnhandledUnityException(@NonNull String message, @Nullable String stacktrace) {
        if (isStarted()) {
            internalEmbraceLogger.logError("message: " + message + " -- stacktrace: " + stacktrace);
            logMessage(
                    EmbraceEvent.Type.ERROR_LOG,
                    message,
                    null,
                    false,
                    null,
                    stacktrace,
                    true);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log Unity exception");
        }
    }

    /**
     * Sets the react native version number.
     *
     * @param version react native version number.
     */
    public void setReactNativeVersionNumber(@NonNull String version) {
        if (isStarted()) {
            if (version == null) {
                internalEmbraceLogger.logError("ReactNative version must not be null");
                return;
            }

            if (version.isEmpty()) {
                internalEmbraceLogger.logError("ReactNative version must have non-zero length");
                return;
            }

            this.preferencesService.setReactNativeVersionNumber(version);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set React Native version number");
        }
    }

    /**
     * Sets javascript patch number.
     *
     * @param number javascript patch number.
     */
    public void setJavaScriptPatchNumber(@NonNull String number) {
        if (isStarted()) {
            if (number == null) {
                internalEmbraceLogger.logError("JavaScript patch number must not be null");
                return;
            }

            if (number.isEmpty()) {
                internalEmbraceLogger.logError("JavaScript patch number must have non-zero length");
                return;
            }

            this.preferencesService.setJavaScriptPatchNumber(number);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set JavaScript patch number");
        }
    }


    /**
     * Sets the path of the javascript bundle.
     *
     * @param url path of the javascript bundle.
     */
    public void setJavaScriptBundleURL(@NonNull String url) {
        if (isStarted()) {
            if (appFramework != AppFramework.REACT_NATIVE) {
                internalEmbraceLogger.logError("Failed to set Java Script bundle ID URL. Current framework: "
                        + appFramework.name() + " is not React Native.");
                return;
            }

            this.metadataService.setReactNativeBundleId(application.getApplicationContext(), url);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set JavaScript bundle URL");
        }
    }

    /**
     * Sets the Unity version and Unity build id.
     *
     * @param unityVersion of the Unity SDK
     * @param buildGUID    if the Unity build
     */
    @InternalApi
    public void setUnityMetaData(@NonNull String unityVersion, @NonNull String buildGUID) {
        if (isStarted()) {
            if (unityVersion == null || buildGUID == null) {
                internalEmbraceLogger.logError("Unity metadata is corrupted or malformed. Unity version is " + unityVersion + " and Unity build id is " + buildGUID);
                return;
            }

            if (this.preferencesService.getUnityVersionNumber().isPresent()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Unity version number is present");
                String currentUnityVersion = this.preferencesService.getUnityVersionNumber().get();

                if (!unityVersion.equals(currentUnityVersion)) {
                    internalEmbraceLogger.logDeveloper("Embrace", "Setting a new Unity version number");
                    this.preferencesService.setUnityVersionNumber(unityVersion);
                }
            } else {
                internalEmbraceLogger.logDeveloper("Embrace", "Setting Unity version number");
                this.preferencesService.setUnityVersionNumber(unityVersion);
            }

            if (this.preferencesService.getUnityBuildIdNumber().isPresent()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Unity build id is present");

                String currentUnityBuildId = this.preferencesService.getUnityBuildIdNumber().get();

                if (!buildGUID.equals(currentUnityBuildId)) {
                    internalEmbraceLogger.logDeveloper("Embrace", "Setting a Unity new build id");
                    this.preferencesService.setUnityBuildIdNumber(buildGUID);
                }
            } else {
                internalEmbraceLogger.logDeveloper("Embrace", "Setting Unity build id");
                this.preferencesService.setUnityBuildIdNumber(buildGUID);
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set Unity metadata");
        }
    }

    /**
     * Registers a {@link ConnectionQualityListener}, notifying the listener each time that there is
     * a change in the connection quality.
     *
     * @param listener the listener to register
     */
    @Deprecated
    public void addConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        internalEmbraceLogger.logWarning("Warning: failed to remove connection quality listener. " +
                "The signal quality service is deprecated.");
    }

    /**
     * Removes a registered {@link ConnectionQualityListener}, suspending connection quality
     * notifications.
     *
     * @param listener the listener to remove
     */
    @Deprecated
    public void removeConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        internalEmbraceLogger.logWarning("Warning: failed to remove connection quality listener. " +
                "The signal quality service is deprecated.");
    }

    /**
     * Ends the current session and starts a new one.
     */
    public synchronized void endSession() {
        endSession(false);
    }

    /**
     * Ends the current session and starts a new one.
     * <p>
     * Cleans all the user info on the device.
     */
    public synchronized void endSession(boolean clearUserInfo) {
        if (isStarted()) {
            if (localConfig.getConfigurations().getSessionConfig().getMaxSessionSecondsAllowed().isPresent()) {
                internalEmbraceLogger.logWarning("Can't close the session, automatic session close enabled.");
                return;
            }

            if (localConfig.getConfigurations().getSessionConfig().getAsyncEnd() ||
                    configService.getConfig().endSessionInBackgroundThread()) {
                internalEmbraceLogger.logWarning("Can't close the session, session ending in background thread enabled.");
                return;
            }

            if (clearUserInfo) {
                userService.clearAllUserInfo();
                // Update user info in NDK service
                ndkService.onUserInfoUpdate();
            }

            sessionService.triggerStatelessSessionEnd(Session.SessionLifeEventType.MANUAL);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("end session");
        }
    }

    /**
     * Get the user identifier assigned to the device by Embrace
     *
     * @return the device identifier created by Embrace
     */
    @NonNull
    public String getDeviceId() {
        return preferencesService.getDeviceIdentifier();
    }

    /**
     * Causes a crash with an exception. Use this for test purposes only
     */
    public void throwException() {
        throw new RuntimeException("EmbraceException", new Throwable("Embrace test exception"));
    }

    /**
     * Log the start of a fragment.
     * <p>
     * A matching call to endFragment must be made.
     *
     * @param name the name of the fragment to log
     */
    public boolean startFragment(@NonNull String name) {
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Starting fragment: " + name);
            return this.breadcrumbService.startFragment(name);
        }

        internalEmbraceLogger.logDeveloper("Embrace", "Cannot start fragment, SDK is not started");
        return false;
    }

    /**
     * Log the end of a fragment.
     * <p>
     * A matching call to startFragment must be made before this is called.
     *
     * @param name the name of the fragment to log
     */
    public boolean endFragment(@NonNull String name) {
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Ending fragment: " + name);
            return this.breadcrumbService.endFragment(name);
        }

        internalEmbraceLogger.logDeveloper("Embrace", "Cannot end fragment, SDK is not started");
        return false;
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    void logView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.logView(screen, System.currentTimeMillis());
        }

        internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log view");
    }

    /**
     * Logs that a particular WebView URL was loaded.
     *
     * @param url the url to log
     */
    void logWebView(String url) {
        if (isStarted()) {
            this.breadcrumbService.logWebView(url, System.currentTimeMillis());
        }

        internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log view");
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will be
     * logged, and not treated as a duplicate.
     *
     * @param screen the name of the view to log
     */
    void forceLogView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.forceLogView(screen, System.currentTimeMillis());
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot force log view");
        }
    }

    /**
     * Logs a tap on a screen element.
     *
     * @param point       the coordinates of the screen tap
     * @param elementName the name of the element which was tapped
     * @param type        the type of tap that occurred
     */
    void logTap(Pair<Float, Float> point, String elementName, TapBreadcrumb.TapBreadcrumbType type) {
        if (isStarted()) {
            this.breadcrumbService.logTap(point, elementName, System.currentTimeMillis(), type);
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log tap");
        }
    }

    @Nullable
    @InternalApi
    public LocalConfig getLocalConfig() {
        if (isStarted()) {
            return localConfig;
        } else {
            internalEmbraceLogger.logSDKNotInitialized("get local config");
        }
        return null;
    }

    @Nullable
    @InternalApi
    public ConfigService getConfigService() {
        if (isStarted()) {
            return configService;
        } else {
            internalEmbraceLogger.logSDKNotInitialized("get local config");
        }
        return null;
    }

    EventService getEventService() {
        return eventService;
    }

    ActivityService getActivityService() {
        return activityService;
    }

    EmbraceRemoteLogger getRemoteLogger() {
        return remoteLogger;
    }

    EmbraceExceptionService getExceptionsService() {
        return exceptionsService;
    }

    MetadataService getMetadataService() {
        return metadataService;
    }

    EmbraceSessionService getSessionService() {
        return sessionService;
    }

    private @Nullable
    Map<String, Object> normalizeProperties(@Nullable Map<String, Object> properties) {
        Map<String, Object> normalizedProperties = new HashMap<>();
        if (properties != null) {
            try {
                internalEmbraceLogger.logDeveloper("Embrace", "normalizing properties");
                normalizedProperties = PropertyUtils.sanitizeProperties(properties);
            } catch (Exception e) {
                internalEmbraceLogger.logError("Exception occurred while normalizing the properties.", e);
            }
            return normalizedProperties;
        } else {
            return null;
        }
    }

    public enum AppFramework {
        NATIVE(1),
        REACT_NATIVE(2),
        UNITY(3);

        private final int value;

        AppFramework(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }
}