package io.embrace.android.embracesdk;

import android.app.Activity;
import android.app.Application;

import androidx.annotation.VisibleForTesting;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.OnLifecycleEvent;
import androidx.lifecycle.ProcessLifecycleOwner;

import android.content.res.Configuration;
import android.os.Bundle;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.ThreadUtils;
import io.embrace.android.embracesdk.utils.optional.Optional;

import java.lang.ref.WeakReference;
import java.util.concurrent.CopyOnWriteArrayList;

import io.embrace.android.embracesdk.annotation.StartupActivity;
import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Service tracking the app's current activity and background state, and dispatching events to other
 * services as required.
 * <p>
 * See activity lifecycle documentation here:
 * https://developer.android.com/guide/components/activities/activity-lifecycle
 * <p>
 * We only need to track the `onForeground` and `onActivityPaused` lifecycle hooks for background
 * detection, as these are the only two which are consistently fired in all scenarios.
 */
final class EmbraceActivityService implements ActivityService {

    private static final String ERROR_FAILED_TO_NOTIFY = "Failed to notify EmbraceActivityService listener";

    /**
     * The memory service, it's provided on the instantiation of the service.
     */
    private final MemoryService memoryService;

    /**
     * The orientation service.
     */
    private final OrientationService orientationService;

    /**
     * List of listeners that subscribe to activity events.
     */
    @VisibleForTesting
    final CopyOnWriteArrayList<ActivityListener> listeners = new CopyOnWriteArrayList<>();

    /**
     * The application.
     */
    private final Application application;

    private final Clock clock;

    /**
     * The currently active activity.
     */
    private volatile WeakReference<Activity> currentActivity = new WeakReference<>(null);

    /**
     * States if the activity foreground phase comes from a cold start or not.
     * Checked every time an activity executes a foreground phase.
     */
    private volatile boolean coldStart = true;

    /**
     * States the initialization time of the EmbraceActivityService, inferring it is initialized
     * from the {@link Embrace#start(android.content.Context)} method.
     */
    private final long startTime;

    /**
     * Describes the application state.
     */
    private volatile boolean isInBackground = true;

    EmbraceActivityService(
            Application application,
            MemoryService memoryService,
            OrientationService orientationService,
            Clock clock) {

        this.clock = Preconditions.checkNotNull(clock);
        this.memoryService = Preconditions.checkNotNull(memoryService);
        this.orientationService = Preconditions.checkNotNull(orientationService);
        this.application = Preconditions.checkNotNull(application);
        this.startTime = clock.now();
        application.registerActivityLifecycleCallbacks(this);
        application.getApplicationContext().registerComponentCallbacks(this);
        // add lifecycle observer on main thread to avoid IllegalStateExceptions with
        // androidx.lifecycle
        ThreadUtils.runOnMainThread(() -> ProcessLifecycleOwner.get().getLifecycle()
                .addObserver(EmbraceActivityService.this)
        );
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle bundle) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity created: " + getActivityName(activity));
        updateStateWithActivity(activity);
        updateOrientationWithActivity(activity);
    }

    @Override
    public void onActivityStarted(Activity activity) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity started: " + getActivityName(activity));
        updateStateWithActivity(activity);
        StreamUtilsKt.stream(listeners, listener -> {
            try {
                listener.onView(activity);
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logDebug(ERROR_FAILED_TO_NOTIFY, ex);
            }
            return null;
        });
    }

    @Override
    public void onActivityResumed(Activity activity) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity resumed: " + getActivityName(activity));
        if (!activity.getClass().isAnnotationPresent(StartupActivity.class)) {
            // If the activity coming to foreground doesn't have the StartupActivity annotation
            // the the SDK will finalize any pending startup moment.

            InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity resumed: " + getActivityName(activity));
            StreamUtilsKt.stream(listeners, listener -> {
                try {
                    listener.applicationStartupComplete();
                } catch (Exception ex) {
                    InternalStaticEmbraceLogger.logDebug(ERROR_FAILED_TO_NOTIFY, ex);
                }
                return null;
            });
        } else {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", getActivityName(activity) + " is @StartupActivity");
        }
    }

    @Override
    public void onActivityPaused(Activity activity) {
    }

    @Override
    public void onActivityStopped(Activity activity) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity stopped: " + getActivityName(activity));
        StreamUtilsKt.stream(listeners, listener -> {
            try {
                listener.onViewClose(activity);
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logDebug(ERROR_FAILED_TO_NOTIFY, ex);
            }
            return null;
        });
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle bundle) {
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
    }

    /**
     * This method will update the current activity for further checking.
     *
     * @param activity the activity involved in the state change.
     */
    @VisibleForTesting
    synchronized void updateStateWithActivity(Activity activity) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Current activity: " + getActivityName(activity));
        this.currentActivity = new WeakReference<>(activity);
    }

    /**
     * This method will update the current activity orientation.
     *
     * @param activity the activity involved in the tracking orientation process.
     */
    private void updateOrientationWithActivity(Activity activity) {
        try {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Updated orientation: " + activity.getResources().getConfiguration().orientation);
            orientationService.onOrientationChanged(Optional.of(activity.getResources().getConfiguration().orientation));
        } catch (Exception ex) {
            InternalStaticEmbraceLogger.logDebug("Failed to register an orientation change", ex);
        }
    }

    /**
     * This method will be called by the ProcessLifecycleOwner when the main app process calls
     * ON START.
     **/
    @OnLifecycleEvent(Lifecycle.Event.ON_START)
    void onForeground() {
        InternalStaticEmbraceLogger.logDebug("AppState: App entered foreground.");
        isInBackground = false;
        StreamUtilsKt.stream(listeners, listener -> {
            try {
                listener.onForeground(coldStart, this.startTime);
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logDebug(ERROR_FAILED_TO_NOTIFY, ex);
            }
            return null;
        });
        coldStart = false;
    }

    /**
     * This method will be called by the ProcessLifecycleOwner when the main app process calls
     * ON STOP.
     **/
    @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
    void onBackground() {
        InternalStaticEmbraceLogger.logDebug("AppState: App entered background");
        updateStateWithActivity(null);
        isInBackground = true;
        StreamUtilsKt.stream(listeners, listener -> {
            try {
                listener.onBackground();
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logDebug(ERROR_FAILED_TO_NOTIFY, ex);
            }
            return null;
        });
    }

    /**
     * Called when the OS has determined that it is a good time for a process to trim unneeded
     * memory.
     *
     * @param trimLevel the context of the trim, giving a hint of the amount of trimming.
     */
    @Override
    public void onTrimMemory(int trimLevel) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "onTrimMemory(). TrimLevel: " + trimLevel);

        if (trimLevel == TRIM_MEMORY_RUNNING_LOW) {
            try {
                memoryService.onMemoryWarning();
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logDebug("Failed to handle onTrimMemory (low memory) event", ex);
            }
        }
    }

    @Override
    public void onConfigurationChanged(Configuration configuration) {
    }

    @Override
    public void onLowMemory() {
    }

    /**
     * Returns if the app's in background or not.
     */
    @Override
    public boolean isInBackground() {
        return isInBackground;
    }

    /**
     * Returns the current activity instance
     */
    @Override
    public Optional<Activity> getForegroundActivity() {
        Optional<Activity> foregroundActivity = Optional.fromNullable(currentActivity.get());
        if (!foregroundActivity.isPresent() ||
                foregroundActivity.get().isFinishing()) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Foreground activity not present");
            return Optional.absent();
        }

        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Foreground activity name: " + getActivityName(foregroundActivity.get()));
        return foregroundActivity;
    }

    @Override
    public void addListener(ActivityListener listener, boolean priority) {
        if (!listeners.contains(listener)) {
            if (priority) {
                listeners.add(0, listener);
            } else {
                listeners.addIfAbsent(listener);
            }
        }
    }

    @Override
    public void addListener(ActivityListener listener) {
        addListener(listener, false);
    }

    @Override
    public void removeListener(ActivityListener listener) {
        listeners.remove(listener);
        InternalStaticEmbraceLogger.logDeveloper("EmbraceActivityService", "Activity listeners removed");
    }

    @Override
    public void close() {
        try {
            InternalStaticEmbraceLogger.logDebug("Shutting down EmbraceActivityService");
            application.getApplicationContext().unregisterComponentCallbacks(this);
            application.unregisterActivityLifecycleCallbacks(this);
            listeners.clear();
        } catch (Exception ex) {
            InternalStaticEmbraceLogger.logDebug("Error when closing EmbraceActivityService", ex);
        }
    }

    private String getActivityName(Activity activity) {
        return activity != null ? activity.getLocalClassName() : "Null";
    }
}
