package io.embrace.android.embracesdk;

import android.os.Debug;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.ThreadUtils;

/**
 * Checks whether the target thread is still responding by using the following strategy:
 * <ol>
 * <li>Creating a {@link Handler}s, on the target thread, and an executor on a monitor thread</li>
 * <li>Using the 'monitoring' thread to message the target thread with a heartbeat</li>
 * <li>Determining whether the target thread responds in time, and if not logging an ANR</li>
 * </ol>
 */
final class EmbraceAnrService implements AnrService, MemoryCleanerListener, ActivityListener {

    private static final long ANR_THRESHOLD_INTERVAL = 1000L;

    /**
     * The number of milliseconds which the monitor thread is allowed to timeout before we
     * assume that the process has been put into the cached state.
     */
    private static final long MONITOR_THREAD_TIMEOUT_MS = 30000L;

    private static final int HEALTHCHECK_REQUEST = 34593;
    private static final long BACKGROUND_ANR_SAFE_INTERVAL_MS = 10L;

    @VisibleForTesting
    final Clock clock;
    ConfigService configService;

    private final Thread targetThread;
    final TargetThreadHandler targetThreadHandler = new TargetThreadHandler(Looper.getMainLooper());
    final ScheduledExecutorService monitorService;

    private ScheduledFuture<?> monitorFuture;
    private long intervalMs;
    final AnrServiceState state;
    private boolean inForeground;
    final AnrStacktraceState stacktraceState;

    private final InternalEmbraceLogger logger;

    private final SharedObjectLoader sharedObjectLoader;

    private final FindGoogleThread findGoogleThread;

    private final GoogleAnrHandlerNativeDelegate googleAnrHandlerNativeDelegate;

    private final GoogleAnrTimestampRepository googleAnrTimestampRepository;

    private final AtomicBoolean googleAnrTrackerInstalled = new AtomicBoolean(false);

    EmbraceAnrService(Clock clock,
                      ConfigService configService,
                      InternalEmbraceLogger logger,
                      SharedObjectLoader sharedObjectLoader,
                      FindGoogleThread findGoogleThread,
                      GoogleAnrHandlerNativeDelegate googleAnrHandlerNativeDelegate,
                      GoogleAnrTimestampRepository googleAnrTimestampRepository) {
        this(clock, configService, Looper.getMainLooper().getThread(),
                new ForegroundDetector(), logger, sharedObjectLoader, findGoogleThread,
                googleAnrHandlerNativeDelegate, googleAnrTimestampRepository);
    }

    @VisibleForTesting
    EmbraceAnrService(Clock clock,
                      ConfigService configService,
                      Thread targetThread,
                      ForegroundDetector foregroundDetector,
                      InternalEmbraceLogger logger,
                      SharedObjectLoader sharedObjectLoader,
                      FindGoogleThread findGoogleThread,
                      GoogleAnrHandlerNativeDelegate googleAnrHandlerNativeDelegate,
                      GoogleAnrTimestampRepository googleAnrTimestampRepository) {
        this.clock = Preconditions.checkNotNull(clock);
        this.configService = Preconditions.checkNotNull(configService);
        this.targetThread = Preconditions.checkNotNull(targetThread);
        this.logger = logger;
        this.sharedObjectLoader = sharedObjectLoader;
        this.findGoogleThread = findGoogleThread;
        this.googleAnrHandlerNativeDelegate = googleAnrHandlerNativeDelegate;
        this.googleAnrTimestampRepository = googleAnrTimestampRepository;

        ThreadFactory threadFactory = WorkerUtils.createThreadFactory("Embrace ANR Healthcheck");
        this.monitorService = Executors.newSingleThreadScheduledExecutor(threadFactory);
        this.intervalMs = configService.getConfig().getCaptureAnrIntervalMs();
        logger.logDeveloper("EmbraceAnrService", "ANR interval millis: " + intervalMs);

        this.stacktraceState = new AnrStacktraceState(configService, targetThread);

        // add listeners
        CopyOnWriteArrayList<BlockedThreadListener> listeners = new CopyOnWriteArrayList<>();
        listeners.add(stacktraceState);
        this.state = new AnrServiceState(clock, listeners);

        // We use process importance as a proxy for whether the app is in the foreground or not.
        // This is important for SDKs like Unity where we miss the initial lifecycle callback.
        this.inForeground = foregroundDetector.isInForeground();
        logger.logDeveloper("EmbraceAnrService", "In Foreground: " + inForeground);
    }

    void startAnrCapture() {
        if (!state.started.getAndSet(true)) {
            logger.logInfo("Started healthchecks to capture any ANRs.");
            scheduleHealthChecks();
        }
    }

    private void scheduleHealthChecks() {
        intervalMs = configService.getConfig().getCaptureAnrIntervalMs();
        Runnable runnable = () -> {
            if (intervalMs != configService.getConfig().getCaptureAnrIntervalMs()) {
                logger.logDeveloper("EmbraceAnrService", "Different interval detected, restarting runnable");
                monitorFuture.cancel(false);
                scheduleHealthChecks();
            } else {
                long now = clock.now();
                Message obtain = Message.obtain(targetThreadHandler, EmbraceAnrService.HEALTHCHECK_REQUEST);
                targetThreadHandler.sendMessage(obtain);
                handleHealthCheckExecute(now);
            }
        };
        try {
            logger.logDeveloper("EmbraceAnrService", "Health check Interval : " + intervalMs);
            monitorFuture = monitorService.scheduleAtFixedRate(runnable, 0, intervalMs, TimeUnit.MILLISECONDS);
        } catch (Exception exc) {
            logger.logError("ANR capture initialization failed", exc);
        }
    }

    @Override
    public void finishInitialization(@NonNull MemoryCleanerService memoryCleanerService,
                                     @NonNull ActivityService activityService,
                                     @NonNull ConfigService configService) {
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        Preconditions.checkNotNull(activityService).addListener(this, true);
        this.configService = Preconditions.checkNotNull(configService);
        stacktraceState.setConfigService(configService);
        logger.logDeveloper("EmbraceAnrService", "Finish initialization");
        initializeGoogleAnrTracking();
        startAnrCapture();
    }

    @Override
    public void addBlockedThreadListener(BlockedThreadListener listener) {
        state.addListeners(Collections.singletonList(listener));
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        inForeground = true;
    }

    @Override
    public void onBackground() {
        inForeground = false;
    }

    @Override
    public List<AnrInterval> getAnrIntervals(long startTime, long endTime) {
        synchronized (this) {
            ArrayList<AnrInterval> results;
            long safeStartTime = startTime + BACKGROUND_ANR_SAFE_INTERVAL_MS;

            if (safeStartTime < endTime) {
                // return _all_ the intervals to allow capturing BG ANRs.
                Collection<AnrInterval> intervals = stacktraceState.anrIntervals.values();
                if (!configService.isBgAnrCaptureEnabled()) {
                    // Filter out ANRs that started before session start
                    results = new ArrayList<>();
                    for (AnrInterval interval : intervals) {
                        if (interval.getStartTime() >= safeStartTime) {
                            results.add(interval);
                        }
                    }
                } else {
                    results = new ArrayList<>(intervals);
                }
            } else {
                results = new ArrayList<>();
            }

            if (state.getAnrInProgress() && !state.isProcessCrashing()) {
                long intervalEndTime = clock.now();
                long responseMs = state.getLastTargetThreadResponseMs();
                long duration = intervalEndTime - responseMs;

                AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
                anrIntervalBuilder
                        .withStartTime(responseMs)
                        .withLastKnownTime(intervalEndTime)
                        .withEndTime(null)
                        .withType(AnrInterval.Type.UI);

                if (configService.isAnrCaptureEnabled() && stacktraceState.isMinimumCaptureDurationExceeded(duration)) {
                    anrIntervalBuilder.withStacktraces(stacktraceState.stacktraces);
                }
                results.add(anrIntervalBuilder.build());
            }
            return results;
        }
    }

    private void initializeGoogleAnrTracking() {
        logger.logDeveloper("EmbraceAnrService", "Deciding whether to initialize Google ANR Tracking");
        if (configService.isGoogleAnrCaptureEnabled()) {
            setupGoogleAnrTracking();
        } else {
            // always install the handler. if config subsequently changes we won't install the tracker twice, nor
            // we will install it if it's disabled.
            configService.addListener((previousConfig, newConfig) -> setupGoogleAnrTracking());
        }
    }

    private void setupGoogleAnrTracking() {
        if (configService.isGoogleAnrCaptureEnabled() && !googleAnrTrackerInstalled.getAndSet(true)) {
            ThreadUtils.runOnMainThread(this::setupGoogleAnrHandler);
        }
    }

    private void setupGoogleAnrHandler() {
        logger.logDeveloper("EmbraceAnrService", "Setting up Google ANR Handler");
        // TODO: split up the ANR tracking and NDK crash reporter libs
        if (!sharedObjectLoader.loadEmbraceNative()) {
            googleAnrTrackerInstalled.set(false);
            return;
        }

        // we must find the Google watcher thread in order to install the Google ANR handle.
        int googleThreadId = findGoogleThread.invoke();
        if (googleThreadId <= 0) {
            logger.logError("Could not initialize Google ANR tracking: Google thread not found.");
            googleAnrTrackerInstalled.set(false);
            return;
        }
        // run the JNI call from main thread since JNI calls return to the thread where
        // they were called.
        installGoogleAnrHandler(googleThreadId);
    }

    @Override
    public void forceANRTrackingStopOnCrash() {
        state.setProcessCrashing(true);
        close();
    }

    @Override
    public void close() {
        try {
            monitorService.shutdown();
            if (!monitorService.awaitTermination(1, TimeUnit.SECONDS)) {
                EmbraceLogger.logDebug("Timed out shutting down EmbraceAnrService after 1s");
            }
        } catch (Exception ex) {
            logger.logDebug("Failed to cleanly shut down EmbraceAnrService");
        }
    }

    @Override
    public void cleanCollections() {
        stacktraceState.cleanCollections();
        googleAnrTimestampRepository.clear();
    }

    @VisibleForTesting
    synchronized void handleHealthCheckResponse(long timestamp) {
        if (isAnrDurationThresholdExceeded(timestamp) && isDebuggerDisabled()) {
            // Application was not responding, but recovered
            logger.logDebug("Main thread recovered from not responding for > 1s");

            // Invoke callbacks
            state.onThreadUnblocked(targetThread, timestamp);
        }
        state.setLastTargetThreadResponseMs(timestamp);
    }

    @VisibleForTesting
    void handleHealthCheckExecute(long timestamp) {
        if (isAnrDurationThresholdExceeded(timestamp) && !state.getAnrInProgress() && isDebuggerDisabled()) {
            logger.logDebug("Main thread not responding for > 1s");

            // Invoke callbacks
            state.onThreadBlocked(targetThread, state.getLastTargetThreadResponseMs());
        }

        if (state.getAnrInProgress()) {
            processAnrTick(timestamp);
        }

        state.setLastMonitorThreadResponseMs(clock.now());
    }

    @VisibleForTesting
    synchronized void processAnrTick(long timestamp) {
        // Check if ANR capture is enabled
        if (!configService.isAnrCaptureEnabled()) {
            logger.logDeveloper("EmbraceAnrService", "ANR capture is disabled, ignoring ANR tick");
            return;
        }

        // Tick limit
        int limit = configService.getConfig().getStacktracesPerInterval();
        if (stacktraceState.stacktraces.size() >= limit) {
            logger.logDebug("ANR stacktrace not captured. Maximum allowed ticks per ANR interval reached.");
            return;
        }

        // Invoke callbacks
        state.onThreadBlockedInterval(targetThread, timestamp);
    }

    @VisibleForTesting
    boolean isAnrDurationThresholdExceeded(long timestamp) {
        long responseMs = state.getLastTargetThreadResponseMs();
        long monitorThreadLag = timestamp - responseMs;

        // If the last monitor thread check greatly exceeds the ANR threshold
        // then it is very probable that the process has been cached or frozen. In this case
        // we need to ignore the first health check as the clock won't have been ticking
        // while the process was cached and this could cause a false positive.
        //
        // Therefore we reset the last response time from the target + monitor threads to
        // the current time so that we can start monitoring for ANRs again.
        // https://developer.android.com/guide/components/activities/process-lifecycle
        if (monitorThreadLag > MONITOR_THREAD_TIMEOUT_MS) {
            logger.logDeveloper("EmbraceAnrService", "Exceeded monitor thread timeout");
            state.setLastTargetThreadResponseMs(clock.now());
            state.setLastMonitorThreadResponseMs(clock.now());
            return false;
        }

        return monitorThreadLag > ANR_THRESHOLD_INTERVAL;
    }

    private boolean isDebuggerDisabled() {
        return !Debug.isDebuggerConnected() && !Debug.waitingForDebugger();
    }

    /**
     * Handles healthcheck messages sent to the target thread. Responds with an acknowledgement to
     * the monitor thread.
     */
    class TargetThreadHandler extends Handler {

        public TargetThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            try {
                long now = clock.now();
                monitorService.submit(() -> handleHealthCheckResponse(now));
            } catch (Exception ex) {
                InternalStaticEmbraceLogger.logError("ANR healthcheck failed in main (monitored) thread", ex);
            }
            super.handleMessage(msg);
        }
    }

    void installGoogleAnrHandler(int googleThreadId) {
        int res = googleAnrHandlerNativeDelegate.install(googleThreadId);
        if (res > 0) {
            googleAnrTrackerInstalled.set(false);
            logger.logError(String.format(Locale.US, "Could not initialize Google ANR tracking {code=%d}", res));
        } else {
            logger.logInfo("Google Anr Tracker handler installed successfully");
        }
    }

}
