package io.embrace.android.embracesdk

import android.app.Activity
import android.content.Intent
import android.os.Handler
import android.os.Looper
import android.widget.Toast
import androidx.annotation.VisibleForTesting
import androidx.appcompat.app.AlertDialog
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logger
import io.embrace.android.embracesdk.network.http.HttpMethod
import io.embrace.android.embracesdk.networking.EmbraceUrl
import io.embrace.android.embracesdk.utils.exceptions.Unchecked
import java.io.File
import java.io.IOException
import java.util.*
import java.util.concurrent.RejectedExecutionException
import java.util.concurrent.TimeUnit
import kotlin.system.exitProcess

/**
 * Class that includes the logic to run the automatic Verification executing the following function:
 * EmbraceSamples.verifyIntegration();
 *
 * Under the hood this function will create a marker File. If a marker file doesn't already exist, then it runs:
1. End a session 5s after we detect the process entered the foreground
2. Start and end a custom moment
3. Send an error log
4. Make a HTTP request
5. Trigger an uncaught JVM exception
6. Relaunch the application
 */
internal class EmbraceAutomaticVerification : ActivityListener {

    val handler = Handler(Looper.getMainLooper())

    @VisibleForTesting
    internal lateinit var activityService: ActivityService

    @VisibleForTesting
    var verificationStatus = VerificationStatus.NOT_STARTED

    @VisibleForTesting
    lateinit var embraceInstance: Embrace

    @VisibleForTesting
    internal lateinit var automaticVerificationChecker: AutomaticVerificationChecker

    @VisibleForTesting
    fun setActivityService() {
        if (!::activityService.isInitialized) {
            activityService = Embrace.getInstance().activityService
        }
    }

    @VisibleForTesting
    fun setActivityListener() {
        setActivityService()
        activityService.addListener(this)
    }

    @VisibleForTesting
    internal fun setAutomaticVerificationCheckerInstance(automaticVerificationChecker: AutomaticVerificationChecker) {
        this.automaticVerificationChecker = automaticVerificationChecker
    }

    private fun showSuccessDialog(activity: Activity) {
        val dialogBuilder = AlertDialog.Builder(activity)
        dialogBuilder
            .setTitle(activity.getString(R.string.automatic_verification_success_title))
            .setMessage(activity.getString(R.string.automatic_verification_success_message))
            .setCancelable(true)
            .setPositiveButton(activity.getString(R.string.got_it)) { dialog, _ ->
                dialog.dismiss()
            }
        dialogBuilder.create().show()
    }

    private fun showDialogWithError() {
        val activity = activityService.foregroundActivity.get()
        val dialogBuilder = AlertDialog.Builder(activity)
        dialogBuilder
            .setTitle(activity.getString(R.string.automatic_verification_error_title))
            .setMessage(activity.getString(R.string.automatic_verification_error_message))
            .setCancelable(true)
            .setPositiveButton(activity.getString(R.string.try_again)) { dialog, _ ->
                automaticVerificationChecker.deleteFile()
                dialog.dismiss()
            }
            .setNegativeButton(activity.getString(R.string.close)) { dialog, _ ->
                dialog.dismiss()
            }
        dialogBuilder.create().show()
    }

    /**
     * Started point to run the verification.
     * We use a Scheduled worker to give enough time to the onForeground callback
     * to be executed in order to have a valid context/activity
     */
    fun runVerifyIntegration() {
        try {
            if (verificationStatus != VerificationStatus.NOT_STARTED) {
                // if verification has started do nothing
                return
            }
            val worker = ScheduledWorker.ofSingleThread("Automatic Verification")
            val runVerification = {
                startVerification()
            }
            worker.scheduleWithDelay(
                runVerification,
                VERIFY_INTEGRATION_DELAY,
                TimeUnit.MILLISECONDS
            )
        } catch (e: RejectedExecutionException) {
            logger.logError("Integration Verification - start verification rejected", e)
            verificationStatus = VerificationStatus.ERROR
            logger.logInfo("Integration Verification - Verification status: $verificationStatus")
        }
    }

    /**
     * Starts the verification to send:
     *  - http request
     *  - error log
     *  - starts and ends a Custom moment
     *  - throw an Exception
     */
    @VisibleForTesting
    fun executeVerificationSteps() {
        logger.logInfo("Integration Verification - Verification status: $verificationStatus")
        sendNetworkHttpRequest()
        createErrorLog()
        createMoment()
        throwAnException()
    }

    @VisibleForTesting
    fun startVerification() {
        val activity = activityService.foregroundActivity.get()
        try {
            if (automaticVerificationChecker.createFile(activity)) {
                verificationStatus = VerificationStatus.IN_PROGRESS
                activity.runOnUiThread {
                    Toast.makeText(
                        activity,
                        activity.getString(R.string.automatic_verification_started),
                        Toast.LENGTH_SHORT
                    ).show()
                }
                executeVerificationSteps()
            }
        } catch (e: IOException) {
            logger.logError("Integration Verification - I/O exception", e)
            verificationStatus = VerificationStatus.ERROR
            logger.logInfo("Integration Verification - Verification status: $verificationStatus")
        }
    }

    @VisibleForTesting
    fun sendNetworkHttpRequest() {
        try {
            val worker = BackgroundWorker.ofSingleThread("Http Request")
            worker.submit {
                logger.logInfo("Integration Verification - Http request")
                val connection = ApiRequest.newBuilder()
                    .withHttpMethod(HttpMethod.GET)
                    .withUrl(Unchecked.wrap<EmbraceUrl> { EmbraceUrl.getUrl("https://httpbin.org/get") })
                    .build()
                    .toConnection()

                connection.setConnectTimeout(NETWORK_REQUEST_TIMEOUT)
                connection.connect()
                val result = connection.responseCode
                logger.logInfo("Integration Verification - Http response: $result")
            }
        } catch (e: RejectedExecutionException) {
            verificationStatus = VerificationStatus.ERROR
            logger.logError("Integration Verification - Http request rejected", e)
            logger.logInfo("Integration Verification - Verification status: $verificationStatus")
        }
    }

    @VisibleForTesting
    fun createErrorLog() {
        val exception = VerifyIntegrationException("Verify Integration log error exception")
        val exProperties = HashMap<String, Any>()
        exProperties["key1"] = "Value1"
        embraceInstance.logError(exception, exProperties)
        logger.logInfo("Integration Verification - Send error log")
    }

    @VisibleForTesting
    fun createMoment() {
        val properties: MutableMap<String, Any> = HashMap()
        properties["testKey"] = "verifyIntegration"
        val momentName = "Verify Integration Moment"
        val momentIdentifier = "Verify Integration identifier"

        embraceInstance
            .startEvent(momentName, momentIdentifier, false, properties)
        embraceInstance.endEvent(momentName, momentIdentifier)
        logger.logInfo("Integration Verification - Send a moment")
    }

    @VisibleForTesting
    fun throwAnException() {
        val looper = Looper.getMainLooper()
        val handler = Handler(looper)

        val sendCrash = Runnable {
            sendCrash()
        }

        handler.postDelayed(sendCrash, THROW_EXCEPTION_DELAY)
    }

    @VisibleForTesting
    fun sendCrash() {
        logger.logInfo("Integration Verification - throw an Exception")
        throw VerifyIntegrationException("Forced Exception to verify integration")
    }

    @VisibleForTesting
    fun runEndSession() {
        embraceInstance.endSession()
        logger.logInfo("Integration Verification - End session manually")
    }

    override fun onForeground(coldStart: Boolean, startupTime: Long) {
        val activity = activityService.foregroundActivity.get()
        val fromVerification = activity.intent.getBooleanExtra("from_verification", false)
        val fromVerificationWithError =
            activity.intent.getStringExtra("verification_status")

        when {
            fromVerificationWithError.toString() == VerificationStatus.ERROR.toString() -> {
                val endSession = Runnable {
                    runEndSession()
                    logger.logInfo("Integration Verification - Verification status: $verificationStatus")
                    showDialogWithError()
                }
                handler.postDelayed(endSession, ON_FOREGROUND_DELAY)
            }
            fromVerification -> {
                val endSession = Runnable {
                    runEndSession()
                    val currentContext = activityService.foregroundActivity.get()
                    verificationStatus = VerificationStatus.FINISHED
                    logger.logInfo("Integration Verification - Verification status: $verificationStatus")
                    showSuccessDialog(currentContext)
                }

                // End session manually after 5 secs
                handler.postDelayed(endSession, ON_FOREGROUND_DELAY)
            }
            else -> logger.logInfo("Integration Verification - on foreground")
        }
    }

    /**
     * Restarts the app after a forced VerifyIntegrationException
     * was captured as part of the automatic verification
     */
    fun restartAppFromPendingIntent() {
        val exitStatus = 2
        val activity = activityService.foregroundActivity.get()
        val intent = activity.intent
            .addFlags(
                Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
            ).putExtra("from_verification", true)
            .putExtra("verification_status", verificationStatus.toString())

        with(activity) {
            finish()
            startActivity(intent)
        }
        exitProcess(exitStatus)
    }

    fun verifyIntegration() {
        instance.embraceInstance = Embrace.getInstance()
        instance.setActivityListener()
        instance.setAutomaticVerificationCheckerInstance(AutomaticVerificationChecker())
        instance.runVerifyIntegration()
    }

    companion object {
        private const val THROW_EXCEPTION_DELAY = 2000L
        private const val ON_FOREGROUND_DELAY = 5000L
        private const val VERIFY_INTEGRATION_DELAY = 200L
        private const val NETWORK_REQUEST_TIMEOUT = 500

        private val instance = EmbraceAutomaticVerification()
    }

    /**
     * Exception Handler that verifies if a VerifyIntegrationException was received,
     * in order to execute restartAppFromPendingIntent
     */
    internal class AutomaticVerificationExceptionHandler constructor(
        private val defaultHandler: Thread.UncaughtExceptionHandler?
    ) :
        Thread.UncaughtExceptionHandler {
        override fun uncaughtException(thread: Thread, exception: Throwable) {
            if (exception.cause?.cause?.javaClass == VerifyIntegrationException::class.java) {
                instance.restartAppFromPendingIntent()
            }
            InternalStaticEmbraceLogger.logDebug(
                "Finished handling exception. Delegating to default handler.",
                exception
            )
            defaultHandler?.uncaughtException(thread, exception)
        }
    }
}

internal class VerifyIntegrationException(message: String) : Exception(message)

@InternalApi
public enum class VerificationStatus {
    NOT_STARTED,
    IN_PROGRESS,
    FINISHED,
    ERROR
}

internal class AutomaticVerificationChecker {
    private val fileName = "emb_marker_file.txt"
    private lateinit var file: File

    fun createFile(activity: Activity): Boolean {
        val directory = activity.cacheDir.absolutePath
        file = File("$directory/$fileName")

        return generateMarkerFile()
    }

    /**
     * Verifies if the marker file exists. It is used to determine whether to run the verification or no.
     * If marker file does not exist, then we have to run the automatic verification,
     * on the other hand, if the marker file exists,
     * it means that the verification was executed before and it shouldn't run again
     *
     * @return true if marker file does not exist, otherwise returns false
     */
    private fun generateMarkerFile(): Boolean {
        var result = false
        if (!file.exists()) {
            result = file.createNewFile()
        }

        return result
    }

    fun deleteFile() {
        if (file.exists() && !file.isDirectory) {
            file.delete()
        }
    }
}
