package io.embrace.android.embracesdk;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import io.embrace.android.embracesdk.Config.StartupSamplingConfig;
import io.embrace.android.embracesdk.config.AnrConfig;
import io.embrace.android.embracesdk.networking.EmbraceUrl;
import io.embrace.android.embracesdk.networking.EmbraceUrlAdapter;
import io.embrace.android.embracesdk.utils.optional.Optional;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;
import kotlin.Lazy;
import kotlin.LazyKt;

/**
 * Handles the reading and writing of objects from the app's cache.
 */
class EmbraceCacheService implements CacheService {

    private static final String EMBRACE_PREFIX = "emb_";
    private static final String EMB_ANR_CONFIG_INI = "emb_anr_config.ini";
    private static final String EMB_STARTUP_SAMPLING_CONFIG_INI = "emb_startup_sampling_config.ini";

    private final Lazy<Gson> gson = LazyKt.lazy(EmbraceCacheService::createGson);
    private final Lazy<File> cacheDir;

    private final InternalEmbraceLogger logger;

    EmbraceCacheService(Context context,
                        InternalEmbraceLogger logger) {
        Preconditions.checkNotNull(context, "context must not be null");
        this.logger = logger;
        cacheDir = LazyKt.lazy(context::getCacheDir);
    }

    /**
     * Writes a file to the cache. Must be serializable by GSON.
     * <p>
     * If writing the object to the cache fails, an exception is logged.
     *
     * @param name   the name of the object to write
     * @param object the object to write
     * @param clazz  the class of the object to write
     * @param <T>    the type of the object to write
     */
    @Override
    public <T> void cacheObject(String name, T object, Class<T> clazz) {
        logger.logDeveloper("EmbraceCacheService", "Attempting to cache object: " + name);
        BufferedWriter bw = null;
        FileWriter fw;
        File file = new File(cacheDir.getValue(), EMBRACE_PREFIX + name);
        try {
            fw = new FileWriter(file.getAbsoluteFile());
            bw = new BufferedWriter(fw);

            gson.getValue().toJson(object, clazz, new JsonWriter(bw));
            logger.logDeveloper("EmbraceCacheService", "Object cached");
        } catch (Exception ex) {
            logger.logDebug("Failed to store cache object " + file.getPath(), ex);
        } finally {
            // BufferedWriter closes the FileWriter it wraps, so no need to close fw
            try {
                if (bw != null) {
                    bw.close();
                }
            } catch (Exception ex) {
                logger.logDebug("Failed to close cache writer " + file.getPath(), ex);
            }
        }
    }

    @Override
    public <T> Optional<T> loadObject(String name, Class<T> clazz) {
        File file = new File(cacheDir.getValue(), EMBRACE_PREFIX + name);
        try (FileInputStream fileInputStream = new FileInputStream(file);
             BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(fileInputStream, Charset.forName("UTF-8")));
             JsonReader jsonreader = new JsonReader(bufferedReader)) {
            jsonreader.setLenient(true);

            T obj = gson.getValue().fromJson(jsonreader, clazz);
            if (obj != null) {
                return Optional.of(obj);
            } else {
                logger.logDeveloper("EmbraceCacheService", "Object " + name + " not found");
            }
        } catch (FileNotFoundException ex) {
            logger.logDebug("Cache file cannot be found " + file.getPath());
        } catch (Exception ex) {
            logger.logDebug("Failed to read cache object " + file.getPath(), ex);
        }
        return Optional.absent();
    }

    @Override
    public <T> List<T> loadObjectsByRegex(String regex, Class<T> clazz) {
        logger.logDeveloper("EmbraceCacheService", "Attempting to load object by regex: " + regex);
        Pattern pattern = Pattern.compile(regex);
        List<T> objects = new ArrayList<>();
        File[] filesInCache = cacheDir.getValue().listFiles();

        if (filesInCache != null) {
            for (File cache : filesInCache) {
                if (pattern.matcher(cache.getName()).find()) {
                    try (FileInputStream fileInputStream = new FileInputStream(cache);
                         BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(fileInputStream, Charset.forName("UTF-8")));
                         JsonReader jsonreader = new JsonReader(bufferedReader)) {
                        jsonreader.setLenient(true);
                        T obj = gson.getValue().fromJson(jsonreader, clazz);
                        if (obj != null) {
                            objects.add(obj);
                        } else {
                            logger.logDeveloper("EmbraceCacheService", "Objects not found by regex");
                        }
                    } catch (FileNotFoundException ex) {
                        logger.logDebug("Cache file cannot be found " + cache.getPath());
                    } catch (Exception ex) {
                        logger.logDebug("Failed to read cache object " + cache.getPath(), ex);
                    }
                } else {
                    logger.logDeveloper("EmbraceCacheService", "Objects not found by regex");
                }
            }
        } else {
            logger.logDeveloper("EmbraceCacheService", "There are not files in cache");
        }

        return objects;
    }

    @Override
    public boolean deleteObject(String name) {
        logger.logDeveloper("EmbraceCacheService", "Attempting to delete: " + name);
        File file = new File(cacheDir.getValue(), EMBRACE_PREFIX + name);
        try {
            return file.delete();
        } catch (Exception ex) {
            logger.logDebug("Failed to delete cache object " + file.getPath());
        }
        return false;
    }

    @Override
    public boolean deleteObjectsByRegex(String regex) {
        logger.logDeveloper("EmbraceCacheService", "Attempting to delete objects by regex: " + regex);
        Pattern pattern = Pattern.compile(regex);
        boolean result = false;
        File[] filesInCache = cacheDir.getValue().listFiles();

        if (filesInCache != null) {
            for (File cache : filesInCache) {
                if (pattern.matcher(cache.getName()).find()) {
                    try {
                        result = cache.delete();
                    } catch (Exception ex) {
                        logger.logDebug("Failed to delete cache object " + cache.getPath());
                    }
                } else {
                    logger.logDeveloper("EmbraceCacheService", "Objects not found by regex");
                }
            }
        } else {
            logger.logDeveloper("EmbraceCacheService", "There are not files in cache");
        }

        return result;
    }

    @Override
    public boolean moveObject(String src, String dst) {
        File cacheDir = this.cacheDir.getValue();
        File srcFile = new File(cacheDir, EMBRACE_PREFIX + src);

        if (!srcFile.exists()) {
            logger.logDeveloper("EmbraceCacheService", "Source file doesn't exist: " + src);
            return false;
        }

        File dstFile = new File(cacheDir, EMBRACE_PREFIX + dst);
        logger.logDeveloper("EmbraceCacheService", "Object moved from " + src + " to " + dst);
        return srcFile.renameTo(dstFile);
    }

    @Override
    public void cacheAnrConfig(@Nullable AnrConfig anrConfig, @NonNull Clock clock) {
        logger.logDeveloper("EmbraceCacheService", "Attempting to cache ANR config");
        if (anrConfig == null) {
            logger.logDeveloper("EmbraceCacheService", "ANR config not found");
            return;
        }
        File dst = new File(cacheDir.getValue(), EMB_ANR_CONFIG_INI);

        try (OutputStream stream = new FileOutputStream(dst)) {
            AnrConfigKeyValueWriter writer = new AnrConfigKeyValueWriter(clock, logger);
            writer.write(anrConfig, stream);
            logger.logDeveloper("EmbraceCacheService", "ANR config cached");
        } catch (Throwable exc) {
            logger.logWarning("Failed to cache ANR config", exc);
        }
    }

    @Override
    public void cacheStartupSamplingConfig(@Nullable StartupSamplingConfig startupSamplingConfig,
                                           @NonNull Clock clock) {
        logger.logDeveloper("EmbraceCacheService", "cacheStartupSamplingConfig");
        if (startupSamplingConfig == null) {
            logger.logDeveloper("EmbraceCacheService", "startupSamplingConfig is NULL");
            return;
        }
        File dst = new File(cacheDir.getValue(), EMB_STARTUP_SAMPLING_CONFIG_INI);

        try (OutputStream stream = new FileOutputStream(dst)) {
            StartupSamplingConfigKeyValueWriter writer = new StartupSamplingConfigKeyValueWriter(clock, logger);
            writer.write(startupSamplingConfig, stream);
            logger.logDeveloper("EmbraceCacheService", "cacheStartupSamplingConfig success");
        } catch (Throwable exc) {
            logger.logWarning("Failed to cache startup sampling config", exc);
        }
    }

    @Override
    @Nullable
    public AnrConfig loadAnrConfig(@NonNull Clock clock) {
        logger.logDeveloper("EmbraceCacheService", "loadAnrConfig");
        File src = new File(cacheDir.getValue(), EMB_ANR_CONFIG_INI);
        try (InputStream stream = new FileInputStream(src)) {
            AnrConfigKeyValueReader reader = new AnrConfigKeyValueReader(logger);
            AnrConfig anrConfig = reader.read(stream);

            if (anrConfig == null) {
                logger.logDeveloper("EmbraceCacheService", "server ANR config is null, getting default");
                // server cfg hasn't overridden anything or hasn't been persisted yet - use the default.
                return AnrConfig.ofDefault();
            } else {
                logger.logDeveloper("EmbraceCacheService", "ANR config loaded");
                return anrConfig;
            }
        } catch (Throwable exc) {
            if (!(exc instanceof FileNotFoundException)) {
                logger.logWarning("Failed to load ANR config", exc);
            }
            return null;
        }
    }

    @Override
    @Nullable
    public StartupSamplingConfig loadStartupSamplingConfig(@NonNull Clock clock) {
        logger.logDeveloper("EmbraceCacheService", "loadStartupSamplingConfig");
        File src = new File(cacheDir.getValue(), EMB_STARTUP_SAMPLING_CONFIG_INI);
        try (InputStream stream = new FileInputStream(src)) {
            StartupSamplingConfigKeyValueReader reader = new StartupSamplingConfigKeyValueReader(logger);
            StartupSamplingConfig config = reader.read(stream);

            if (config == null) {
                logger.logDeveloper("EmbraceCacheService", "server StartupSampling config is null, getting default");
                // server cfg hasn't overridden anything or hasn't been persisted yet - use the default.
                return StartupSamplingConfig.ofDefault();
            } else {
                logger.logDeveloper("EmbraceCacheService", "StartupSampling config loaded");
                return config;
            }
        } catch (Throwable exc) {
            if (!(exc instanceof FileNotFoundException)) {
                logger.logWarning("Failed to load startup sampling config", exc);
            }
            return null;
        }
    }

    private static Gson createGson() {
        return new GsonBuilder()
                .registerTypeAdapter(EmbraceUrl.class, new EmbraceUrlAdapter())
                .create();
    }
}
