package io.embrace.android.embracesdk;

import androidx.annotation.VisibleForTesting;

import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.optional.Optional;

/**
 * Loads configuration for the app from the Embrace API.
 */
final class EmbraceConfigService implements ConfigService, ActivityListener {

    /**
     * Config lives for 1 hour before attempting to retrieve again.
     */
    private static final long CONFIG_TTL = 60 * 60 * 1000L;

    /**
     * Name of the file used to cache config values.
     */
    private static final String CONFIG_FILE_NAME = "config.json";

    /**
     * Config refresh default retry period.
     */
    private static final long DEFAULT_RETRY_WAIT_TIME = 2; // 2 seconds

    /**
     * Config max allowed refresh retry period.
     */
    private static final long MAX_ALLOWED_RETRY_WAIT_TIME = 300; // 5 minutes

    /**
     * The listeners subscribed to configuration changes.
     */
    private final Set<ConfigListener> listeners = new CopyOnWriteArraySet<>();
    private final BackgroundWorker bgWorker;
    private final LocalConfig localConfig;
    private final ApiClient apiClient;
    private final CacheService cacheService;
    private final MetadataService metadataService;
    private final PreferencesService preferencesService;
    private final Clock clock;
    private final Object lock = new Object();

    volatile Config config = Config.ofDefault();
    private volatile long lastUpdated;
    private volatile long lastRefreshConfigAttempt;
    private volatile double configRetrySafeWindow = DEFAULT_RETRY_WAIT_TIME;

    private static final long STARTUP_FETCH_DELAY_MS = 500;

    private final InternalEmbraceLogger logger;
    private final PatternCache patternCache = new PatternCache();

    EmbraceConfigService(
            LocalConfig localConfig,
            ApiClient apiClient,
            ActivityService activityService,
            CacheService cacheService,
            MetadataService metadataService,
            PreferencesService preferencesService,
            Clock clock,
            InternalEmbraceLogger logger,
            BackgroundWorker bgRegistrationWorker) {
        this.localConfig = localConfig;
        this.apiClient = Preconditions.checkNotNull(apiClient, "apiClient must not be null");
        this.cacheService = Preconditions.checkNotNull(cacheService, "cacheService must not be null");
        this.preferencesService = Preconditions.checkNotNull(preferencesService, "preferenceService must not be null");
        this.clock = Preconditions.checkNotNull(clock, "clock must not be null");
        this.metadataService = Preconditions.checkNotNull(metadataService, "metadataService must not be null");
        this.bgWorker = Preconditions.checkNotNull(bgRegistrationWorker, "BackgroundWorker must not be null");
        this.logger = logger;
        Preconditions.checkNotNull(activityService).addListener(this);

        performInitialConfigLoad();
        attemptConfigRefresh();
    }

    /**
     * Schedule an action that loads the config from the cache.
     * This is deferred to lessen it´s impact upon startup.
     */
    private void performInitialConfigLoad() {
        logger.logDeveloper("EmbraceConfigService", "performInitialConfigLoad");
        try {
            bgWorker.submit(() -> {
                loadConfigFromCache();
                return null;
            });
        }
        catch (RejectedExecutionException ex) {
            logger.logDebug("Failed to schedule initial config load from cache.", ex);
        }
    }

    /**
     * Load Config from cache if present.
     */
    @VisibleForTesting
    void loadConfigFromCache() {
        logger.logDeveloper("EmbraceConfigService", "Attempting to load config from cache");
        Optional<Config> optionalConfig = cacheService.loadObject(CONFIG_FILE_NAME, Config.class);
        if (optionalConfig.isPresent()) {
            config = optionalConfig.get();
            logger.logDeveloper("EmbraceConfigService", "Loaded config from cache");
        } else {
            logger.logDeveloper("EmbraceConfigService", "config not found in local cache");
        }
    }

    @Override
    public Config getConfig() {
        attemptConfigRefresh();
        return config;
    }

    private void attemptConfigRefresh() {
        if (configRequiresRefresh() && configRetryIsSafe()) {
            logger.logDeveloper("EmbraceConfigService", "Attempting to update config");
            // Attempt to asynchronously update the config if it is out of date
            refreshConfig();
        }
    }

    private void refreshConfig() {
        logger.logDeveloper("EmbraceConfigService", "Attempting to refresh config");

        Config previousConfig = config;
        bgWorker.submit((Callable<Object>) () -> {
            synchronized (lock) {
                logger.logDeveloper("EmbraceConfigService", "Updating config in background thread");

                // Ensure that another thread didn't refresh it already in the meantime
                if (configRequiresRefresh() && configRetryIsSafe()) {
                    try {
                        lastRefreshConfigAttempt = clock.now();
                        config = apiClient.getConfig().get();
                        persistConfig();

                        lastUpdated = clock.now();
                        if (!config.equals(previousConfig)) {
                            logger.logDeveloper("EmbraceConfigService", "Notify listeners about new config");
                            // Only notify listeners if the config has actually changed value
                            notifyListeners(previousConfig, config);
                        }
                        configRetrySafeWindow = DEFAULT_RETRY_WAIT_TIME;
                        logger.logDeveloper("EmbraceConfigService", "Config updated");
                    } catch (Exception ex) {
                        configRetrySafeWindow = Math.min(MAX_ALLOWED_RETRY_WAIT_TIME, configRetrySafeWindow * 2);
                        logger.logWarning("Failed to load SDK config from the server. " +
                                "Trying again in " + configRetrySafeWindow + " seconds.");
                    }
                }
                return config;
            }
        });
    }

    private void persistConfig() {
        logger.logDeveloper("EmbraceConfigService", "persistConfig");
        preferencesService.setSDKDisabled(computeIfSDKIsDisabled(config));
        cacheService.cacheObject(CONFIG_FILE_NAME, config, Config.class);
        cacheService.cacheAnrConfig(config.getAnrConfig(), clock);
        cacheService.cacheStartupSamplingConfig(config.getStartupSamplingConfig(), clock);
    }

    @Override
    public boolean isScreenshotDisabledForEvent(String eventName) {
        return patternCache.doesStringMatchesPatternInSet(eventName, config.getDisabledScreenshotPatterns());
    }

    @Override
    public boolean isEventDisabled(String eventName) {
        boolean isEventDisabled = patternCache.doesStringMatchesPatternInSet(eventName, config.getDisabledEventAndLogPatterns());
        logger.logDeveloper("EmbraceConfigService", "Event: " + eventName + "isEventDisabled: " + isEventDisabled);
        return isEventDisabled;
    }

    @Override
    public boolean isLogMessageDisabled(String logMessage) {
        return patternCache.doesStringMatchesPatternInSet(logMessage, config.getDisabledEventAndLogPatterns());
    }

    @Override
    public boolean isMessageTypeDisabled(MessageType type) {
        return config.getDisabledMessageTypes().contains(type.name().toLowerCase());
    }

    @Override
    public boolean isUrlDisabled(String url) {
        return patternCache.doesStringMatchesPatternInSet(url, config.getDisabledUrlPatterns());
    }

    @Override
    public boolean isInternalExceptionCaptureEnabled() {
        return config.getInternalExceptionCaptureEnabled();
    }

    @Override
    public boolean isSdkDisabled() {
        return preferencesService.getSDKDisabled();
    }

    @Override
    public boolean isAnrCaptureEnabled() {
        return isBehaviorEnabled(getConfig().getAnrUsersEnabledPercentage());
    }

    @Override
    public boolean isUnityNdkSamplingEnabled() {
        return isBehaviorEnabled(getConfig().getAnrConfig().getPctUnityNdkSamplingEnabled());
    }

    @Override
    public boolean isBgAnrCaptureEnabled() {
        return isBehaviorEnabled(getConfig().getAnrBgUsersEnabledPercentage());
    }

    @Override
    public boolean isBetaFeaturesEnabled() {
        return localConfig.getConfigurations().isBetaFeaturesEnabled()
                && isBehaviorEnabled(getConfig().getDefaultBetaFeaturesPct());
    }

    @Override
    public boolean isGoogleAnrCaptureEnabled() {
        return localConfig.getConfigurations().getAnr().getCaptureGoogle() &&
                isBehaviorEnabled(getConfig().getAnrUsersGoogleEnabledPercentage());
    }

    @Override
    public boolean isSigHandlerDetectionEnabled() {
        boolean enabled = localConfig.getConfigurations().isSigHandlerDetection();
        return enabled && getConfig().getKillSwitchConfig().isSigHandlerDetectionEnabled();
    }

    public void addListener(ConfigListener configListener) {
        listeners.add(configListener);
    }

    @Override
    public void removeListener(ConfigListener configListener) {
        listeners.remove(configListener);
    }

    @Override
    public boolean isSessionControlEnabled() {
        return config.getSessionControl();
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        // Refresh the config on resume if it has expired
        getConfig();
        if (Embrace.getInstance().isStarted() && isSdkDisabled()) {
            logger.logInfo("Embrace SDK disabled by config");
            Embrace.getInstance().stop();
        }
    }

    /**
     * Notifies the listeners that a new config was fetched from the server.
     *
     * @param previousConfig old config
     * @param newConfig      new config
     */
    private void notifyListeners(Config previousConfig, Config newConfig) {
        //TODO: future: use a repository for store the config
        StreamUtilsKt.stream(listeners, listener -> {
            try {
                listener.onConfigChange(previousConfig, newConfig);
            } catch (Exception ex) {
                logger.logDebug("Failed to notify ConfigListener", ex);
            }
            return null;
        });
    }

    /**
     * Checks if the time diff since the last fetch exceeds the
     * {@link EmbraceConfigService#CONFIG_TTL} millis.
     *
     * @return if the config requires to be fetched from the remote server again or not.
     */
    private boolean configRequiresRefresh() {
        return clock.now() - lastUpdated > CONFIG_TTL;
    }

    /**
     * Checks if the time diff since the last attempt is enough to try again.
     *
     * @return if the config can be fetched from the remote server again or not.
     */
    private boolean configRetryIsSafe() {
        return clock.now() > (lastRefreshConfigAttempt + configRetrySafeWindow * 1000);
    }

    /**
     * Determines whether behaviour is enabled for a percentage roll-out. This is achieved
     * by taking a normalized hex value from the last 6 digits of the device ID, and comparing
     * it against the enabled percentage. This ensures that devices are consistently in a given
     * group for beta functionality.
     * <p>
     * The normalized device ID has 16^6 possibilities (roughly 1.6m) which should be sufficient
     * granularity for our needs.
     *
     * @param pctEnabled the % enabled for a given config value. This should be a float rather than
     *                   an integer for maximum granularity.
     * @return whether the behaviour is enabled or not.
     */
    public boolean isBehaviorEnabled(float pctEnabled) {
        if (pctEnabled <= 0 || pctEnabled > 100) {
            logger.logDeveloper("EmbraceConfigService", "behaviour disabled");
            return false;
        }
        float deviceId = getNormalizedLargeDeviceId();
        return pctEnabled >= deviceId;
    }

    /**
     * Use {@link #isBehaviorEnabled(float)} instead as it allows rollouts to be controlled
     * at greater granularity.
     */
    @Deprecated()
    float getNormalizedDeviceId() {
        return getNormalizedDeviceId(2);
    }

    float getNormalizedLargeDeviceId() {
        return getNormalizedDeviceId(6);
    }

    private float getNormalizedDeviceId(int bits) {
        String deviceId = metadataService.getDeviceId();
        String finalChars = deviceId.substring(deviceId.length() - bits);

        // Normalize the device ID to a value between 0-100.
        int radix = 16;
        int space = (int) (Math.pow(radix, bits) - 1);
        Integer hexValue = Integer.valueOf(finalChars, radix);
        float normalizedDeviceId = (((float) hexValue / space) * 100);
        logger.logDeveloper("EmbraceConfigService", "normalizedDeviceId: " + normalizedDeviceId);
        return normalizedDeviceId;
    }

    /**
     * Given a Config instance, computes if the SDK is disabled based on the threshold and the offset.
     *
     * @return true if the sdk is disabled, false otherwise
     */
    private boolean computeIfSDKIsDisabled(Config sdkConfig) {
        float result = getNormalizedDeviceId();
        //Check if this is lower than the threshold, to determine whether
        // we should enable/disable the SDK.
        int lowerBound = Math.max(0, sdkConfig.getOffset());
        int upperBound = Math.min(sdkConfig.getOffset() + sdkConfig.getThreshold(), 100);
        return lowerBound == upperBound || result < lowerBound || result > upperBound;
    }

    @Override
    public void close() {
        logger.logDebug("Shutting down EmbraceConfigService");
    }
}
