package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Handles the lifecycle of events (moments).
 * <p>
 * An event is started, timed, and then ended. If the event takes longer than a specified period of
 * time, then the event is considered late, and a screenshot is taken.
 */
final class EmbraceEventService implements EventService, ActivityListener, MemoryCleanerListener {
    public static final String STARTUP_EVENT_NAME = "_startup";

    public static final boolean ALLOW_SCREENSHOT = false;

    private final LocalConfig localConfig;
    private final InternalEmbraceLogger logger;
    private final Clock clock;

    /**
     * Timeseries of event IDs, keyed on the start time of the event.
     */
    private final NavigableMap<Long, String> eventIds = new ConcurrentSkipListMap<>();
    /**
     * Map of active events, keyed on their event ID (event name + identifier).
     */
    @VisibleForTesting
    final ConcurrentMap<String, EventDescription> activeEvents = new ConcurrentHashMap<>();
    /**
     * Session properties
     */
    private final EmbraceSessionProperties sessionProperties;

    private final long startupStartTime;

    private StartupEventInfo startupEventInfo;

    private boolean startupSent = false;

    @VisibleForTesting
    EventHandler eventHandler;

    public EmbraceEventService(
            long startTime,
            ApiClient apiClient,
            ConfigService configService,
            LocalConfig localConfig,
            MetadataService metadataService,
            PerformanceInfoService performanceInfoService,
            UserService userService,
            ScreenshotService screenshotService,
            EmbraceActivityService activityService,
            MemoryCleanerService memoryCleanerService,
            GatingService gatingService,
            EmbraceSessionProperties sessionProperties,
            InternalEmbraceLogger logger,
            Clock clock) {

        this.startupStartTime = startTime;
        Preconditions.checkNotNull(apiClient);
        Preconditions.checkNotNull(configService);
        this.localConfig = Preconditions.checkNotNull(localConfig);
        Preconditions.checkNotNull(metadataService);
        Preconditions.checkNotNull(performanceInfoService);
        Preconditions.checkNotNull(userService);
        Preconditions.checkNotNull(screenshotService);
        Preconditions.checkNotNull(gatingService);
        this.clock = Preconditions.checkNotNull(clock);

        Preconditions.checkNotNull(activityService).addListener(this);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);

        // Session properties
        this.sessionProperties = Preconditions.checkNotNull(sessionProperties);
        this.logger = logger;
        this.eventHandler = new EventHandler(
                metadataService,
                configService,
                userService,
                screenshotService,
                performanceInfoService,
                gatingService,
                apiClient,
                logger,
                clock
        );
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "coldStart: " + coldStart);

        if (coldStart) {
            // Using the system current timestamp here as the startup timestamp is related to the
            // the actual SDK starts ( when the app context starts ). The app context can start
            // in the background, registering a startup time that will later be sent with the
            // app coming to foreground, resulting in a *pretty* long startup moment.
            sendStartupMoment();
        }
    }

    @Override
    public void applicationStartupComplete() {
        if (localConfig.getConfigurations().getStartupMoment().getAutomaticallyEnd()) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Automatically ending startup event");
            endEvent(STARTUP_EVENT_NAME);
        } else {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Application startup automatically end is disabled");
        }
    }

    @Override
    public void sendStartupMoment() {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "sendStartupMoment");

        synchronized (this) {
            if (startupSent) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Startup is already sent");
                return;
            }
            startupSent = true;
        }

        logger.logDebug("Sending startup start event.");
        startEvent(
                STARTUP_EVENT_NAME,
                null,
                localConfig.getConfigurations().getStartupMoment().getTakeScreenshot(),
                null,
                startupStartTime);
    }

    @Override
    public void startEvent(String name) {
        //extract constant
        startEvent(name, null, ALLOW_SCREENSHOT, null, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier) {
        startEvent(name, identifier, ALLOW_SCREENSHOT, null, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier, boolean allowScreenshot) {
        startEvent(name, identifier, allowScreenshot, null, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        startEvent(name, identifier, ALLOW_SCREENSHOT, properties, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier, boolean allowScreenshot, @Nullable Map<String, Object> properties) {
        startEvent(name, identifier, allowScreenshot, properties, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier, @Nullable boolean allowScreenshot, @Nullable Map<String, Object> properties, @Nullable Long startTime) {
        try {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Start event: " + name);

            if (!eventHandler.isAllowedToStart(name)) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Event handler not allowed to start ");
                return;
            }

            String eventKey = EventHandler.getInternalEventKey(name, identifier);
            if (activeEvents.containsKey(eventKey)) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Ending previous event with same name");
                endEvent(name, identifier, false, null);
            }

            long now = clock.now();
            if (startTime == null) {
                startTime = now;
            }

            String eventId = Uuid.getEmbUuid();
            eventIds.put(now, eventId);

            EventDescription eventDescription = eventHandler.onEventStarted(
                    eventId,
                    name,
                    startTime,
                    allowScreenshot,
                    sessionProperties,
                    properties,
                    () -> endEvent(name, identifier, true, null)
            );

            // event started, update active events
            activeEvents.put(eventKey, eventDescription);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Event started : " + name);

        } catch (Exception ex) {
            logger.logError(
                    "Cannot start event with name: " + name + ", identifier: " + identifier + " due to an exception",
                    ex, false);
        }
    }

    @Override
    public void endEvent(@NonNull String name) {
        endEvent(name, null, false, null);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable String identifier) {
        endEvent(name, identifier, false, null);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable Map<String, Object> properties) {
        endEvent(name, null, false, properties);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        endEvent(name, identifier, false, properties);
    }

    private void endEvent(@NonNull String name, @Nullable String identifier, boolean late, @Nullable Map<String, Object> properties) {
        try {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Ending event: " + name);

            if (!eventHandler.isAllowedToEnd(name)){
                InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Event handler not allowed to end");
                return;
            }

            String eventKey = EventHandler.getInternalEventKey(name, identifier);

            EventDescription originEventDescription;
            if (late) {
                originEventDescription = activeEvents.get(eventKey);
            } else {
                originEventDescription = activeEvents.remove(eventKey);
            }

            if (originEventDescription == null) {
                // We avoid logging that there's no startup event in the activeEvents collection
                // as the user might have completed it manually on a @StartupActivity.
                if (!EventHandler.isStartupEvent(name)) {
                    logger.logError(
                            "No start event found when ending an event with name: " + name + ", identifier: " + identifier);
                }
                return;
            }

            EventMessage endEventMessage = eventHandler.onEventEnded(originEventDescription, late, properties, sessionProperties);
            if (EventHandler.isStartupEvent(name)) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "Ending Startup Ending");

                startupEventInfo = eventHandler.buildStartupEventInfo(
                        originEventDescription.getEvent(),
                        endEventMessage.getEvent());
            }
        } catch (Exception ex) {
            logger.logError(
                    "Cannot end event with name: " + name + ", identifier: " + identifier + " due to an exception",
                    ex);
        }
    }

    @Override
    public List<String> findEventIdsForSession(long startTime, long endTime) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "findEventIdsForSession");
        return new ArrayList<>(this.eventIds.subMap(startTime, endTime).values());
    }

    @Override
    public List<String> getActiveEventIds() {
        List<String> ids = new ArrayList<>();
        StreamUtilsKt.stream(activeEvents.values(), eventDescription -> {
            ids.add(eventDescription.getEvent().getEventId());
            return null;
        });
        return ids;
    }

    @Override
    public StartupEventInfo getStartupMomentInfo() {
        return startupEventInfo;
    }

    @Override
    public void close() {
        eventHandler.onClose();
        InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "close");
    }

    @Override
    public void cleanCollections() {
        this.eventIds.clear();
        this.activeEvents.clear();
        InternalStaticEmbraceLogger.logDeveloper("EmbraceEventService", "collections cleaned");
    }
}
