package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.EmbraceSessionService.APPLICATION_STATE_ACTIVE;
import static io.embrace.android.embracesdk.EmbraceSessionService.APPLICATION_STATE_BACKGROUND;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Intercepts Embrace SDK's exceptions errors and forwards them to the Embrace API.
 */
final class EmbraceExceptionService {

    private final ActivityService activityService;

    private final Clock clock;

    private ConfigService configService;

    private ExceptionError exceptionError;


    // ignore network-related exceptions since they are expected
    private static final Class[] ignoredExceptionClasses = new Class[]{
                java.net.BindException.class,
                java.net.ConnectException.class,
                java.net.HttpRetryException.class,
                java.net.NoRouteToHostException.class,
                java.net.PortUnreachableException.class,
                java.net.ProtocolException.class,
                java.net.SocketException.class,
                java.net.SocketTimeoutException.class,
                java.net.UnknownHostException.class,
                java.net.UnknownServiceException.class,
    };
    private Set ignoredExceptions;
    private Set ignoredExceptionStrings;
    private final Boolean logStrictMode;

    EmbraceExceptionService(ActivityService activityService, Clock clock, boolean logStrictMode) {
        this.activityService = Preconditions.checkNotNull(activityService);
        this.clock = Preconditions.checkNotNull(clock);
        this.configService = null;
        this.logStrictMode = logStrictMode;
    }

    public void setConfigService(ConfigService configService) {
        this.configService = configService;
    }

    private boolean ignoreThrowableCause(Throwable throwable, HashSet<Throwable> capturedThrowable) {
        if (throwable != null) {
            if (ignoredExceptions.contains(throwable.getClass())) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceExceptionService", "Exception ignored: " + throwable.getClass());
                return true;
            } else {
                /* if Hashset#add returns true means that the throwable was properly added,
                if it returns false, the object already exists in the set so we return false
                because we are in presence of a cycle in the Throwable cause */
                boolean addResult = capturedThrowable.add(throwable);
                return addResult && ignoreThrowableCause(throwable.getCause(), capturedThrowable);
            }
        }

        return false;
    }

    synchronized void handleExceptionError(Throwable throwable) {
        // Lazy convert the ignoredExceptions to speed up initialization time;
        InternalStaticEmbraceLogger.logDebug("ignoreThrowableCause - handleExceptionError");
        if (ignoredExceptions == null) {
            ignoredExceptions = new HashSet<>(Arrays.asList(ignoredExceptionClasses));
        }
        if (ignoredExceptions.contains(throwable.getClass())) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceExceptionService", "Exception ignored: " + throwable.getClass());
            return;
        } else {
            HashSet<Throwable> capturedThrowable = new HashSet<>();
            if (ignoreThrowableCause(throwable.getCause(), capturedThrowable)) {
                return;
            }
        }

        // If the exception has been wrapped in another exception, the ignored exception name will
        // show up as the start of the message, delimited by a semicolon.
        String message = throwable.getMessage();

        // Lazy convert the ignoredExceptionStrings to speed up initialization time;
        if (ignoredExceptionStrings == null) {
            ignoredExceptionStrings = new HashSet<>();

            for (Class<?> ignoredException : ignoredExceptionClasses) {
                ignoredExceptionStrings.add(ignoredException.getName());
            }
        }
        if (message != null && ignoredExceptionStrings.contains(message.split(":")[0])) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceExceptionService", "Ignored exception: " + throwable);
            return;
        }

        if (exceptionError == null) {
            exceptionError = new ExceptionError(logStrictMode);
        }

        // if the config service has not been set yet, capture the exception
        if ((configService == null) || configService.isInternalExceptionCaptureEnabled()) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceExceptionService", "Capturing exception, config service is not set yet: ", throwable);

            exceptionError.addException(
                        throwable,
                        getApplicationState(),
                        clock
            );
        }
    }

    private String getApplicationState() {
        if (activityService.isInBackground()) {
            return APPLICATION_STATE_BACKGROUND;
        } else {
            return APPLICATION_STATE_ACTIVE;
        }
    }

    ExceptionError getCurrentExceptionError() {
        return exceptionError;
    }

    synchronized void resetExceptionErrorObject() {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceExceptionService", "Exception error = NULL");
        this.exceptionError = null;
    }
}
