package io.embrace.android.embracesdk

import androidx.annotation.VisibleForTesting
import io.embrace.android.embracesdk.config.GatingConfig
import io.embrace.android.embracesdk.config.GatingConfig.Companion.FULL_SESSION_CRASHES
import io.embrace.android.embracesdk.config.GatingConfig.Companion.FULL_SESSION_ERROR_LOGS
import io.embrace.android.embracesdk.config.GatingConfig.Companion.LOGS_INFO
import io.embrace.android.embracesdk.config.GatingConfig.Companion.LOGS_WARN
import io.embrace.android.embracesdk.config.GatingConfig.Companion.SESSION_MOMENTS
import io.embrace.android.embracesdk.config.GatingConfig.Companion.STARTUP_MOMENT
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logDeveloper

/**
 * Receives the local and remote config to build the Gating config and define the amount of
 * data (components) that the SDK sends to the backend as part of sessions or event messages.
 * The service is listening to the remote config changes and determines if the gating config should
 * be updated.
 * Event service, Session service and Crash service check if should gate data based on the gating config.
 * Also defines if a full session data should be sent under certain conditions based on configurations.
 */
internal class EmbraceGatingService(
    private val logger: InternalEmbraceLogger,
    localConfig: LocalConfig,
    remoteConfig: ConfigService
) :
    GatingService, ConfigListener {

    private var gatingConfig: GatingConfig

    init {
        gatingConfig = buildSessionGatingConfig(
            localConfig.configurations.sessionConfig.sessionComponents,
            localConfig.configurations.sessionConfig.getFullSessionEvents()
        )

        remoteConfig.addListener(this)
    }

    /**
     * Get the current Session Gating config
     */
    fun getGatingConfig() = gatingConfig

    /**
     * Update the gating config based on the new remote config received
     */
    private fun updateGatingConfig(
        components: Set<String>?,
        fullSessionEvents: Set<String>?
    ) =
        buildSessionGatingConfig(components, fullSessionEvents)

    /**
     * Build a new Session Gating config
     * @param components of the session that should be included in the payload
     * @param fullSessionEvents that should send a full session payload
     */
    private fun buildSessionGatingConfig(
        components: Set<String>?,
        fullSessionEvents: Set<String>?
    ) =
        GatingConfig(components, fullSessionEvents)

    override fun onConfigChange(previousConfig: Config, newConfig: Config) {
        logDeveloper("EmbraceGatingService", "Gating configuration changed")
        gatingConfig = updateGatingConfig(
            newConfig.sessionComponents,
            newConfig.fullSessionComponents
        )
    }

    override fun gateSessionMessage(sessionMessage: SessionMessage): SessionMessage {
        gatingConfig.components?.let { components ->
            logger.logDebug("Session gating feature enabled. Attempting to sanitize the session message")

            // check if the session has error logs IDs. If so, send the full session payload.
            if (sessionMessage.session.errorLogIds?.isNotEmpty() == true && shouldSendFullForErrorLog()) {
                logDeveloper(
                    "EmbraceGatingService",
                    "Error logs detected - Sending full session payload"
                )
                return sessionMessage
            }

            // check if the session has a crash report id. If so, send the full session payload.
            if (sessionMessage.session.crashReportId != null) {
                logDeveloper(
                    "EmbraceGatingService",
                    "Crash detected - Sending full session payload"
                )
                return sessionMessage
            }

            return SessionSanitizerFacade(sessionMessage, components).getSanitizedMessage()
        }

        logDeveloper("EmbraceGatingService", "Gating feature disabled")
        return sessionMessage
    }

    override fun gateEventMessage(eventMessage: EventMessage): EventMessage {
        gatingConfig.components?.let {
            logger.logDebug("Session gating feature enabled. Attempting to sanitize the event message")

            if (shouldSendFullMessage(eventMessage)) {
                logDeveloper(
                    "EmbraceGatingService",
                    "Crash or error detected - Sending full session payload"
                )
                return eventMessage
            }

            return EventSanitizerFacade(eventMessage, it).getSanitizedMessage()
        }

        logDeveloper("EmbraceGatingService", "Gating feature disabled")
        return eventMessage
    }

    private fun shouldSendFullMessage(eventMessage: EventMessage): Boolean {
        return (eventMessage.event.type == EmbraceEvent.Type.ERROR_LOG && shouldSendFullForErrorLog()) ||
            (eventMessage.event.type == EmbraceEvent.Type.CRASH && shouldSendFullForCrash())
    }

    /**
     * Check if should gate Moment based on gating config.
     */
    override fun shouldGateMoment() = gatingConfig.components?.let {
        !it.contains(SESSION_MOMENTS)
    } ?: false

    /**
     * Check if should gate Info Logs based on gating config.
     */
    override fun shouldGateInfoLog() = gatingConfig.components?.let {
        !it.contains(LOGS_INFO)
    } ?: false

    /**
     * Check if should gate Warning Logs based on gating config.
     */
    override fun shouldGateWarnLog() = gatingConfig.components?.let {
        !it.contains(LOGS_WARN)
    } ?: false

    /**
     * Check if should gate Startup moment based on gating config.
     */
    override fun shouldGateStartupMoment() = gatingConfig.components?.let {
        !it.contains(STARTUP_MOMENT)
    } ?: false

    @VisibleForTesting
    fun shouldSendFullForCrash() =
        gatingConfig.fullSessionEvents?.contains(FULL_SESSION_CRASHES) ?: false

    @VisibleForTesting
    fun shouldSendFullForErrorLog() =
        gatingConfig.fullSessionEvents?.contains(FULL_SESSION_ERROR_LOGS) ?: false
}
