package io.embrace.android.embracesdk;

import android.app.ActivityManager;
import android.content.Context;

import androidx.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.List;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Polls for the device's available and used memory.
 * <p>
 * Stores memory warnings when the {@link ActivityService} detects a memory trim event.
 */
class EmbraceMemoryService implements MemoryService, MemoryCleanerListener {

    private static final int BYTES_IN_MB = 1024 * 1024;

    private static final int MAX_CAPTURED_MEMORY_WARNINGS = 100;

    private final Runtime runtime = Runtime.getRuntime();

    private final NavigableMap<Long, MemorySample> memorySamples = new ConcurrentSkipListMap<>();

    private final long[] memoryTimestamps = new long[MAX_CAPTURED_MEMORY_WARNINGS];

    private int offset = 0;

    private final Clock clock;

    private final ScheduledWorker memoryWorker;

    public EmbraceMemoryService(Clock clock, ActivityManager activityManager, MemoryCleanerService memoryCleanerService, ScheduledWorker memoryWorker) {
        this.clock = clock;
        this.memoryWorker = memoryWorker;
        Preconditions.checkNotNull(activityManager, "activityManager must not be null");
        this.memoryWorker.scheduleAtFixedRate(() -> queryMemory(activityManager), 0, 2, TimeUnit.SECONDS);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    public static EmbraceMemoryService ofContext(Clock clock, Context context, MemoryCleanerService memoryCleanerService, ScheduledWorker memoryWorker) {
        return new EmbraceMemoryService(clock, (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE),
                memoryCleanerService, memoryWorker);
    }

    @VisibleForTesting
    void queryMemory(ActivityManager activityManager) {
        try {
            ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
            activityManager.getMemoryInfo(memoryInfo);
            long heapUsed = (runtime.totalMemory() - runtime.freeMemory()) / BYTES_IN_MB;
            long systemAvailable = memoryInfo.availMem / BYTES_IN_MB;
            long timestamp = clock.now();
            memorySamples.put(timestamp, new MemorySample(timestamp, heapUsed, systemAvailable));
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMemoryService", "Query memory: heapUsed: " + heapUsed + "MB / systemAvailable: " + systemAvailable + "MB / timestamp: " + timestamp);

        } catch (Exception ex) {
            InternalStaticEmbraceLogger.logDebug("Failed to query for memory usage", ex);
        }
    }

    @Override
    public List<MemorySample> getMemorySamples(long startTime, long endTime) {
        return new ArrayList<>(this.memorySamples.subMap(startTime, endTime).values());
    }

    @Override
    public List<MemoryWarning> getMemoryWarnings(long startTime, long endTime) {
        NavigableMap<Long, MemoryWarning> memoryWarnings = new ConcurrentSkipListMap<>();
        for (int i = 0; i < offset; i++) {
            memoryWarnings.put(memoryTimestamps[i], new MemoryWarning(memoryTimestamps[i]));
        }

        return new ArrayList<>(memoryWarnings.subMap(startTime, endTime).values());
    }

    @Override
    public void close() {
        InternalStaticEmbraceLogger.logDebug("Stopping EmbraceMemoryService");
    }

    @Override
    public void onMemoryWarning() {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMemoryService", "Memory warning number: " + offset);
        if (offset < MAX_CAPTURED_MEMORY_WARNINGS) {
            memoryTimestamps[offset] = clock.now();
            offset++;
        }
    }

    @Override
    public void cleanCollections() {
        this.memorySamples.clear();
        this.offset = 0;
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMemoryService", "Memory samples cleared");
    }
}
