package io.embrace.android.embracesdk;

import android.app.usage.StorageStatsManager;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.view.WindowManager;

import androidx.annotation.NonNull;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Locale;

import io.embrace.android.embracesdk.Embrace.AppFramework;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.optional.Optional;
import kotlin.Lazy;
import kotlin.LazyKt;

/**
 * Provides information about the state of the device, retrieved from Android system services,
 * which is used as metadata with telemetry submitted to the Embrace API.
 */
final class EmbraceMetadataService implements MetadataService, ActivityListener {
    /**
     * Default string value for app info missing strings
     */
    private static final String UNKNOWN_VALUE = "UNKNOWN";

    private final WindowManager windowManager;

    private final PackageManager packageManager;

    private final StorageStatsManager storageStatsManager;

    private final ApplicationInfo applicationInfo;

    private final PreferencesService preferencesService;

    private final ActivityService activityService;

    private final BuildInfo buildInfo;

    private final LocalConfig localConfig;

    private final Lazy<String> deviceId;

    private final String packageName;

    private final String appVersionName;

    private final String appVersionCode;

    private final Lazy<StatFs> statFs = LazyKt.lazy(() -> new StatFs(Environment.getDataDirectory().getPath()));

    private final String javaScriptPatchNumber;

    private final String reactNativeVersion;

    private final String unityVersion;

    private final String buildGUID;

    /**
     * This field is defined during instantiation as by the end of the startup
     */
    private final Lazy<Boolean> appUpdated;

    private final Lazy<Boolean> osUpdated;

    private final AppFramework appFramework;


    private final BackgroundWorker metadataRetrieveWorker;

    private volatile String activeSessionId;
    private volatile DiskUsage diskUsage;
    private volatile String screenResolution;
    private volatile Boolean isJailbroken;
    private Lazy<String> reactNativeBundleID;

    private EmbraceMetadataService(
            WindowManager windowManager,
            PackageManager packageManager,
            StorageStatsManager storageStatsManager,
            BuildInfo buildInfo,
            LocalConfig localConfig,
            ApplicationInfo applicationInfo,
            Lazy<String> deviceId,
            String packageName,
            String appVersionName,
            String appVersionCode,
            AppFramework appFramework,
            Lazy<Boolean> appUpdated,
            Lazy<Boolean> osUpdated,
            PreferencesService preferencesService,
            ActivityService activityService,
            Lazy<String> reactNativeBundleID,
            String javaScriptPatchNumber,
            String reactNativeVersion,
            String unityVersion,
            String buildGUID,
            BackgroundWorker metadataRetrieveWorker) {

        this.windowManager = windowManager;
        this.packageManager = packageManager;
        this.storageStatsManager = storageStatsManager;
        this.buildInfo = Preconditions.checkNotNull(buildInfo);
        this.localConfig = Preconditions.checkNotNull(localConfig);
        this.applicationInfo = applicationInfo;
        this.deviceId = deviceId;
        this.packageName = Preconditions.checkNotNull(packageName);
        this.appVersionName = Preconditions.checkNotNull(appVersionName);
        this.appVersionCode = Preconditions.checkNotNull(appVersionCode);
        this.appFramework = Preconditions.checkNotNull(appFramework);
        this.appUpdated = appUpdated;
        this.osUpdated = osUpdated;
        this.preferencesService = Preconditions.checkNotNull(preferencesService);
        this.activityService = Preconditions.checkNotNull(activityService);
        this.metadataRetrieveWorker = Preconditions.checkNotNull(metadataRetrieveWorker);

        if (appFramework == AppFramework.REACT_NATIVE) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Setting RN settings");
            this.reactNativeBundleID = reactNativeBundleID;
            this.javaScriptPatchNumber = javaScriptPatchNumber;
            this.reactNativeVersion = reactNativeVersion;
        } else {
            this.reactNativeBundleID = LazyKt.lazy(buildInfo::getBuildId);
            this.javaScriptPatchNumber = null;
            this.reactNativeVersion = null;
        }

        if (appFramework == AppFramework.UNITY) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Setting Unity settings");
            this.unityVersion = unityVersion;
            this.buildGUID = buildGUID;
        } else {
            this.unityVersion = null;
            this.buildGUID = null;
        }

        activityService.addListener(this);

        precomputeValues();
    }

    /**
     * Creates an instance of the {@link EmbraceMetadataService} from the device's {@link Context}
     * for creating Android system services.
     *
     * @param context            the {@link Context}
     * @param buildInfo          the build information
     * @param appFramework       the framework used by the app
     * @param preferencesService the preferences service
     * @return an instance
     */
    static EmbraceMetadataService ofContext(
            Context context,
            BuildInfo buildInfo,
            LocalConfig localConfig,
            AppFramework appFramework,
            PreferencesService preferencesService,
            ActivityService activityService,
            BackgroundWorker metadataRetrieveWorker) {

        Preconditions.checkNotNull(context, "Device context is null");

        StorageStatsManager storageStatsManager = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "SDK version " + Build.VERSION.SDK_INT + ", setting StorageStatsManager");
            storageStatsManager = (StorageStatsManager) context.getSystemService(Context.STORAGE_STATS_SERVICE);
        }

        PackageInfo packageInfo;
        String appVersionName;
        String appVersionCode;
        try {
            packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            // some customers have trailing white-space for the app version. remove this.
            appVersionName = String.valueOf(packageInfo.versionName).trim();
            appVersionCode = String.valueOf(packageInfo.versionCode);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "App version name: " + appVersionName + " - App version code: " + appVersionCode);

        } catch (Exception e) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Cannot set appVersionName and appVersionCode, setting UNKNOWN_VALUE", e);
            appVersionName = UNKNOWN_VALUE;
            appVersionCode = UNKNOWN_VALUE;
        }

        String finalAppVersionName = appVersionName;
        Lazy<Boolean> isAppUpdated = LazyKt.lazy(() -> {
            Optional<String> lastKnownAppVersion = preferencesService.getAppVersion();
            boolean appUpdated = lastKnownAppVersion.isPresent() && !lastKnownAppVersion.get().equalsIgnoreCase(finalAppVersionName);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "App updated: " + appUpdated);
            return appUpdated;
        });

        Lazy<Boolean> isOsUpdated = LazyKt.lazy(() -> {
            Optional<String> lastKnownOsVersion = preferencesService.getOsVersion();
            boolean osUpdated = lastKnownOsVersion.isPresent() && !lastKnownOsVersion.get().equalsIgnoreCase(String.valueOf(Build.VERSION.RELEASE));
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "OS updated: " + osUpdated);
            return osUpdated;
        });

        Lazy<String> deviceIdentifier = LazyKt.lazy(preferencesService::getDeviceIdentifier);

        String javaScriptPatchNumber = null;
        String reactNativeVersion = null;
        Lazy<String> reactNativeBundleID;

        if (appFramework == AppFramework.REACT_NATIVE) {
            reactNativeBundleID = LazyKt.lazy(() -> {
                String lastKnownJSBundleUrl = preferencesService.getJavaScriptBundleURL();
                if (lastKnownJSBundleUrl != null) {
                    return computeReactNativeBundleId(context, lastKnownJSBundleUrl, buildInfo.getBuildId());
                } else {
                    // If JS bundle ID URL is not found we assume that the App is not using Codepush.
                    // Use JS bundle ID URL as React Native bundle ID.
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "setting JSBundleUrl as buildId: " + buildInfo.getBuildId());
                    return buildInfo.getBuildId();
                }
            });

            if (preferencesService.getJavaScriptPatchNumber().isPresent()) {
                javaScriptPatchNumber = preferencesService.getJavaScriptPatchNumber().get();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Java script patch number: " + javaScriptPatchNumber);
            }
            if (preferencesService.getReactNativeVersionNumber().isPresent()) {
                reactNativeVersion = preferencesService.getReactNativeVersionNumber().get();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "RN version: " + reactNativeVersion);
            }
        } else {
            reactNativeBundleID = LazyKt.lazy(buildInfo::getBuildId);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "setting default RN as buildId");
        }

        String unityVersion = null;
        String buildGUI = null;
        if (appFramework == AppFramework.UNITY) {
            if (preferencesService.getUnityVersionNumber().isPresent()) {
                unityVersion = preferencesService.getUnityVersionNumber().get();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Unity version: " + unityVersion);
            } else {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Unity version is not present");
            }

            if (preferencesService.getUnityBuildIdNumber().isPresent()) {
                buildGUI = preferencesService.getUnityBuildIdNumber().get();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Unity build id: " + buildGUI);
            } else {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Unity build id number is not present");
            }
        }

        return new EmbraceMetadataService(
                (WindowManager) context.getSystemService(Context.WINDOW_SERVICE),
                context.getPackageManager(),
                storageStatsManager,
                buildInfo,
                localConfig,
                context.getApplicationInfo(),
                deviceIdentifier,
                context.getPackageName(),
                appVersionName,
                appVersionCode,
                appFramework,
                isAppUpdated,
                isOsUpdated,
                preferencesService,
                activityService,
                reactNativeBundleID,
                javaScriptPatchNumber,
                reactNativeVersion,
                unityVersion,
                buildGUI,
                metadataRetrieveWorker);
    }

    private static String getBundleAssetName(String bundleUrl) {
        String name = bundleUrl.substring(bundleUrl.indexOf("://") + 3);
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Asset name: " + name);
        return name;
    }

    @Nullable
    private static InputStream getBundleAsset(Context context, String bundleUrl) {
        try {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Attempting to read bundle asset: " + bundleUrl);
            return context.getAssets().open(getBundleAssetName(bundleUrl));
        } catch (Exception e) {
            EmbraceLogger.logError("Failed to retrieve RN bundle file from assets.", e);
        }

        return null;
    }

    @Nullable
    private static InputStream getCustomBundleStream(String bundleUrl) {
        try {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Attempting to load bundle from custom path: " + bundleUrl);
            return new FileInputStream(bundleUrl);
        } catch (NullPointerException | FileNotFoundException e) {
            EmbraceLogger.logError("Failed to retrieve the custom RN bundle file.", e);
        }

        return null;
    }

    private static String computeReactNativeBundleId(
            Context context,
            @NonNull String bundleUrl,
            @NonNull String defaultBundleId) {

        InputStream bundleStream;
        // checks if the bundle url is an asset
        if (bundleUrl.contains("assets")) {
            // looks for the bundle file in assets
            bundleStream = getBundleAsset(context, bundleUrl);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Loaded bundle file asset: " + bundleStream);
        } else {
            // looks for the bundle file from the custom path
            bundleStream = getCustomBundleStream(bundleUrl);
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Loaded bundle file from custom path: " + bundleStream);
        }

        if (bundleStream == null) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Setting default RN bundleId: " + defaultBundleId);
            return defaultBundleId;
        }

        try (InputStream inputStream = bundleStream;
             ByteArrayOutputStream buffer = new ByteArrayOutputStream()) {

            int nRead;
            //The hash size for the MD5 algorithm is 128 bits - 16 bytes.
            byte[] data = new byte[16];
            while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
                buffer.write(data, 0, nRead);
            }

            return hashBundleToMD5(buffer.toByteArray());
        } catch (Exception e) {
            InternalStaticEmbraceLogger.logError("Failed to compute the RN bundle file.", e);
        }

        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Setting default RN bundleId: " + defaultBundleId);
        // if the hashing of the JS bundle URL fails, returns the default bundle ID
        return defaultBundleId;
    }

    private static String hashBundleToMD5(byte[] bundle) throws NoSuchAlgorithmException {

        String hashBundle;
        MessageDigest md = MessageDigest.getInstance("MD5");

        byte[] bundleHashed = md.digest(bundle);

        StringBuilder sb = new StringBuilder();
        for (byte b : bundleHashed) {
            sb.append(String.format("%02x", b & 0xff));
        }

        hashBundle = sb.toString().toUpperCase(Locale.getDefault());
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Setting RN bundleId: " + hashBundle);
        return hashBundle;
    }

    /**
     * Queues in a single thread executor callables to retrieve values in background
     */
    private void precomputeValues() {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Precomputing values asynchronously: Jailbroken/ScreenResolution/DiskUsage");
        asyncRetrieveIsJailbroken();
        asyncRetrieveScreenResolution();

        // Always retrieve the DiskUsage last because it can take the longest to run
        asyncRetrieveDiskUsage(Build.VERSION.SDK_INT >= Build.VERSION_CODES.O);
    }

    private void asyncRetrieveScreenResolution() {
        // if the screenResolution exists in memory, don't try to retrieve it
        if (this.screenResolution != null && !this.screenResolution.isEmpty()) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Screen resolution already exists");
            return;
        }
        metadataRetrieveWorker.submit(() -> {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Async retrieve screen resolution");
            Optional<String> storedScreenResolution = preferencesService.getScreenResolution();
            // get from shared preferences
            if (storedScreenResolution.isPresent()) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Screen resolution is present, loading from store");
                this.screenResolution = storedScreenResolution.get();
            } else {
                this.screenResolution = MetadataUtils.getScreenResolution(windowManager).orNull();
                preferencesService.setScreenResolution(this.screenResolution);
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Screen resolution computed and stored");
            }
            return null;
        });
    }

    private void asyncRetrieveIsJailbroken() {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Async retrieve Jailbroken");

        // if the isJailbroken property exists in memory, don't try to retrieve it
        if (this.isJailbroken != null) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Jailbroken already exists");
            return;
        }
        metadataRetrieveWorker.submit(() -> {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Async retrieve jailbroken");
            Optional<Boolean> storedIsJailbroken = preferencesService.getJailbroken();
            // load value from shared preferences
            if (storedIsJailbroken.isPresent()) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Jailbroken is present, loading from store");
                this.isJailbroken = storedIsJailbroken.get();
            } else {
                this.isJailbroken = MetadataUtils.isJailbroken();
                preferencesService.setIsJailbroken(this.isJailbroken);
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Jailbroken processed and stored");
            }
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Jailbroken: " + isJailbroken);
            return null;
        });
    }

    @VisibleForTesting
    void asyncRetrieveDiskUsage(boolean isAndroid26OrAbove) {
        metadataRetrieveWorker.submit(() -> {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Async retrieve disk usage");
            long free = MetadataUtils.getInternalStorageFreeCapacity(statFs.getValue());
                if (isAndroid26OrAbove && localConfig.getConfigurations().getApp().getReportDiskUsage()) {
                    Optional<Long> deviceDiskAppUsage = MetadataUtils.getDeviceDiskAppUsage(storageStatsManager, packageManager, packageName);
                    if (deviceDiskAppUsage.isPresent()) {
                        InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Disk usage is present");
                        this.diskUsage = new DiskUsage(deviceDiskAppUsage.get(), free);
                    }
                }

                this.diskUsage = new DiskUsage(free);
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Device disk free: " + free);
                return null;
            });
        }

        @VisibleForTesting
        String getReactNativeBundleID () {
            return reactNativeBundleID.getValue();
        }

        @Override
        public String getDeviceId () {
            return deviceId.getValue();
        }

        @Override
        public String getAppVersionCode () {
            return appVersionCode;
        }

        @Override
        public String getAppVersionName () {
            return appVersionName;
        }

        @Override
        public DeviceInfo getDeviceInfo () {
            return getDeviceInfo(true);
        }

        @Override
        public DeviceInfo getLightweightDeviceInfo () {
            return getDeviceInfo(false);
        }

        private DeviceInfo getDeviceInfo (boolean populateAllFields) {
            return DeviceInfo.newBuilder()
                    .withManufacturer(MetadataUtils.getDeviceManufacturer())
                    .withModel(MetadataUtils.getModel())
                    .withArchitecture(MetadataUtils.getArchitecture())
                    .withJailbroken(populateAllFields ? getIsJailbroken() : false)
                    .withLocale(MetadataUtils.getLocale())
                    .withInternalStorageTotalCapacity(
                            populateAllFields ?
                                    MetadataUtils.getInternalStorageTotalCapacity(statFs.getValue()) :
                                    0)
                    .withOperatingSystemType(MetadataUtils.getOperatingSystemType())
                    .withOperatingSystemVersion(MetadataUtils.getOperatingSystemVersion())
                    .withOperatingSystemVersionCode(MetadataUtils.getOperatingSystemVersionCode())
                    .withScreenResolution(getScreenResolution())
                    .withTimezoneDescription(MetadataUtils.getTimezoneId())
                    .withUptime(MetadataUtils.getSystemUptime())
                    .build();
        }

        @Override
        public AppInfo getAppInfo () {
            return getAppInfo(true);
        }

        @Override
        public AppInfo getLightweightAppInfo() {
            return getAppInfo(false);
        }

        private AppInfo getAppInfo(boolean populateAllFields) {
            // TODO default these all to UNKNOWN_VALUE
            String buildId = buildInfo.getBuildId();

            AppInfo.Builder builder = AppInfo.newBuilder()
                    .withSdkVersion(BuildConfig.VERSION_NAME)
                    .withSdkSimpleVersion(BuildConfig.VERSION_CODE)
                    .withBuildId(buildId)
                    .withBuildType(buildInfo.getBuildType())
                    .withBuildFlavor(buildInfo.getBuildFlavor())
                    .withAppVersion(appVersionName)
                    .withAppFramework(appFramework)
                    .withBundleVersion(appVersionCode)
                    .withEnvironment(MetadataUtils.appEnvironment(applicationInfo))
                    .withAppUpdated(populateAllFields ? appUpdated.getValue() : false)
                    .withOsUpdated(populateAllFields ? osUpdated.getValue() : false)
                    .withAppUpdatedThisLaunch(populateAllFields ? appUpdated.getValue() : false)
                    .withOsUpdatedThisLaunch(populateAllFields ? osUpdated.getValue() : false);

            // applies to Unity builds only.
            if (appFramework == AppFramework.UNITY) {
                if (unityVersion != null) {
                    builder.withUnityVersion(unityVersion);
                } else if (preferencesService.getUnityVersionNumber().isPresent()) {
                    builder.withUnityVersion(preferencesService.getUnityVersionNumber().get());
                }

                if (buildGUID != null) {
                    builder.withUnityBuildId(buildGUID);
                } else if (preferencesService.getUnityBuildIdNumber().isPresent()) {
                    builder.withUnityBuildId(preferencesService.getUnityBuildIdNumber().get());
                }
            }

            // applies to React Native builds only
            if (appFramework == AppFramework.REACT_NATIVE) {
                builder.withReactNativeBundleID(reactNativeBundleID.getValue())
                        .withJavaScriptPatchNumber(javaScriptPatchNumber)
                        .withReactNativeVersion(reactNativeVersion);
            }

            return builder.build();
        }

        @Override
        public boolean isDebug () {
            return MetadataUtils.isDebug(applicationInfo);
        }

        @Override
        public String getAppId () {
            return localConfig.getAppId();
        }

        @Override
        public boolean isAppUpdated () {
            return appUpdated.getValue();
        }

        @Override
        public boolean isOsUpdated () {
            return osUpdated.getValue();
        }

        @Override
        public Optional<String> getActiveSessionId () {
            return Optional.fromNullable(activeSessionId);
        }

        @Override
        public void setActiveSessionId (String sessionId){
            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "Active session Id: " + sessionId);
            this.activeSessionId = sessionId;
        }

        @Override
        public String getAppState () {
            if (activityService.isInBackground()) {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "App state: BACKGROUND");
                return "background";
            } else {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "App state: ACTIVE");
                return "active";
            }
        }

        @Override
        public DiskUsage getDiskUsage () {
            return diskUsage;
        }

        @Override
        public String getScreenResolution () {
            return screenResolution;
        }

        @Override
        public Boolean getIsJailbroken () {
            return isJailbroken;
        }

        @Override
        public void setReactNativeBundleId (Context context, String jsBundleIdUrl){
            if (jsBundleIdUrl.isEmpty()) {
                EmbraceLogger.logError("JavaScript bundle URL must have non-zero length");
                this.reactNativeBundleID = LazyKt.lazy(buildInfo::getBuildId);
                return;
            }

            String currentUrl = preferencesService.getJavaScriptBundleURL();
            if (currentUrl != null && currentUrl.equals(jsBundleIdUrl)) {
                // if the JS bundle ID URL didn't change, use the value from preferences
                EmbraceLogger.logDebug("JavaScript bundle URL already exists and didn't change. " +
                        "Using: " + currentUrl + ".");
                this.reactNativeBundleID = LazyKt.lazy(buildInfo::getBuildId);
                return;
            }

            // if doesn't exists or if is a new JS bundle ID URL, save the new value in preferences
            this.preferencesService.setJavaScriptBundleURL(jsBundleIdUrl);

            // get the hashed bundle ID file from the bundle ID URL
            this.reactNativeBundleID = LazyKt.lazy(() -> computeReactNativeBundleId(context, jsBundleIdUrl, buildInfo.getBuildId()));
        }

        @Override
        public void applicationStartupComplete () {
            String appVersion = getAppVersionName();
            String osVersion = String.valueOf(Build.VERSION.RELEASE);
            String localDeviceId = getDeviceId();
            long installDate = System.currentTimeMillis();
            InternalStaticEmbraceLogger.logDebug(String.format(Locale.getDefault(), "Setting metadata on preferences service. " +
                            "App version: {%s}, OS version {%s}, device ID: {%s}, install date: {%d}",
                    appVersion,
                    osVersion,
                    localDeviceId,
                    installDate));
            preferencesService.setAppVersion(appVersion);
            preferencesService.setOsVersion(osVersion);
            preferencesService.setDeviceIdentifier(localDeviceId);
            if (!preferencesService.getInstallDate().isPresent()) {
                preferencesService.setInstallDate(installDate);
            }

            InternalStaticEmbraceLogger.logDeveloper("EmbraceMetadataService", "- Application Startup Complete -");
        }
    }
