package io.embrace.android.embracesdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

class EmbraceNetworkConnectivityService extends BroadcastReceiver implements NetworkConnectivityService, MemoryCleanerListener {

    private final IntentFilter intentFilter = new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION);
    private final NavigableMap<Long, NetworkStatus> networkReachable = new TreeMap<>();
    private final Context context;
    private final Clock clock;

    private final BackgroundWorker bgRegistrationWorker;

    private ConnectivityManager connectivityManager;

    private final InternalEmbraceLogger logger;

    public EmbraceNetworkConnectivityService(Context context,
                                             Clock clock,
                                             MemoryCleanerService memoryCleanerService,
                                             BackgroundWorker bgRegistrationWorker,
                                             InternalEmbraceLogger logger) {
        this.context = Preconditions.checkNotNull(context, "context must not be null");
        this.clock = Preconditions.checkNotNull(clock);
        this.logger = logger;
        this.bgRegistrationWorker = Preconditions.checkNotNull(bgRegistrationWorker);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        registerConnectivityActionReceiver();
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        handleNetworkStatus();
    }

    @Override
    public List<Interval> getNetworkInterfaceIntervals(long startTime, long endTime) {
        logger.logDeveloper("EmbraceNetworkConnectivityService", "getNetworkInterfaceIntervals");
        synchronized (this) {
            List<Interval> results = new ArrayList<>();
            for (Map.Entry<Long, NetworkStatus> entry : networkReachable.subMap(startTime, endTime).entrySet()) {
                long currentTime = entry.getKey();
                Long next = networkReachable.higherKey(currentTime);
                results.add(new Interval(currentTime, next != null ? next : endTime, entry.getValue().getName()));
            }
            return results;
        }
    }

    @Override
    public void networkStatusOnSessionStarted(long startTime) {
        handleNetworkStatus(startTime);
    }

    private void handleNetworkStatus() {
        handleNetworkStatus(clock.now());
    }

    private void handleNetworkStatus(long timestamp) {
        try {
            logger.logDeveloper("EmbraceNetworkConnectivityService", "handleNetworkStatus");
            NetworkInfo networkInfo = getConnectivityManager().getActiveNetworkInfo();
            if (networkInfo != null && networkInfo.isConnected()) {
                // Network is reachable
                if (networkInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                    logger.logDeveloper("EmbraceNetworkConnectivityService", "Network connected to WIFI");
                    saveStatus(timestamp, NetworkStatus.WIFI);
                } else if (networkInfo.getType() == ConnectivityManager.TYPE_MOBILE) {
                    logger.logDeveloper("EmbraceNetworkConnectivityService", "Network connected to MOBILE");
                    saveStatus(timestamp, NetworkStatus.WAN);
                }
            } else {
                // Network is not reachable
                logger.logDeveloper("EmbraceNetworkConnectivityService", "Network not reachable");
                saveStatus(timestamp, NetworkStatus.NOT_REACHABLE);
            }
        } catch (Exception ex) {
            logger.logDebug("Failed to record network connectivity", ex);
        }
    }

    private void saveStatus(long timestamp, NetworkStatus networkStatus) {
        synchronized (this) {
            if (networkReachable.isEmpty() || !networkReachable.lastEntry().getValue().equals(networkStatus)) {
                networkReachable.put(timestamp, networkStatus);
            }
        }
    }

    private synchronized ConnectivityManager getConnectivityManager() {
        if (connectivityManager == null) {
            connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        }
        return connectivityManager;
    }

    private void registerConnectivityActionReceiver() {
        bgRegistrationWorker.submit(() -> {
            try {
                context.registerReceiver(this, intentFilter);
            } catch (Exception ex) {
                logger.logDebug("Failed to register EmbraceNetworkConnectivityService broadcast receiver. Connectivity status will be unavailable.", ex);
            }
            return null;
        });
    }

    @Override
    public void close() {
        context.unregisterReceiver(this);
        logger.logDeveloper("EmbraceNetworkConnectivityService", "closed");
    }

    @Override
    public void cleanCollections() {
        this.networkReachable.clear();
        logger.logDeveloper("EmbraceNetworkConnectivityService", "Collections cleaned");
    }

    private enum NetworkStatus {
        NOT_REACHABLE("none"),
        WIFI("wifi"),
        WAN("wan");

        private final String name;

        NetworkStatus(String name) {
            this.name = name;
        }

        String getName() {
            return this.name;
        }
    }
}
