package io.embrace.android.embracesdk

import io.embrace.android.embracesdk.NetworkSessionV2.DomainCount
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.utils.NetworkUtils
import io.embrace.android.embracesdk.utils.NetworkUtils.getDomain
import io.embrace.android.embracesdk.utils.NetworkUtils.getValidTraceId
import io.embrace.android.embracesdk.utils.NetworkUtils.isIpAddress
import io.embrace.android.embracesdk.utils.NetworkUtils.stripUrl
import io.embrace.android.embracesdk.utils.optional.Optional
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ConcurrentSkipListMap
import java.util.concurrent.atomic.AtomicInteger
import kotlin.math.max

/**
 * Logs network calls according to defined limits per domain.
 *
 *
 * Limits can be defined either in server-side configuration or within the embrace configuration file.
 * A limit of 0 disables logging for the domain. All network calls are captured up to the limit,
 * and the number of calls is also captured if the limit is exceeded.
 */
internal class EmbraceNetworkLoggingService(
    private val configService: ConfigService,
    private val localConfig: LocalConfig,
    memoryCleanerService: MemoryCleanerService,
    private val logger: InternalEmbraceLogger
) : NetworkLoggingService, MemoryCleanerListener {

    /**
     * Network calls per domain prepared for the session.
     */
    private val sessionNetworkCalls = ConcurrentSkipListMap<Long, NetworkCallV2>()

    private val domainSettings = ConcurrentHashMap<String, DomainSettings>()

    private val callsPerDomain = hashMapOf<String, DomainCount>()

    private val captureLimit: Int = getCaptureLimit()

    private val ipAddressCount = AtomicInteger(0)

    init {
        logger.logDeveloper("EmbraceNetworkLoggingService", "starting NetworkLoggingService")
        memoryCleanerService.addListener(this)
    }

    override fun getNetworkCallsForSession(startTime: Long, lastKnownTime: Long): NetworkSessionV2 {
        logger.logDeveloper("EmbraceNetworkLoggingService", "getNetworkCallsForSession")

        val calls: List<NetworkCallV2> = ArrayList(
            sessionNetworkCalls.subMap(startTime, lastKnownTime).values
        )
        val overLimit = hashMapOf<String, DomainCount>()
        for ((key, value) in callsPerDomain) {
            if (value.requestCount > value.captureLimit) {
                overLimit[key] = value
            }
        }

        // clear calls per domain and session network calls lists before be used by the next session
        callsPerDomain.clear()
        return NetworkSessionV2(calls, overLimit)
    }

    override fun logNetworkCall(
        url: String,
        httpMethod: String,
        statusCode: Int,
        startTime: Long,
        endTime: Long,
        bytesSent: Long,
        bytesReceived: Long,
        traceId: String?
    ) {
        val duration = max(endTime - startTime, 0)
        val strippedUrl = stripUrl(url)
        val validTraceId = getValidTraceId(traceId)
        val networkCall = NetworkCallV2(
            url = strippedUrl,
            httpMethod = httpMethod,
            responseCode = statusCode,
            bytesSent = bytesSent,
            bytesReceived = bytesReceived,
            startTime = startTime,
            endTime = endTime,
            duration = duration,
            traceId = validTraceId
        )
        processNetworkCall(startTime, networkCall)
        storeSettings(url)
    }

    override fun logNetworkError(
        url: String,
        httpMethod: String,
        startTime: Long,
        endTime: Long,
        errorType: String?,
        errorMessage: String?,
        traceId: String?
    ) {
        val duration = max(endTime - startTime, 0)
        val strippedUrl = stripUrl(url)
        val validTraceId = getValidTraceId(traceId)
        val networkCall = NetworkCallV2(
            url = strippedUrl,
            httpMethod = httpMethod,
            startTime = startTime,
            endTime = endTime,
            duration = duration,
            traceId = validTraceId,
            errorMessage = errorMessage,
            errorType = errorType
        )
        processNetworkCall(startTime, networkCall)
        storeSettings(url)
    }

    /**
     * Process network calls to be ready when the session requests them.
     *
     * @param startTime   is the time when the network call was captured
     * @param networkCall that is going to be captured
     */
    private fun processNetworkCall(startTime: Long, networkCall: NetworkCallV2) {
        logger.logDeveloper("EmbraceNetworkLoggingService", "processNetworkCall at: $startTime")

        // Get the domain, if it can be successfully parsed
        val domain = networkCall.url?.let {
            getDomain(it)
        } ?: Optional.absent()

        if (!domain.isPresent) {
            logger.logDeveloper("EmbraceNetworkLoggingService", "Domain is not present")
            return
        }

        logger.logDeveloper("EmbraceNetworkLoggingService", "Domain: $domain")

        if (isIpAddress(domain.get())) {
            logger.logDeveloper("EmbraceNetworkLoggingService", "Domain is an ip address")
            if (ipAddressCount.getAndIncrement() < captureLimit) {
                // only capture if the ipAddressCount has not exceeded defaultLimit
                logger.logDeveloper("EmbraceNetworkLoggingService", "capturing network call")
                sessionNetworkCalls[startTime] = networkCall
            } else {
                logger.logDeveloper("EmbraceNetworkLoggingService", "capture limit exceeded")
            }
            return
        }

        val settings = domainSettings[domain.get()]
        if (settings == null) {
            logger.logDeveloper("EmbraceNetworkLoggingService", "no domain settings")
            sessionNetworkCalls[startTime] = networkCall
        } else {
            val suffix = settings.suffix
            val limit = settings.limit
            var count = callsPerDomain[suffix]

            if (count == null) {
                count = DomainCount(1, limit)
            }

            // Exclude if the network call exceeds the limit
            if (count.requestCount < limit) {
                sessionNetworkCalls[startTime] = networkCall
            } else {
                logger.logDeveloper("EmbraceNetworkLoggingService", "capture limit exceeded")
            }

            // Track the number of calls for each domain (or configured suffix)
            suffix?.let {
                callsPerDomain[it] = DomainCount(count.requestCount + 1, limit)
                logger.logDeveloper(
                    "EmbraceNetworkLoggingService",
                    "Call per domain $domain ${count.requestCount + 1}"
                )
            }
        }
    }

    /**
     * Get the capture limit defined by the application at build-time
     *
     * @return defined capture limit
     */
    private fun getCaptureLimit(): Int {
        var customLimit = Optional.absent<Int>()
        val networkConfig = localConfig.configurations.getNetworking()

        if (networkConfig.getDefaultCaptureLimit().isPresent) {
            customLimit = networkConfig.getDefaultCaptureLimit()
            logger.logDeveloper("EmbraceNetworkLoggingService", "Capture limit is: $customLimit")
        } else {
            logger.logDeveloper("EmbraceNetworkLoggingService", "Capture limit not present")
        }

        return configService.config
            .defaultNetworkCallLimit
            .or(customLimit)
            .or(NetworkUtils.DEFAULT_NETWORK_CALL_LIMIT)
    }

    private fun storeSettings(url: String) {
        try {
            val remoteLimits = configService.config.networkCallLimitsPerDomain
            val mergedLimits: MutableMap<String?, Int?> = HashMap()
            if (localConfig.configurations.getNetworking().getDomains().isNotEmpty()) {
                for (domain in localConfig.configurations.getNetworking().getDomains()) {
                    if (domain.getLimit().isPresent) {
                        logger.logDeveloper(
                            "EmbraceNetworkLoggingService",
                            "Domain limit: ${domain.getLimit().get()}"
                        )
                        mergedLimits[domain.getDomain()] = domain.getLimit().get()
                    } else {
                        logger.logDebug(
                            "Config issue: Failed to merge domain. Domain limit is not specified for $url"
                        )
                    }
                }
            }
            mergedLimits.putAll(remoteLimits)

            val domain = getDomain(url)
            if (!domain.isPresent) {
                logger.logDeveloper("EmbraceNetworkLoggingService", "Domain not present")
                return
            }
            val domainString = domain.get()
            if (domainSettings.containsKey(domainString)) {
                logger.logDeveloper("EmbraceNetworkLoggingService", "No settings for $domainString")
                return
            }

            for ((key, value) in mergedLimits) {
                if (key != null && value != null && domainString.endsWith(key)) {
                    domainSettings[domainString] = DomainSettings(value, key)
                    return
                }
            }

            val defaultLimit = configService.config
                .defaultNetworkCallLimit
                .or(NetworkUtils.DEFAULT_NETWORK_CALL_LIMIT)
            logger.logDeveloper("EmbraceNetworkLoggingService", "Default limit is: $defaultLimit")
            domainSettings[domainString] = DomainSettings(defaultLimit, domainString)
        } catch (ex: Exception) {
            logger.logDebug("Failed to determine limits for URL: $url", ex)
        }
    }

    override fun cleanCollections() {
        domainSettings.clear()
        callsPerDomain.clear()
        sessionNetworkCalls.clear()
        // reset counters
        ipAddressCount.set(0)
        logger.logDeveloper("EmbraceNetworkLoggingService", "Collections cleaned")
    }
}
