package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.optional.Optional;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import io.embrace.android.embracesdk.utils.Preconditions;

final class EmbraceOrientationService implements OrientationService, MemoryCleanerListener,
        SessionStartListener, SessionEndListener {

    /**
     * States the activity orientations.
     */
    private final LinkedList<Orientation> orientations = new LinkedList<>();
    private final Clock clock;

    EmbraceOrientationService(MemoryCleanerService memoryCleanerService, Clock clock) {
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        this.clock = Preconditions.checkNotNull(clock);
    }

    @Override
    public void onOrientationChanged(Optional<Integer> orientation) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceOrientationService", "onOrientationChanged");
        if (orientation.isPresent() && (this.orientations.isEmpty() || this.orientations.getLast().getInternalOrientation() != orientation.get())) {
            this.orientations.add(new Orientation(orientation.get(), clock.now()));
            InternalStaticEmbraceLogger.logDeveloper("EmbraceOrientationService", "added new orientation " + orientation);
        }
    }

    @Override
    public List<Orientation> getOrientations() {
        return new ArrayList<>(orientations);
    }

    @Override
    public void cleanCollections() {
        orientations.clear();
        InternalStaticEmbraceLogger.logDeveloper("EmbraceOrientationService", "Collection cleaned");
    }

    @Override
    public void onSessionStart(@NonNull Session.Builder builder) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceOrientationService", "onSessionStart");
        builder.withOrientations(getOrientations());
    }

    @Override
    public void onSessionEnd(@NonNull Session.Builder builder) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceOrientationService", "onSessionEnd");
        builder.withOrientations(getOrientations());
    }
}
