package io.embrace.android.embracesdk;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

final class EmbracePerformanceInfoService implements PerformanceInfoService {

    private final AnrService anrService;

    private final NetworkConnectivityService networkConnectivityService;

    private final NetworkLoggingService networkLoggingService;

    private final PowerService powerService;

    private final MemoryService memoryService;

    private final MetadataService metadataService;

    private final GoogleAnrTimestampRepository googleAnrTimestampRepository;

    EmbracePerformanceInfoService(
            AnrService anrService,
            NetworkConnectivityService networkConnectivityService,
            NetworkLoggingService networkLoggingService,
            PowerService powerService,
            MemoryService memoryService,
            MetadataService metadataService,
            GoogleAnrTimestampRepository googleAnrTimestampRepository) {

        this.anrService = Preconditions.checkNotNull(anrService);
        this.networkConnectivityService = Preconditions.checkNotNull(networkConnectivityService);
        this.networkLoggingService = Preconditions.checkNotNull(networkLoggingService);
        this.powerService = Preconditions.checkNotNull(powerService);
        this.memoryService = Preconditions.checkNotNull(memoryService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.googleAnrTimestampRepository = Preconditions.checkNotNull(googleAnrTimestampRepository);
    }

    @Override
    public SessionPerformanceInfo getSessionPerformanceInfo(long startTime, long lastKnownTime) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePerformanceInfoService", "Session performance info start time: " + startTime);
        SessionPerformanceInfo.NetworkRequests requests =
                new SessionPerformanceInfo.NetworkRequests(networkLoggingService.getNetworkCallsForSession(startTime, lastKnownTime));

        SessionPerformanceInfo.Builder builder = SessionPerformanceInfo.newBuilder()
                .withNetworkRequests(requests);
        builder.withAnrIntervals(anrService.getAnrIntervals(startTime, lastKnownTime));
        builder.withGoogleAnrTimestamps(googleAnrTimestampRepository.getGoogleAnrTimestamps(startTime, lastKnownTime));
        addPerformanceInfo(builder, startTime, lastKnownTime);

        return builder.build();
    }

    @Override
    public PerformanceInfo getPerformanceInfo(long startTime, long endTime) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePerformanceInfoService", "Building performance info");
        PerformanceInfo.Builder builder = PerformanceInfo.newBuilder();
        addPerformanceInfo(builder, startTime, endTime);
        return builder.build();
    }

    private <T extends PerformanceInfo.Builder> void addPerformanceInfo(T builder, long startTime, long endTime) {
        builder.withBatteryMeasurements(powerService.getBatteryMeasurements(startTime, endTime))
                .withChargingIntervals(powerService.getChargingIntervals(startTime, endTime))
                .withDiskUsage(metadataService.getDiskUsage())
                .withMemorySamples(memoryService.getMemorySamples(startTime, endTime))
                .withMemoryWarnings(memoryService.getMemoryWarnings(startTime, endTime))
                .withNetworkInterfaceIntervals(networkConnectivityService.getNetworkInterfaceIntervals(startTime, endTime));
        if (powerService.getLatestBatteryLevel().isPresent()) {
            builder.withBatteryLevel(powerService.getLatestBatteryLevel().get());
        }
    }
}
