package io.embrace.android.embracesdk;

import android.content.SharedPreferences;
import android.text.TextUtils;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.optional.Optional;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;

import io.embrace.android.embracesdk.utils.Preconditions;
import kotlin.Lazy;
import kotlin.LazyKt;

final class EmbracePreferencesService implements PreferencesService, ActivityListener {

    static final String SDK_STARTUP_IN_PROGRESS = "startup_entered";
    static final String SDK_STARTUP_COMPLETED = "startup_completed";
    private static final String SDK_STARTUP_STATUS_KEY = "io.embrace.sdkstartup";

    private static final String DEVICE_IDENTIFIER_KEY = "io.embrace.deviceid";
    private static final String PREVIOUS_APP_VERSION_KEY = "io.embrace.lastappversion";
    private static final String PREVIOUS_OS_VERSION_KEY = "io.embrace.lastosversion";
    private static final String INSTALL_DATE_KEY = "io.embrace.installtimestamp";
    private static final String USER_IDENTIFIER_KEY = "io.embrace.userid";
    private static final String USER_EMAIL_ADDRESS_KEY = "io.embrace.useremail";
    private static final String USER_USERNAME_KEY = "io.embrace.username";
    private static final String USER_IS_PAYER_KEY = "io.embrace.userispayer";
    private static final String USER_PERSONAS_KEY = "io.embrace.userpersonas";
    private static final String CUSTOM_PERSONAS_KEY = "io.embrace.custompersonas";
    private static final String LAST_USER_MESSAGE_FAILED_KEY = "io.embrace.userupdatefailed";
    private static final String LAST_SESSION_NUMBER_KEY = "io.embrace.sessionnumber";
    private static final String JAVA_SCRIPT_BUNDLE_URL_KEY = "io.embrace.jsbundle.url";
    private static final String JAVA_SCRIPT_PATCH_NUMBER_KEY = "io.embrace.javascript.patch";
    private static final String REACT_NATIVE_VERSION_KEY = "io.embrace.reactnative.version";
    private static final String SESSION_PROPERTIES_KEY = "io.embrace.session.properties";
    private static final String UNITY_VERSION_NUMBER_KEY = "io.embrace.unity.version";
    private static final String UNITY_BUILD_ID_NUMBER_KEY = "io.embrace.unity.build.id";
    private static final String IS_JAILBROKEN_KEY = "io.embrace.is_jailbroken";
    private static final String SCREEN_RESOLUTION_KEY = "io.embrace.screen.resolution";

    private static final String SDK_DISABLED_KEY = "io.embrace.disabled";
    private static final String SDK_CONFIG_FETCHED_TIMESTAMP = "io.embrace.sdkfetchedtimestamp";

    private final Future<SharedPreferences> preferences;

    private static final Lazy<Gson> gson = LazyKt.lazy(Gson::new);
    private final BackgroundWorker bgRegistrationWorker;
    private final Lazy<SharedPreferences> lazyPrefs;

    @Override
    public void applicationStartupComplete() {
        alterStartupStatus(SDK_STARTUP_COMPLETED);
    }

    EmbracePreferencesService(ActivityService activityService,
                              BackgroundWorker bgRegistrationWorker,
                              Lazy<SharedPreferences> lazyPrefs) {
        this.bgRegistrationWorker = Preconditions.checkNotNull(bgRegistrationWorker);
        this.lazyPrefs = lazyPrefs;
        Preconditions.checkNotNull(activityService);
        activityService.addListener(this);

        // We get SharedPreferences on a background thread because it loads data from disk
        // and can block. When client code needs to set/get a preference, getSharedPrefs() will
        // block if necessary with Future.get(). Eagerly offloading buys us more time
        // for SharedPreferences to load the File and reduces the likelihood of blocking
        // when invoked by client code.
        this.preferences = bgRegistrationWorker.submit(lazyPrefs::getValue);
        alterStartupStatus(SDK_STARTUP_IN_PROGRESS);
    }

    private void alterStartupStatus(String status) {
        bgRegistrationWorker.submit(() -> {
            InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Startup key: " + status);
            setStringPreference(getSharedPrefs(), SDK_STARTUP_STATUS_KEY, status);
            return null;
        });
    }

    private SharedPreferences getSharedPrefs() {
        try {
            return preferences.get();
        } catch (Throwable exc) {
            // fallback from this very unlikely case by just loading on the main thread
            return lazyPrefs.getValue();
        }
    }

    private Optional<String> getStringPreference(SharedPreferences preferences, String key) {
        String value = null;
        if (preferences != null) {
            value = preferences.getString(key, null);
        }
        return Optional.fromNullable(value);
    }

    private void setStringPreference(SharedPreferences preferences, String key, String value) {
        if (preferences != null && !TextUtils.isEmpty(key)) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putString(key, value);
            editor.apply();
        }
    }

    private Optional<Long> getLongPreference(SharedPreferences preferences, String key) {
        long defaultValue = -1;
        long value = defaultValue;
        if (preferences != null) {
            value = preferences.getLong(key, defaultValue);
        }
        if (value == -1) {
            return Optional.absent();
        } else {
            return Optional.fromNullable(value);
        }
    }

    private void setLongPreference(SharedPreferences preferences, String key, long value) {

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putLong(key, value);
            editor.apply();
        }
    }

    private Optional<Integer> getIntegerPreference(SharedPreferences preferences, String key) {
        int defaultValue = -1;
        int value = defaultValue;
        if (preferences != null) {
            value = preferences.getInt(key, defaultValue);
        }
        if (value == -1) {
            return Optional.absent();
        } else {
            return Optional.fromNullable(value);
        }
    }

    private void setIntegerPreference(SharedPreferences preferences, String key, int value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putInt(key, value);
            editor.apply();
        }
    }

    private boolean getBooleanPreference(SharedPreferences preferences, String key, boolean defaultValue) {
        boolean value = defaultValue;
        if (preferences != null) {
            value = preferences.getBoolean(key, defaultValue);
        }
        return value;
    }

    private void setBooleanPreference(SharedPreferences preferences, String key, boolean value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putBoolean(key, value);
            editor.apply();
        }
    }

    private void setArrayPreference(SharedPreferences preferences, String key, Set<String> value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putStringSet(key, value);
            editor.apply();
        }
    }

    private Optional<Set<String>> getArrayPreference(SharedPreferences preferences, String key) {
        Set<String> defaultValue = null;
        Set<String> value = new HashSet<>();
        if (preferences != null) {
            value = preferences.getStringSet(key, defaultValue);
        }
        return Optional.fromNullable(value);
    }

    private void setMapPreference(SharedPreferences preferences, String key, Map<String, String> value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            String mapString = value == null ? null : gson.getValue().toJson(value);
            editor.putString(key, mapString);
            editor.apply();
        }
    }

    private Optional<Map<String, String>> getMapPreference(SharedPreferences preferences, String key) {
        if (preferences == null) {
            return Optional.fromNullable(null);
        }

        String mapString = preferences.getString(key, null);
        if (mapString == null) {
            return Optional.fromNullable(null);
        }
        java.lang.reflect.Type type = new TypeToken<HashMap<String, String>>() {
        }.getType();
        HashMap<String, String> value = gson.getValue().fromJson(mapString, type);
        return Optional.fromNullable(value);
    }

    @Override
    public Optional<String> getAppVersion() {
        return getStringPreference(getSharedPrefs(), PREVIOUS_APP_VERSION_KEY);
    }

    @Override
    public void setAppVersion(String appVersion) {
        setStringPreference(getSharedPrefs(), PREVIOUS_APP_VERSION_KEY, appVersion);
    }

    @Override
    public Optional<String> getOsVersion() {
        return getStringPreference(getSharedPrefs(), PREVIOUS_OS_VERSION_KEY);
    }

    @Override
    public void setOsVersion(String osVersion) {
        setStringPreference(getSharedPrefs(), PREVIOUS_OS_VERSION_KEY, osVersion);
    }

    @Override
    public Optional<Long> getInstallDate() {
        return getLongPreference(getSharedPrefs(), INSTALL_DATE_KEY);
    }

    @Override
    public void setInstallDate(long installDate) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set install date: " + installDate);
        setLongPreference(getSharedPrefs(), INSTALL_DATE_KEY, installDate);
    }

    @Override
    public String getDeviceIdentifier() {
        String deviceId = getStringPreference(getSharedPrefs(), DEVICE_IDENTIFIER_KEY).orNull();
        if (deviceId == null) {
            deviceId = Uuid.getEmbUuid();
            InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Device ID is null, created new one: " + deviceId);
            setDeviceIdentifier(deviceId);
        }
        return deviceId;
    }

    @Override
    public void setDeviceIdentifier(String deviceIdentifier) {
        setStringPreference(getSharedPrefs(), DEVICE_IDENTIFIER_KEY, deviceIdentifier);
    }

    @Override
    public Optional<String> getSDKStartupStatus() {
        return getStringPreference(getSharedPrefs(), SDK_STARTUP_STATUS_KEY);
    }

    @Override
    public boolean getSDKDisabled() {
        return getBooleanPreference(getSharedPrefs(), SDK_DISABLED_KEY, false);
    }

    @Override
    public void setSDKDisabled(boolean disabled) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set SDK disabled: " + disabled);
        setBooleanPreference(getSharedPrefs(), SDK_DISABLED_KEY, disabled);
    }

    @Override
    public boolean getUserPayer() {
        return getBooleanPreference(getSharedPrefs(), USER_IS_PAYER_KEY, false);
    }

    @Override
    public void setUserPayer(boolean payer) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user payer: " + payer);
        setBooleanPreference(getSharedPrefs(), USER_IS_PAYER_KEY, payer);
    }

    @Override
    public Optional<String> getUserIdentifier() {
        return getStringPreference(getSharedPrefs(), USER_IDENTIFIER_KEY);
    }

    @Override
    public void setUserIdentifier(String identifier) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user identifier: " + identifier);
        setStringPreference(getSharedPrefs(), USER_IDENTIFIER_KEY, identifier);
    }

    @Override
    public Optional<String> getUserEmailAddress() {
        return getStringPreference(getSharedPrefs(), USER_EMAIL_ADDRESS_KEY);
    }

    @Override
    public void setUserEmailAddress(String emailAddress) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user email address: " + emailAddress);
        setStringPreference(getSharedPrefs(), USER_EMAIL_ADDRESS_KEY, emailAddress);
    }

    @Override
    public Optional<Set<String>> getUserPersonas() {
        return getArrayPreference(getSharedPrefs(), USER_PERSONAS_KEY);
    }

    @Override
    public void setUserPersonas(Set<String> personas) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user personas: " + personas.toString());
        setArrayPreference(getSharedPrefs(), USER_PERSONAS_KEY, personas);
    }

    @Override
    public Optional<Map<String, String>> getPermanentSessionProperties() {
        return getMapPreference(getSharedPrefs(), SESSION_PROPERTIES_KEY);
    }

    @Override
    public void setPermanentSessionProperties(Map<String, String> properties) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", properties != null ? properties.toString() : "-");
        setMapPreference(getSharedPrefs(), SESSION_PROPERTIES_KEY, properties);
    }

    @Override
    public Optional<Set<String>> getCustomPersonas() {
        return getArrayPreference(getSharedPrefs(), CUSTOM_PERSONAS_KEY);
    }

    @Override
    public Optional<String> getUsername() {
        return getStringPreference(getSharedPrefs(), USER_USERNAME_KEY);
    }

    @Override
    public void setUsername(String username) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user name: " + username);
        setStringPreference(getSharedPrefs(), USER_USERNAME_KEY, username);
    }

    @Override
    public Optional<Long> getLastConfigFetchDate() {
        return getLongPreference(getSharedPrefs(), SDK_CONFIG_FETCHED_TIMESTAMP);
    }

    @Override
    public void setLastConfigFetchDate(long fetchDate) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set las configuration fetch date: " + fetchDate);
        setLongPreference(getSharedPrefs(), SDK_CONFIG_FETCHED_TIMESTAMP, fetchDate);
    }

    @Override
    public boolean userMessageNeedsRetry() {
        return getBooleanPreference(getSharedPrefs(), LAST_USER_MESSAGE_FAILED_KEY, false);
    }

    @Override
    public void setUserMessageNeedsRetry(boolean needsRetry) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set user message needs retry: " + needsRetry);
        setBooleanPreference(getSharedPrefs(), LAST_USER_MESSAGE_FAILED_KEY, needsRetry);
    }

    @Override
    public void setSessionNumber(int sessionNumber) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set session number: " + sessionNumber);
        setIntegerPreference(getSharedPrefs(), LAST_SESSION_NUMBER_KEY, sessionNumber);
    }

    @Override
    public Optional<Integer> getSessionNumber() {
        return getIntegerPreference(getSharedPrefs(), LAST_SESSION_NUMBER_KEY);
    }

    @Override
    public void setJavaScriptBundleURL(String url) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Java script bundle url: " + url);
        setStringPreference(getSharedPrefs(), JAVA_SCRIPT_BUNDLE_URL_KEY, url);
    }

    @Override
    public String getJavaScriptBundleURL() {
        return getStringPreference(getSharedPrefs(), JAVA_SCRIPT_BUNDLE_URL_KEY).orNull();
    }

    @Override
    public void setJavaScriptPatchNumber(String number) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Java script patch number: " + number);
        setStringPreference(getSharedPrefs(), JAVA_SCRIPT_PATCH_NUMBER_KEY, number);
    }

    @Override
    public Optional<String> getJavaScriptPatchNumber() {
        return getStringPreference(getSharedPrefs(), JAVA_SCRIPT_PATCH_NUMBER_KEY);
    }

    @Override
    public void setReactNativeVersionNumber(String version) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set RN version number: " + version);
        setStringPreference(getSharedPrefs(), REACT_NATIVE_VERSION_KEY, version);
    }

    @Override
    public Optional<String> getReactNativeVersionNumber() {
        return getStringPreference(getSharedPrefs(), REACT_NATIVE_VERSION_KEY);
    }

    @Override
    public void setUnityVersionNumber(String version) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Unity version number: " + version);
        setStringPreference(getSharedPrefs(), UNITY_VERSION_NUMBER_KEY, version);
    }

    @Override
    public Optional<String> getUnityVersionNumber() {
        return getStringPreference(getSharedPrefs(), UNITY_VERSION_NUMBER_KEY);
    }

    @Override
    public void setUnityBuildIdNumber(String version) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Unity build id number: " + version);
        setStringPreference(getSharedPrefs(), UNITY_BUILD_ID_NUMBER_KEY, version);
    }

    @Override
    public Optional<String> getUnityBuildIdNumber() {
        return getStringPreference(getSharedPrefs(), UNITY_BUILD_ID_NUMBER_KEY);
    }

    @Override
    public void setIsJailbroken(Boolean isJailbroken) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Jailbroken: " + isJailbroken);
        setBooleanPreference(getSharedPrefs(), IS_JAILBROKEN_KEY, isJailbroken);
    }

    @Override
    public Optional<Boolean> getJailbroken() {
        Boolean isJailbroken = getBooleanPreference(getSharedPrefs(), IS_JAILBROKEN_KEY, false);
        return Optional.of(isJailbroken);
    }

    @Override
    public void setScreenResolution(String screenResolution) {
        InternalStaticEmbraceLogger.logDeveloper("EmbracePreferencesService", "Set Screen resolution: " + screenResolution);
        setStringPreference(getSharedPrefs(), SCREEN_RESOLUTION_KEY, screenResolution);
    }

    @Override
    public Optional<String> getScreenResolution() {
        return getStringPreference(getSharedPrefs(), SCREEN_RESOLUTION_KEY);
    }
}
