package io.embrace.android.embracesdk;

import androidx.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicInteger;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.optional.Optional;

/**
 * Logs messages remotely, so that they can be viewed as events during a user's session.
 */
class EmbraceRemoteLogger implements MemoryCleanerListener {

    /**
     * The default limit of logs that can be send.
     */
    private static final int DEFAULT_LOG_INFO_LIMIT = 100;
    private static final int DEFAULT_LOG_WARNING_LIMIT = 100;
    private static final int DEFAULT_LOG_ERROR_LIMIT = 250;
    private static final int LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH = 128;
    private static final int LOG_MESSAGE_UNITY_MAXIMUM_ALLOWED_LENGTH = 16384;

    private final MetadataService metadataService;

    private final ScreenshotService screenshotService;

    private final ApiClient apiClient;

    private final UserService userService;

    private final ConfigService configService;

    private final GatingService gatingService;

    private final EmbraceSessionProperties sessionProperties;

    private final BackgroundWorker worker;

    private final Object lock = new Object();
    private final NavigableMap<Long, String> infoLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> warningLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> errorLogIds = new ConcurrentSkipListMap<>();

    private final AtomicInteger logsInfoCount = new AtomicInteger(0);
    private final AtomicInteger logsErrorCount = new AtomicInteger(0);
    private final AtomicInteger logsWarnCount = new AtomicInteger(0);
    private final AtomicInteger unhandledExceptionCount = new AtomicInteger(0);
    private final int maxLength;

    private final InternalEmbraceLogger logger;
    private final Clock clock;

    @VisibleForTesting
    EmbraceRemoteLogger(MetadataService metadataService,
                        ScreenshotService screenshotService,
                        ApiClient apiClient,
                        UserService userService,
                        ConfigService configService,
                        MemoryCleanerService memoryCleanerService,
                        EmbraceSessionProperties sessionProperties,
                        InternalEmbraceLogger logger,
                        Clock clock,
                        BackgroundWorker worker,
                        GatingService gatingService) {
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.screenshotService = Preconditions.checkNotNull(screenshotService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        this.gatingService = Preconditions.checkNotNull(gatingService);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);

        // Session properties
        this.sessionProperties = Preconditions.checkNotNull(sessionProperties);
        this.clock = Preconditions.checkNotNull(clock);
        this.logger = logger;
        this.worker = worker;

        if (configService.getConfig().getLogMessageMaximumAllowedLength().isPresent()) {
            maxLength = configService.getConfig().getLogMessageMaximumAllowedLength().get();
            logger.logDeveloper("EmbraceRemoteLogger", "Log message maximum allowed length: " + maxLength);
        } else {
            maxLength = LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH;
            logger.logDeveloper("EmbraceRemoteLogger", "Log message maximum allowed length - default: " + maxLength);
        }
    }

    EmbraceRemoteLogger(
            MetadataService metadataService,
            ScreenshotService screenshotService,
            ApiClient apiClient,
            UserService userService,
            ConfigService configService,
            MemoryCleanerService memoryCleanerService,
            EmbraceSessionProperties sessionProperties,
            InternalEmbraceLogger logger,
            Clock clock,
            GatingService sessionGatingService) {
        this(metadataService,
                screenshotService,
                apiClient,
                userService,
                configService,
                memoryCleanerService,
                sessionProperties,
                logger,
                clock,
                BackgroundWorker.ofSingleThread("Remote logging"),
                sessionGatingService);
    }

    /**
     * Gets the current thread's stack trace.
     *
     * @return the stack trace for the current thread or a throwable
     */
    static List<String> getWrappedStackTrace() {
        return getWrappedStackTrace(Thread.currentThread().getStackTrace());
    }

    /**
     * Gets the stack trace of the throwable.
     *
     * @return the stack trace of a throwable
     */
    static List<String> getWrappedStackTrace(StackTraceElement[] stackTraceElements) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Processing wrapped stack trace");

        List<String> augmentedStackReturnAddresses = new ArrayList<>();
        for (StackTraceElement element : stackTraceElements) {
            augmentedStackReturnAddresses.add(element.toString());
        }
        return augmentedStackReturnAddresses;
    }

    /**
     * Creates a remote log.
     *
     * @param message        the message to log
     * @param type           the type of message to log, which must be INFO_LOG, WARNING_LOG, or ERROR_LOG
     * @param takeScreenshot whether to take a screenshot when logging the event
     * @param properties     custom properties to send as part of the event
     */
    void log(String message,
             EmbraceEvent.Type type,
             boolean takeScreenshot,
             Map<String, Object> properties) {
        log(message, type, takeScreenshot, false, properties, null, null, Embrace.AppFramework.NATIVE);
    }

    /**
     * Creates a remote log.
     *
     * @param message            the message to log
     * @param type               the type of message to log, which must be INFO_LOG, WARNING_LOG, or ERROR_LOG
     * @param takeScreenshot     whether to take a screenshot when logging the event
     * @param isException        whether the log is an exception
     * @param properties         custom properties to send as part of the event
     * @param stackTraceElements the stacktrace elements of a throwable
     * @param customStackTrace   stacktrace string for non-JVM exceptions
     */
    void log(String message,
             EmbraceEvent.Type type,
             boolean takeScreenshot,
             boolean isException,
             Map<String, Object> properties,
             StackTraceElement[] stackTraceElements,
             String customStackTrace,
             Embrace.AppFramework framework
    ) {
        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Attempting to log");
        final long timestamp = clock.now();

        final Stacktraces stacktraces = new Stacktraces(
                stackTraceElements != null ? getWrappedStackTrace(stackTraceElements) : getWrappedStackTrace(),
                customStackTrace,
                framework
        );

        // As the event is sent asynchronously and user info may change, preserve the user info
        // at the time of the log call
        final UserInfo logUserInfo = UserInfo.newBuilder(userService.getUserInfo()).build();
        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Added user info to log");

        worker.submit(() -> {
            synchronized (lock) {
                if (configService.isLogMessageDisabled(message)) {
                    logger.logWarning("Log message disabled. Ignoring log with message " + message);
                    return null;
                }

                if (configService.isMessageTypeDisabled(MessageType.LOG)) {
                    logger.logWarning("Log message disabled. Ignoring all Logs.");
                    return null;
                }

                String id = Uuid.getEmbUuid();
                if (type.equals(EmbraceEvent.Type.INFO_LOG)) {
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "New INFO log");
                    logsInfoCount.incrementAndGet();
                    if (infoLogIds.size() < configService.getConfig().getInfoLogLimit().or(DEFAULT_LOG_INFO_LIMIT)) {
                        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Logging INFO log number " + logsInfoCount);
                        infoLogIds.put(timestamp, id);
                    } else {
                        logger.logWarning("Info Log limit has been reached.");
                        return null;
                    }
                } else if (type.equals(EmbraceEvent.Type.WARNING_LOG)) {
                    logsWarnCount.incrementAndGet();
                    if (warningLogIds.size() < configService.getConfig().getWarnLogLimit().or(DEFAULT_LOG_WARNING_LIMIT)) {
                        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Logging WARNING log number " + logsWarnCount);
                        warningLogIds.put(timestamp, id);
                    } else {
                        logger.logWarning("Warning Log limit has been reached.");
                        return null;
                    }
                } else if (type.equals(EmbraceEvent.Type.ERROR_LOG)) {
                    logsErrorCount.incrementAndGet();
                    if (errorLogIds.size() < configService.getConfig().getErrorLogLimit().or(DEFAULT_LOG_ERROR_LIMIT)) {
                        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Logging ERROR log number " + logsErrorCount);
                        errorLogIds.put(timestamp, id);
                    } else {
                        logger.logWarning("Error Log limit has been reached.");
                        return null;
                    }
                } else {
                    logger.logWarning("Unknown log level " + type.toString());
                    return null;
                }

                String processedMessage;
                if (framework == Embrace.AppFramework.UNITY) {
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Process Unity Log message");
                    processedMessage = processUnityLogMessage(message);
                    unhandledExceptionCount.incrementAndGet();
                } else {
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Process simple Log message");
                    processedMessage = processLogMessage(message);
                }

                // TODO validate event metadata here!

                // Build event
                Event.Builder builder = Event.newBuilder()
                        .withType(type)
                        .withName(processedMessage)
                        .withIsException(isException)
                        .withTimestamp(clock.now())
                        .withAppState(metadataService.getAppState())
                        .withMessageId(id)
                        .withCustomProperties(properties)
                        .withSessionProperties(sessionProperties.get());
                Optional<String> optionalSessionId = metadataService.getActiveSessionId();
                if (optionalSessionId.isPresent()) {
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Adding SessionId to event");
                    builder.withSessionId(optionalSessionId.get());
                }
                if (takeScreenshot && !configService.isScreenshotDisabledForEvent(message)) {
                    InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Adding screenshot to event");
                    boolean screenshotTaken = screenshotService.takeScreenshotLogEvent(id);
                    builder.withScreenshotTaken(screenshotTaken);
                }
                Event event = builder.build();

                // Build event message
                EventMessage.Builder eventMessageBuilder = EventMessage.newBuilder()
                        .withEvent(event)
                        .withDeviceInfo(metadataService.getDeviceInfo())
                        .withAppInfo(metadataService.getAppInfo())
                        .withUserInfo(logUserInfo)
                        .withStacktraces(stacktraces);

                if (checkIfShouldGateLog(type)) {
                    logger.logDebug(type + " was gated by config. The event was" + "not sent.");
                    return null;
                }

                // Sanitize log event
                EventMessage logEvent = gatingService.gateEventMessage(eventMessageBuilder.build());
                InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Attempt to Send log Event");
                apiClient.sendLogs(logEvent);
                InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "LogEvent api call running in background job");
            }
            return null;
        });
    }

    /**
     * Finds all IDs of log events at info level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findInfoLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.infoLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at warning level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findWarningLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.warningLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at error level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findErrorLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.errorLogIds.subMap(startTime, endTime).values());
    }

    /**
     * The total number of info logs that the app attempted to send.
     */
    int getInfoLogsAttemptedToSend() {
        return logsInfoCount.get();
    }

    /**
     * The total number of warning logs that the app attempted to send.
     */
    int getWarnLogsAttemptedToSend() {
        return logsWarnCount.get();
    }

    /**
     * The total number of error logs that the app attempted to send.
     */
    int getErrorLogsAttemptedToSend() {
        return logsErrorCount.get();
    }

    int getUnhandledExceptionsSent() {
        if (unhandledExceptionCount.get() > 0) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "UnhandledException number: " + unhandledExceptionCount);
        }

        return unhandledExceptionCount.get();
    }

    private String processLogMessage(String message) {
        return processLogMessage(message, maxLength);
    }

    private String processUnityLogMessage(String message) {
        return processLogMessage(message, LOG_MESSAGE_UNITY_MAXIMUM_ALLOWED_LENGTH);
    }

    private String processLogMessage(String message, int maxLength) {
        if (message.length() > maxLength) {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Message length exceeds the allowed max length");
            String endChars = "...";

            // ensure that we never end up with a negative offset when extracting substring, regardless of the config value set
            int allowedLength = maxLength >= endChars.length() ? maxLength - endChars.length() : LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH - endChars.length();
            logger.logWarning("Truncating message to " + message + " characters");
            return message.substring(0, allowedLength) + endChars;
        } else {
            InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Allowed message length");
            return message;
        }
    }

    /**
     * Checks if the info or warning log event should be gated based on gating config. Error logs
     * should never be gated.
     *
     * @param type of the log event
     * @return true if the log should be gated
     */
    boolean checkIfShouldGateLog(EmbraceEvent.Type type) {
        switch (type) {
            case INFO_LOG: {
                boolean shouldGate = gatingService.shouldGateInfoLog();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Should gate INFO log: " + shouldGate);
                return shouldGate;
            }
            case WARNING_LOG: {
                boolean shouldGate = gatingService.shouldGateWarnLog();
                InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Should gate WARN log: " + shouldGate);
                return shouldGate;
            }
            default: {
                InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Should gate log: false");
                return false;
            }
        }
    }

    @Override
    public void cleanCollections() {
        this.logsInfoCount.set(0);
        this.logsWarnCount.set(0);
        this.logsErrorCount.set(0);
        this.unhandledExceptionCount.set(0);
        this.infoLogIds.clear();
        this.warningLogIds.clear();
        this.errorLogIds.clear();
        InternalStaticEmbraceLogger.logDeveloper("EmbraceRemoteLogger", "Collections cleaned");
    }
}
