package io.embrace.android.embracesdk;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.view.View;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.optional.Optional;

import java.io.ByteArrayOutputStream;

import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Takes screenshots and submits them to the Embrace API.
 */
final class EmbraceScreenshotService implements ScreenshotService {
    private static final long SCREENSHOT_COOLDOWN = 3000;

    private static final int SCREENSHOT_JPEG_COMPRESSION_VALUE = 70;

    private final ActivityService activityService;

    private final ConfigService configService;

    private final ApiClient apiClient;

    private volatile long lastScreenshot;

    private final InternalEmbraceLogger logger;

    public EmbraceScreenshotService(ActivityService activityService,
                                    ConfigService configService,
                                    ApiClient apiClient,
                                    InternalEmbraceLogger logger) {
        this.activityService = Preconditions.checkNotNull(activityService);
        this.configService = Preconditions.checkNotNull(configService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.logger = logger;
    }

    @Override
    public boolean takeScreenshotLogEvent(String logId) {
        return takeScreenshot(logId, null);
    }

    @Override
    public boolean takeScreenshotMoment(String eventId) {
        return takeScreenshot(null, eventId);
    }

    private boolean takeScreenshot(String logId, String eventId) {
        synchronized (this) {
            long timestamp = System.currentTimeMillis();
            if (activityService.isInBackground() ||
                    !configService.getConfig().getScreenshotsEnabled() ||
                    (timestamp - lastScreenshot) < SCREENSHOT_COOLDOWN) {

                logger.logDebug("Screenshots are disabled, app is backgrounded, or cooling down");
                return false;
            }
            Optional<Activity> optionalActivity = activityService.getForegroundActivity();
            if (optionalActivity.isPresent()) {
                byte[] screenshot = screenshot(optionalActivity.get());
                lastScreenshot = timestamp;
                if (eventId != null) {
                    logger.logDeveloper("EmbraceScreenshotService", "Attempting to send moment screenshot");
                    apiClient.sendMomentScreenshot(screenshot, eventId);
                    logger.logDeveloper("EmbraceScreenshotService", "Send moment screenshot running on background");
                } else if (logId != null) {
                    logger.logDeveloper("EmbraceScreenshotService", "Attempting to send log screenshot");
                    apiClient.sendLogScreenshot(screenshot, logId);
                    logger.logDeveloper("EmbraceScreenshotService", "Send Log screenshot running on background");
                } else {
                    logger.logDeveloper("EmbraceScreenshotService", "Foreground activity not present");
                    return false;
                }
                return true;
            } else {
                logger.logDebug("Screenshot cannot be taken as there is no active activity");
                return false;
            }
        }
    }

    private byte[] screenshot(Activity activity) {
        Bitmap screenshot = takeScreenshot(activity);
        if (screenshot == null) {
            logger.logDebug("Screenshot returned was null");
            return new byte[0];
        }
        logger.logDebug("Compressing screenshot");
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        try {
            screenshot.compress(Bitmap.CompressFormat.JPEG, SCREENSHOT_JPEG_COMPRESSION_VALUE, stream);
            logger.logDeveloper("EmbraceScreenshotService", "Screenshot compressed");
        } catch (OutOfMemoryError e) {
            logger.logDebug("Failed to compress screenshot due insufficient memory.", e);
            return new byte[0];
        }
        screenshot.recycle();
        logger.logDeveloper("EmbraceScreenshotService", "Screenshot recycled");

        return stream.toByteArray();
    }

    private Bitmap takeScreenshot(Activity activity) {
        if (activity != null) {
            View view = activity.getWindow().getDecorView();
            if (view != null && view.getWidth() > 0 && view.getHeight() > 0) {
                try {
                    Bitmap screenshot = Bitmap.createBitmap(view.getWidth(), view.getHeight(), Bitmap.Config.ARGB_8888);
                    logger.logDeveloper("EmbraceScreenshotService", "Bitmap created");
                    view.draw(new Canvas(screenshot));
                    return screenshot;
                } catch (OutOfMemoryError e) {
                    logger.logDebug("Failed to take screenshot due insufficient memory.", e);
                    return null;
                }
            }
        }
        return null;
    }
}