package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.Session.SessionLifeEventType;
import static io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.logDeveloper;

import androidx.annotation.VisibleForTesting;

import java.util.Collection;
import java.util.Map;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;
import io.embrace.android.embracesdk.utils.optional.Optional;

/**
 * Handles the lifecycle of an Embrace session.
 * <p>
 * A session encapsulates all metrics gathered between the app being foregrounded and the app being
 * backgrounded. A caching service runs on an interval of every
 * {@value EmbraceSessionService#SESSION_CACHING_INTERVAL} seconds which saves the current session
 * to disk, so if the app is terminated, the session can resume from where it left off.
 */
final class EmbraceSessionService implements SessionService, ActivityListener {

    /**
     * Signals to the API that the application was in the foreground.
     */
    public static final String APPLICATION_STATE_ACTIVE = "active";
    /**
     * Signals to the API that the application was in the background.
     */
    public static final String APPLICATION_STATE_BACKGROUND = "background";

    /**
     * The minimum threshold for how long a session must last. Package-private for test accessibility
     */
    static Long minSessionTime = 5000L;
    /**
     * Session caching interval in seconds.
     */
    static final int SESSION_CACHING_INTERVAL = 2;

    /**
     * Synchronization lock.
     */
    private final Object lock = new Object();
    /**
     * SDK startup time. Only set for cold start sessions.
     */
    private long sdkStartupDuration = 0;

    /**
     * Embrace service dependencies of the session service.
     */
    private final ActivityService activityService;
    private final NdkService ndkService;
    private final SessionHandler sessionHandler;
    private final SessionCacheManager sessionCacheManager;

    /**
     * Asynchronous workers.
     */
    private final BackgroundWorker nativeCrashSearchBackgroundWorker;

    /**
     * The currently active session.
     */
    private volatile Session activeSession;

    /**
     * Session properties
     */
    private final EmbraceSessionProperties sessionProperties;
    private final InternalEmbraceLogger logger;

    EmbraceSessionService(
            PreferencesService preferencesService,
            ActivityService activityService,
            NdkService ndkService,
            EmbraceSessionProperties sessionProperties,
            InternalEmbraceLogger logger,
            SessionCacheManager sessionCacheManager,
            SessionHandler sessionHandler,
            Boolean isNdkEnabled) {

        Preconditions.checkNotNull(preferencesService);
        this.activityService = Preconditions.checkNotNull(activityService);
        this.activityService.addListener(this, true);
        this.nativeCrashSearchBackgroundWorker = BackgroundWorker.ofSingleThread("Native Crash Search");
        this.ndkService = Preconditions.checkNotNull(ndkService);
        this.sessionProperties = Preconditions.checkNotNull(sessionProperties);
        this.sessionCacheManager = Preconditions.checkNotNull(sessionCacheManager);
        this.logger = logger;
        this.sessionHandler = Preconditions.checkNotNull(sessionHandler);

        if (!this.activityService.isInBackground()) {
            // If the app goes to foreground before the SDK finishes its startup,
            // the session service will not be registered to the activity listener and will not
            // start the cold session.
            // If so, force a cold session start.
            logger.logDeveloper("EmbraceSessionService", "Forcing cold start");
            startStateSession(true);
        }

        if (isNdkEnabled) {
            logger.logDeveloper("EmbraceSessionService", "NDK enabled, checking for native crashes");
            ndkService.checkForNativeCrash();
        } else {
            logger.logDeveloper("EmbraceSessionService", "NDK disabled, not checking for native crashes");
        }
    }

    /**
     * record the time taken to initialize the SDK
     *
     * @param sdkStartupDuration the time taken to initialize the SDK in milliseconds
     */
    public void setSdkStartupDuration(long sdkStartupDuration) {
        logger.logDeveloper("EmbraceSessionService", "Setting startup duration: " + sdkStartupDuration);
        this.sdkStartupDuration = sdkStartupDuration;
    }

    @Override
    public void startSession(
            boolean coldStart,
            SessionLifeEventType startType
    ) {

        Runnable automaticSessionCloserCallback = () -> {
            try {
                synchronized (lock) {
                    logger.logInfo("Automatic session closing triggered.");
                    triggerStatelessSessionEnd(SessionLifeEventType.TIMED);
                }
            } catch (Exception ex) {
                logger.logError("Error while trying to close the session " +
                        "automatically", ex);
            }
        };

        SessionMessage sessionMessage = sessionHandler.onSessionStarted(
                coldStart,
                startType,
                sessionProperties,
                automaticSessionCloserCallback,
                this::onPeriodicCacheActiveSession);

        if (sessionMessage != null) {
            logger.logDeveloper("EmbraceSessionService", "Session Message is created");
            this.activeSession = sessionMessage.getSession();
            logger.logDeveloper("EmbraceSessionService", "Active session: " + activeSession.getSessionId());
        } else {
            logger.logDeveloper("EmbraceSessionService", "Session Message is NULL");
        }
    }

    @Override
    public void handleCrash(String crashId) {
        logger.logDeveloper("EmbraceSessionService", "Attempt to handle crash id: " + crashId);

        synchronized (lock) {
            sessionHandler.onCrash(
                    activeSession,
                    crashId,
                    sessionProperties,
                    sdkStartupDuration);
        }
    }

    @Override
    public void addSessionStartListeners(Collection<SessionStartListener> listeners) {
        logger.logDeveloper("EmbraceSessionService", "Adding Session start listeners");
        sessionHandler.addSessionStartListeners(listeners);
    }

    @Override
    public void addSessionEndListeners(Collection<SessionEndListener> listeners) {
        logger.logDeveloper("EmbraceSessionService", "Adding Session end listeners");
        sessionHandler.addSessionEndListeners(listeners);
    }

    /**
     * Caches the session, with performance information generated up to the current point.
     */
    @VisibleForTesting
    void onPeriodicCacheActiveSession() {
        try {
            synchronized (lock) {
                sessionHandler.onPeriodicCacheActiveSession(activeSession, sessionProperties, sdkStartupDuration);
            }
        } catch (Exception ex) {
            logger.logDebug("Error while caching active session", ex);
        }
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        logger.logDeveloper("EmbraceSessionService", "OnForeground. Starting session.");
        startStateSession(coldStart);
    }

    private void startStateSession(boolean coldStart) {
        logger.logDeveloper("EmbraceSessionService", "Start state session. Is cold start: " + coldStart);

        synchronized (lock) {
            if (coldStart) {
                sessionCacheManager.sendPreviousCachedSession();
            } else {
                logger.logDeveloper("EmbraceSessionService", "Cold start =false. No attempt to send previous cache session");
            }
            startSession(coldStart, SessionLifeEventType.STATE);
        }
    }

    @Override
    public void onBackground() {
        logger.logDeveloper("EmbraceSessionService", "OnBackground. Ending session.");
        endSession(SessionLifeEventType.STATE);
    }

    /**
     * It will try to end session. Note that it will either be for MANUAL or TIMED types.
     *
     * @param endType the origin of the event that ends the session.
     */
    @Override
    public void triggerStatelessSessionEnd(SessionLifeEventType endType) {
        if (SessionLifeEventType.STATE == endType) {
            logger.logWarning("triggerStatelessSessionEnd is not allowed to be called for SessionLifeEventType=" + endType);
            return;
        }

        // Ends active session.
        endSession(endType);

        // Starts a new session.
        if (!activityService.isInBackground()) {
            logger.logDeveloper("EmbraceSessionService", "Activity is not in background, starting session.");
            startSession(false, endType);
        } else {
            logger.logDeveloper("EmbraceSessionService", "Activity in background, not starting session.");
        }

        logger.logInfo("Session successfully closed.");
    }

    /**
     * This will trigger all necessary events to end the current session and send it to the server.
     *
     * @param endType the origin of the event that ends the session.
     */
    private synchronized void endSession(SessionLifeEventType endType) {
        logger.logDebug("Will try to end session.");
        sessionHandler.onSessionEnded(endType, activeSession, sessionProperties, sdkStartupDuration);

        // clear active session
        activeSession = null;
        logger.logDeveloper("EmbraceSessionService", "Active session cleared");
    }

    @Override
    public void close() {
        logger.logInfo("Shutting down EmbraceSessionService");
        sessionHandler.close();
    }

    Session getActiveSession() {
        return activeSession;
    }

    @Override
    public boolean addProperty(String key, String value, boolean permanent) {
        logger.logDeveloper("EmbraceSessionService", "Add Property: " + key + " - " + value);

        boolean added = sessionProperties.add(key, value, permanent);
        if (added) {
            logger.logDeveloper("EmbraceSessionService", "Session properties updated");
            ndkService.onSessionPropertiesUpdate(sessionProperties.get());
        } else {
            logger.logDeveloper("EmbraceSessionService", "Cannot add property: " + key);
        }
        return added;
    }

    @Override
    public boolean removeProperty(String key) {
        logger.logDeveloper("EmbraceSessionService", "Remove Property: " + key);

        boolean removed = sessionProperties.remove(key);
        if (removed) {
            logger.logDeveloper("EmbraceSessionService", "Session properties updated");
            ndkService.onSessionPropertiesUpdate(sessionProperties.get());
        } else {
            logger.logDeveloper("EmbraceSessionService", "Cannot remove property: " + key);
        }
        return removed;
    }

    @Override
    public Map<String, String> getProperties() {
        return this.sessionProperties.get();
    }

    @Override
    public void handleNativeCrash(Optional<NativeCrashData> nativeCrashData) {
        try {
            nativeCrashSearchBackgroundWorker.submit(() -> {
                logger.logDeveloper("EmbraceSessionService", "Processing handleNativeCrash() runnable");
                try {
                    if (nativeCrashData.isPresent()) {
                        // retrieve last session message and update crash report id if the session id matches
                        // with the session id attached to the crash.
                        Optional<SessionMessage> previousSessionMessage = sessionCacheManager.fetchPreviousSessionMessage();
                        if (previousSessionMessage.isPresent()) {
                            logger.logDeveloper("EmbraceSessionService", "Loading previous session message");

                            SessionMessage sessionMessage = previousSessionMessage.get();
                            NativeCrashData crash = nativeCrashData.get();
                            Session session = sessionMessage.getSession();
                            if (session.getSessionId().equals(crash.getSessionId())) {
                                logger.logDeveloper("EmbraceSessionService", "Building session message to add native crash ID");

                                SessionMessage.Builder messageBuilder = SessionMessage.newBuilder(sessionMessage);
                                Session.Builder sessionBuilder = Session.newBuilder(session);
                                sessionBuilder.withCrashReportId(crash.getNativeCrashId());
                                messageBuilder.withSession(sessionBuilder.build());

                                logger.logDeveloper("EmbraceSessionService", "Overwriting previously cached session message to add native crash ID");

                                // Overrides the previous session message with the one that has the native crash.
                                sessionCacheManager.updatePreviousSessionMessageCache(messageBuilder.build());
                                sessionCacheManager.unlockPreviousSessionCache();
                                sessionCacheManager.sendPreviousCachedSession();
                            } else {
                                logger.logDebug(
                                        "Crash report did not match with last session message. Not updating cached session {lastSessionId=" + session.getSessionId() +
                                                ", nativeCrashId=" + nativeCrashData.get().getNativeCrashId() +
                                                ", activeSessionId=" + session.getSessionId() + "}.");
                            }
                        } else {
                            logger.logInfo(
                                    "Could not find session to try to match native crash {nativeCrashId=" + nativeCrashData.get().getNativeCrashId() +
                                            ", activeSessionId=" + activeSession.getSessionId() + "}.");
                        }
                    }

                    logger.logDeveloper("EmbraceSessionService", "Finished looking for native crash");

                } catch (Exception ex) {
                    logger.logError("Failed to update cached session message with native crash report id.", ex);
                } finally {
                    // no matter what, unlock cache
                    sessionCacheManager.unlockPreviousSessionCache();
                }
                return null;
            });
        } catch (Exception ex) {
            logger.logError("Failed to create background worker to update pending session with the native crash report id.", ex);
        }
    }
}
