package io.embrace.android.embracesdk;

import android.os.Looper;
import android.os.Process;
import android.os.SystemClock;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.OnLifecycleEvent;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import kotlin.Lazy;

/**
 * Samples stacktraces from the main thread at startup.
 */
class EmbraceStartupTracingService implements StartupTracingService {

    private final List<StartupStacktrace> stacktraces = new ArrayList<>();
    private final Clock clock;
    private final Thread mainThread;
    private final ScheduledExecutorService scheduler;
    private final Random random;
    private final InternalEmbraceLogger logger;

    @VisibleForTesting
    ScheduledFuture<?> sampleHandler = null;

    @VisibleForTesting
    int samplesRemaining;

    @VisibleForTesting
    StartupStacktrace lastStacktrace = null;
    private int maxStackTraceLength;

    EmbraceStartupTracingService(Clock clock, Lazy<CacheService> cacheService, InternalEmbraceLogger logger) {
        this(clock, new Random(), Executors.newScheduledThreadPool(1), Looper.getMainLooper().getThread(), cacheService, logger);
    }

    @VisibleForTesting
    EmbraceStartupTracingService(Clock clock,
                                 Random random,
                                 ScheduledExecutorService executorService,
                                 Thread mainThread,
                                 Lazy<CacheService> cacheService,
                                 InternalEmbraceLogger logger) {
        this.clock = clock;
        this.scheduler = executorService;
        this.mainThread = mainThread;
        this.random = random;
        this.logger = logger;
        scheduler.submit(() -> setup(cacheService));
    }

    void setup(Lazy<CacheService> cacheService) {
        logger.logDeveloper("EmbraceStartupTracingService", "setup");

        try {
            CacheService service = cacheService.getValue();
            Config config = getConfig(service);
            if (!config.isStartupSamplingEnabled()) {
                scheduler.shutdown();
                logger.logInfo("Startup sampling is disabled for this session.");
                return;
            } else {
                logger.logInfo("Startup sampling is enabled for this session.");
            }
            Config.StartupSamplingConfig startupSamplingConfig = config.getStartupSamplingConfig();
            int intervalMs = startupSamplingConfig.getSampleInterval();
            int granularityMs = startupSamplingConfig.getSampleGranularity();
            int durationMs = startupSamplingConfig.getSamplingDuration();
            maxStackTraceLength = startupSamplingConfig.getMaxStacktraceLength();

            int offsetMs = calculateSampleDelay(intervalMs, granularityMs);
            this.samplesRemaining = durationMs / intervalMs;

            // schedule sampling
            try {
                sampleHandler = scheduler.scheduleAtFixedRate(this::takeSample, offsetMs, intervalMs,
                        TimeUnit.MILLISECONDS);
                logger.logDeveloper("EmbraceStartupTracingService", "Sample handler scheduled. Offset: " + offsetMs + "ms - Interval: " + intervalMs + "ms");

            } catch (RejectedExecutionException e) {
                // Should only happen if something wonky happened with the scheduler and it shut down.
                logger.logWarning("Could not schedule startup sampling.", e);
            }

            // schedule stop of sampling
            try {
                int delay = calculateDescheduleDelay(intervalMs, durationMs, offsetMs);
                scheduler.schedule(this::deschedule, delay, TimeUnit.MILLISECONDS);
                logger.logDeveloper("EmbraceStartupTracingService", "Scheduler delay: " + delay);
            } catch (RejectedExecutionException e) {
                // Should only happen if something wonky happened with the scheduler and it shut down.
                logger.logWarning("Could not deschedule startup sampling.", e);
            }
        } catch (Throwable e) {
            logger.logError("Startup sampling initialization failed.", e);
        }
    }

    /**
     * We want to try to stagger the initial samples evenly so we roughly get buckets that
     * are the size of STARTUP_SAMPLE_GRANULARITY_MS. We then pick a random bucket to start
     * sampling in. Since we are expecting data from a large number of devices, we can then
     * sample at a relatively low rate on each device, while still getting a reasonable
     * resolution in what is happening at a given point in the startup process as long as
     * we have a sufficiently large data set.
     */
    @VisibleForTesting
    int calculateSampleDelay(int intervalMs, int granularityMs) {
        int maxOffsetInterval = intervalMs / granularityMs;
        int k = random.nextInt(maxOffsetInterval);
        return k * granularityMs;
    }

    @VisibleForTesting
    int calculateDescheduleDelay(int intervalMs, int durationMs, int offsetMs) {
        return offsetMs + durationMs + (intervalMs / 2);
    }

    @VisibleForTesting
    Config getConfig(CacheService service) {
        Config.StartupSamplingConfig cachedStartupSamplingConfig = service.loadStartupSamplingConfig(() -> 0);

        // If we have no cached config then assume that sampling should be enabled since the use of
        // this functionality is opt-in.
        Config config;

        if (cachedStartupSamplingConfig != null) {
            config = Config.ofDefault(null, cachedStartupSamplingConfig, random);
            logger.logDeveloper("EmbraceStartupTracingService", "Config initialized with cachedStartupSamplingConfig");
        } else {
            config = Config.ofDefault(null, null, random);
            logger.logDeveloper("EmbraceStartupTracingService", "Config initialized with null cachedStartupSamplingConfig");
        }
        return config;
    }

    @Override
    @NonNull
    public List<StartupStacktrace> getStacktraces() {
        return new ArrayList<>(stacktraces);
    }

    @Override
    public void close() {
        try {
            if (sampleHandler != null) {
                sampleHandler.cancel(false);
                logger.logDeveloper("EmbraceStartupTracingService", "SampleHandler cancelled");
            }
            if (scheduler != null) {
                scheduler.shutdownNow();
                logger.logDeveloper("EmbraceStartupTracingService", "Scheduler shut down");
            }
        } catch (Exception ex) {
            logger.logDebug("Failed to cleanly shut down EmbraceStartupTracingService");
        }
    }

    // stop sampling when the app goes into the background
    @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
    @VisibleForTesting
    void stopSampling() {
        logger.logDebug("Stopping startup sampling");
        close();
    }

    @VisibleForTesting
    void deschedule() {
        logger.logDebug("Descheduling periodic startup sampling");
        close();
    }

    @VisibleForTesting
    void takeSample() {
        /*
         * Prevent against extra samples being taken if the descheduling task ends up being slower
         * than expected.
         */
        if (samplesRemaining <= 0 || this.mainThread == null) {
            logger.logDeveloper("EmbraceStartupTracingService", "No samples remaining, or mainThread null");
            return;
        }
        samplesRemaining -= 1;

        long timestamp = clock.now();
        long offset = 0;
        logger.logDebug("Taking startup sample at " + timestamp);
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            // Compute how long it is has been since the process launched.
            offset = SystemClock.elapsedRealtime() - Process.getStartElapsedRealtime();
            logger.logDeveloper("EmbraceStartupTracingService", "SO version > android N. Offset is: " + offset);
        }

        try {
            StackTraceElement[] stackTraceElements = mainThread.getStackTrace();
            List<String> lines = sanitizeStacktrace(stackTraceElements);
            StartupStacktrace stacktrace = new StartupStacktrace(timestamp, offset, lines);
            if (lastStacktrace != null) {
                /* Clear the lines for the current sample if they are the same as for the
                 * previous sample to reduce the payload size. Keep the comparison to the
                 * last stacktrace that was not cleared.
                 */
                if (!stacktrace.clearLinesIfTheyMatch(lastStacktrace)) {
                    logger.logDeveloper("EmbraceStartupTracingService", "New stacktrace is: " + stacktrace.getLines().toString());
                    lastStacktrace = stacktrace;
                } else {
                    logger.logDeveloper("EmbraceStartupTracingService", "lastStacktrace !clearLinesIfTheyMatch");
                }
            } else {
                logger.logDeveloper("EmbraceStartupTracingService", "Last stacktrace is null");
                lastStacktrace = stacktrace;
                logger.logDeveloper("EmbraceStartupTracingService", "New stacktrace is: " + stacktrace.getLines().toString());
            }
            stacktraces.add(stacktrace);
            logger.logDeveloper("EmbraceStartupTracingService", "stacktrace added");
        } catch (OutOfMemoryError ex) {
            // do nothing since we do not want to incur more memory usage
        } catch (Exception ex) {
            logger.logDebug("Startup sampling failed in sampling thread", ex);
        }
    }

    @NonNull
    @VisibleForTesting
    List<String> sanitizeStacktrace(StackTraceElement[] stackTraceElements) {
        logger.logDeveloper("EmbraceStartupTracingService", "sanitizeStacktrace");
        List<String> lines = new ArrayList<>();

        for (StackTraceElement element : stackTraceElements) {
            // Limit the size of the stacktrace to control the payload size
            if (lines.size() < maxStackTraceLength) {
                lines.add(element.toString());
            }
        }
        return lines;
    }
}
